"""
Beal Conjecture - p-adic Valuation Verification

Extended testing to verify Prime Collision Necessity hypothesis
Tests valuations for coprime bases to find mismatches
"""

from beal_kernel import BealKernel
import json
from collections import defaultdict


def p_adic_valuation(n, p):
    """Compute p-adic valuation v_p(n)"""
    if n == 0:
        return float('inf')
    val = 0
    while n % p == 0:
        val += 1
        n //= p
    return val


def test_valuation_mismatch():
    """
    Test for valuation mismatches in coprime sums
    
    For coprime a,b, compute v_p(a^x + b^y) for various p
    Check if high valuations occur (would need to match c^z)
    """
    print("="*70)
    print("P-ADIC VALUATION VERIFICATION - PRIME COLLISION TEST")
    print("="*70)
    print()
    
    kernel = BealKernel()
    
    # Test coprime pairs
    coprime_pairs = [
        (3, 4), (3, 5), (3, 7), (5, 7), (5, 9),
        (7, 9), (7, 11), (9, 11), (11, 13)
    ]
    
    primes = [2, 3, 5, 7, 11, 13, 17, 19]
    
    high_valuations = []
    mismatch_examples = []
    
    print("[Phase 1] Testing Coprime Sums for High Valuations")
    print("-"*70)
    print()
    
    for a, b in coprime_pairs:
        if kernel.gcd(a, b) != 1:
            continue
        
        for x in range(3, 6):
            for y in range(3, 6):
                sum_val = a**x + b**y
                
                # Check valuations for all primes
                valuations = {}
                for p in primes:
                    v = p_adic_valuation(sum_val, p)
                    if v > 0:
                        valuations[p] = v
                
                # Check if any valuation is high (≥3)
                high_vals = {p: v for p, v in valuations.items() if v >= 3}
                
                if high_vals:
                    high_valuations.append({
                        'a': a, 'b': b, 'x': x, 'y': y,
                        'sum': sum_val,
                        'high_vals': high_vals
                    })
                    print(f"{a}^{x} + {b}^{y} = {sum_val}")
                    for p, v in high_vals.items():
                        print(f"  v_{p} = {v} (HIGH)")
                
                # For demonstration, assume this sum = c^z for some c,z
                # Check if valuations would match
                for z in range(3, 6):
                    c_approx = round(sum_val ** (1/z))
                    
                    if c_approx ** z == sum_val:
                        # Found a power! Check valuations
                        print(f"\n  POWER FOUND: {a}^{x} + {b}^{y} = {c_approx}^{z}")
                        
                        # Check gcd
                        gcd_val = kernel.gcd_multiple(a, b, c_approx)
                        print(f"  gcd({a},{b},{c_approx}) = {gcd_val}")
                        
                        if gcd_val == 1:
                            print(f"  ⚠️  COPRIME SOLUTION FOUND!")
                            
                            # Check valuations
                            for p in primes:
                                v_left = p_adic_valuation(sum_val, p)
                                v_right = z * p_adic_valuation(c_approx, p)
                                
                                if v_left != v_right:
                                    mismatch_examples.append({
                                        'a': a, 'b': b, 'c': c_approx,
                                        'x': x, 'y': y, 'z': z,
                                        'prime': p,
                                        'v_left': v_left,
                                        'v_right': v_right
                                    })
                                    print(f"  Prime {p}: v_p(left)={v_left} ≠ v_p(right)={v_right} ❌")
    
    print()
    print("[Phase 2] Summary")
    print("-"*70)
    print()
    
    print(f"High valuations found: {len(high_valuations)}")
    print(f"Valuation mismatches: {len(mismatch_examples)}")
    print()
    
    if high_valuations:
        print("Examples with v_p ≥ 3:")
        for ex in high_valuations[:5]:
            print(f"  {ex['a']}^{ex['x']} + {ex['b']}^{ex['y']} = {ex['sum']}")
            print(f"    High valuations: {ex['high_vals']}")
    
    print()
    
    if mismatch_examples:
        print("⚠️  VALUATION MISMATCHES IN COPRIME SOLUTIONS:")
        for ex in mismatch_examples:
            print(f"  {ex['a']}^{ex['x']} + {ex['b']}^{ex['y']} = {ex['c']}^{ex['z']}")
            print(f"    Prime {ex['prime']}: {ex['v_left']} ≠ {ex['v_right']}")
    else:
        print("✅ No coprime solutions found (as expected)")
    
    print()
    print("="*70)
    print("VERIFICATION COMPLETE")
    print("="*70)
    
    return {
        'high_valuations': len(high_valuations),
        'mismatches': len(mismatch_examples)
    }


if __name__ == "__main__":
    results = test_valuation_mismatch()
    
    with open('valuation_verification.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("\nResults saved to valuation_verification.json")
