"""
Beal Conjecture - Valuation Debt Kernel

Implements the Δ_p framework to verify Prime Collision Necessity
For each prime p, computes: Δ_p = v_p(c^z) - v_p(a^x + b^y)
Theorem requires Δ_p = 0 for ALL primes
"""

from beal_kernel import BealKernel
import json
from collections import defaultdict


def p_adic_valuation(n, p):
    """Compute p-adic valuation v_p(n)"""
    if n == 0:
        return float('inf')
    val = 0
    while n % p == 0:
        val += 1
        n //= p
    return val


def prime_factors_set(n):
    """Get set of prime factors"""
    factors = set()
    d = 2
    while d * d <= n:
        if n % d == 0:
            factors.add(d)
            while n % d == 0:
                n //= d
        d += 1
    if n > 1:
        factors.add(n)
    return factors


def compute_valuation_debt(a, b, c, x, y, z):
    """
    Compute valuation debt Δ_p for all relevant primes
    
    Returns dict: {prime: Δ_p}
    """
    # Get all primes involved
    primes = prime_factors_set(a) | prime_factors_set(b) | prime_factors_set(c)
    
    # Add small primes for completeness
    for p in [2, 3, 5, 7, 11, 13]:
        primes.add(p)
    
    sum_val = a**x + b**y
    
    debts = {}
    
    for p in sorted(primes):
        v_left = p_adic_valuation(sum_val, p)
        v_right = z * p_adic_valuation(c, p)
        
        delta = v_right - v_left
        
        if delta != 0:
            debts[p] = {
                'v_left': v_left,
                'v_right': v_right,
                'delta': delta,
                'p_divides_a': p_adic_valuation(a, p) > 0,
                'p_divides_b': p_adic_valuation(b, p) > 0,
                'p_divides_c': p_adic_valuation(c, p) > 0
            }
    
    return debts


def test_valuation_debt_framework():
    """
    Test Valuation Debt framework on known solutions
    
    For valid solutions (gcd > 1): Δ_p = 0 for all p
    For coprime attempts: Δ_p ≠ 0 for some p
    """
    print("="*70)
    print("VALUATION DEBT KERNEL - PRIME COLLISION VERIFICATION")
    print("="*70)
    print()
    
    kernel = BealKernel()
    
    # Test known solutions (should have Δ_p = 0 for all p)
    print("[Phase 1] Testing Known Solutions (gcd > 1)")
    print("-"*70)
    print()
    
    known_solutions = [
        (3, 6, 3, 3, 3, 5),   # gcd=3
        (2, 2, 2, 3, 3, 4),   # gcd=2
        (7, 7, 14, 3, 4, 3),  # gcd=7
    ]
    
    for a, b, c, x, y, z in known_solutions:
        if not kernel.verify_equation(a, b, c, x, y, z):
            print(f"⚠️  {a}^{x} + {b}^{y} ≠ {c}^{z} (equation doesn't hold)")
            continue
        
        gcd_val = kernel.gcd_multiple(a, b, c)
        debts = compute_valuation_debt(a, b, c, x, y, z)
        
        print(f"{a}^{x} + {b}^{y} = {c}^{z}, gcd={gcd_val}")
        
        if debts:
            print(f"  ⚠️  Non-zero debts found:")
            for p, d in debts.items():
                print(f"    p={p}: Δ={d['delta']} (v_left={d['v_left']}, v_right={d['v_right']})")
        else:
            print(f"  ✅ All Δ_p = 0 (valid solution)")
        print()
    
    # Test coprime attempts (should have Δ_p ≠ 0 for some p)
    print("[Phase 2] Testing Coprime Attempts")
    print("-"*70)
    print()
    
    coprime_attempts = []
    
    # Search for coprime pairs
    for a in range(2, 20):
        for b in range(a+1, 20):
            if kernel.gcd(a, b) != 1:
                continue
            
            for x in range(3, 5):
                for y in range(3, 5):
                    sum_val = a**x + b**y
                    
                    # Check if sum is a perfect power
                    for z in range(3, 6):
                        c_approx = round(sum_val ** (1/z))
                        
                        if c_approx ** z == sum_val:
                            gcd_val = kernel.gcd_multiple(a, b, c_approx)
                            
                            if gcd_val == 1:
                                coprime_attempts.append((a, b, c_approx, x, y, z))
    
    if coprime_attempts:
        print(f"Found {len(coprime_attempts)} coprime solutions:")
        for a, b, c, x, y, z in coprime_attempts:
            debts = compute_valuation_debt(a, b, c, x, y, z)
            
            print(f"\n{a}^{x} + {b}^{y} = {c}^{z}, gcd=1")
            print(f"  Valuation debts:")
            
            if debts:
                for p, d in debts.items():
                    print(f"    p={p}: Δ={d['delta']} (MISMATCH)")
                print(f"  ❌ Prime Collision detected!")
            else:
                print(f"  ⚠️  NO DEBTS - This would be a counterexample!")
    else:
        print("✅ No coprime solutions found (as predicted by theorem)")
    
    print()
    print("="*70)
    print("VALUATION DEBT VERIFICATION COMPLETE")
    print("="*70)
    
    return {
        'coprime_attempts': len(coprime_attempts)
    }


if __name__ == "__main__":
    results = test_valuation_debt_framework()
    
    with open('valuation_debt_results.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("\nResults saved to valuation_debt_results.json")
