"""
Beal Conjecture - Prime Collision Necessity Hypothesis

HYPOTHESIS: If a^x + b^y = c^z with x,y,z > 2 and gcd(a,b,c) = 1,
then there exists a prime p such that v_p(a^x + b^y) ≠ v_p(c^z),
creating a valuation contradiction.

This test attempts to FALSIFY the hypothesis by searching for
coprime solutions that satisfy all p-adic valuation constraints.
"""

from beal_kernel import BealKernel
import json


def p_adic_valuation(n, p):
    """Compute p-adic valuation v_p(n)"""
    if n == 0:
        return float('inf')
    val = 0
    while n % p == 0:
        val += 1
        n //= p
    return val


def prime_factorization_full(n):
    """Get complete prime factorization as dict {prime: power}"""
    factors = {}
    d = 2
    while d * d <= n:
        while n % d == 0:
            factors[d] = factors.get(d, 0) + 1
            n //= d
        d += 1
    if n > 1:
        factors[n] = factors.get(n, 0) + 1
    return factors


def test_prime_collision_hypothesis():
    """
    Test Prime Collision Necessity Hypothesis
    
    Search for coprime solutions and check if valuation contradiction exists.
    """
    print("="*70)
    print("PRIME COLLISION NECESSITY HYPOTHESIS - FALSIFICATION TEST")
    print("="*70)
    print()
    print("HYPOTHESIS:")
    print("  If a^x + b^y = c^z with x,y,z > 2 and gcd(a,b,c) = 1,")
    print("  then ∃ prime p: v_p(a^x + b^y) ≠ v_p(c^z)")
    print()
    print("TEST: Search for coprime solutions that DON'T have valuation contradiction")
    print("      (i.e., try to FALSIFY the hypothesis)")
    print()
    
    kernel = BealKernel()
    
    # Phase 1: Search for coprime solutions
    print("[Phase 1] Searching for Coprime Solutions")
    print("-"*70)
    
    coprime_solutions = []
    
    # Systematic search with coprime bases
    print("Searching coprime triples (a,b,c) with gcd=1...")
    print()
    
    tested = 0
    for x in range(3, 7):
        for y in range(3, 7):
            for z in range(3, 7):
                # Test coprime bases
                for a in range(2, 50):
                    for b in range(2, 50):
                        if kernel.gcd(a, b) != 1:
                            continue  # Skip if a,b not coprime
                        
                        lhs = a**x + b**y
                        c_approx = round(lhs ** (1/z))
                        
                        for c in [c_approx - 1, c_approx, c_approx + 1]:
                            if c <= 0:
                                continue
                            
                            tested += 1
                            
                            if kernel.verify_equation(a, b, c, x, y, z):
                                gcd_val = kernel.gcd_multiple(a, b, c)
                                
                                if gcd_val == 1:
                                    coprime_solutions.append({
                                        'a': a, 'b': b, 'c': c,
                                        'x': x, 'y': y, 'z': z
                                    })
    
    print(f"Tested {tested:,} combinations")
    print(f"Coprime solutions found: {len(coprime_solutions)}")
    print()
    
    if len(coprime_solutions) == 0:
        print("✅ NO coprime solutions found")
        print("   Hypothesis SURVIVES (cannot be falsified in this range)")
        print()
    else:
        print("⚠️  COPRIME SOLUTIONS FOUND - Analyzing...")
        print()
    
    # Phase 2: Analyze coprime solutions for valuation contradictions
    if coprime_solutions:
        print("[Phase 2] Valuation Contradiction Analysis")
        print("-"*70)
        print()
        
        for i, sol in enumerate(coprime_solutions):
            a, b, c = sol['a'], sol['b'], sol['c']
            x, y, z = sol['x'], sol['y'], sol['z']
            
            print(f"Solution {i+1}: {a}^{x} + {b}^{y} = {c}^{z}")
            print(f"  gcd({a},{b},{c}) = 1 ✓")
            
            # Get all primes involved
            factors_a = prime_factorization_full(a)
            factors_b = prime_factorization_full(b)
            factors_c = prime_factorization_full(c)
            
            all_primes = set(factors_a.keys()) | set(factors_b.keys()) | set(factors_c.keys())
            
            print(f"  Primes involved: {sorted(all_primes)}")
            
            # Check valuation for each prime
            contradiction_found = False
            
            for p in sorted(all_primes):
                v_a = p_adic_valuation(a, p)
                v_b = p_adic_valuation(b, p)
                v_c = p_adic_valuation(c, p)
                
                # Compute valuations of terms
                v_ax = x * v_a
                v_by = y * v_b
                v_cz = z * v_c
                
                # For a^x + b^y, valuation is min if different
                if v_ax != v_by:
                    v_lhs = min(v_ax, v_by)
                else:
                    # If equal, need to check actual sum
                    # This is the tricky case
                    v_lhs = v_ax  # Approximation
                
                v_rhs = v_cz
                
                if v_lhs != v_rhs:
                    print(f"  Prime {p}: v_p(LHS)={v_lhs}, v_p(RHS)={v_rhs} ❌ CONTRADICTION")
                    contradiction_found = True
                else:
                    print(f"  Prime {p}: v_p(LHS)={v_lhs}, v_p(RHS)={v_rhs} ✓")
            
            if contradiction_found:
                print(f"  → Valuation contradiction EXISTS (hypothesis holds)")
            else:
                print(f"  → NO valuation contradiction (FALSIFIES hypothesis!)")
            
            print()
    
    # Phase 3: Hypothesis verdict
    print("="*70)
    print("HYPOTHESIS VERDICT")
    print("="*70)
    print()
    
    if len(coprime_solutions) == 0:
        print("✅ HYPOTHESIS SURVIVES")
        print()
        print("Result:")
        print("  No coprime solutions found in tested range")
        print("  Hypothesis cannot be falsified")
        print()
        print("Interpretation:")
        print("  Strong evidence that gcd=1 is impossible")
        print("  Consistent with Lemma B1 (parity obstruction)")
        print()
        print("Status: STRONG EVIDENCE (not yet proof)")
    else:
        print("⚠️  HYPOTHESIS REQUIRES REFINEMENT")
        print()
        print(f"Found {len(coprime_solutions)} coprime solutions")
        print("Need to analyze valuation contradictions")
        print()
        print("Status: HYPOTHESIS INCOMPLETE")
    
    print()
    print("="*70)
    
    return {
        'tested': tested,
        'coprime_found': len(coprime_solutions),
        'hypothesis_survives': len(coprime_solutions) == 0
    }


if __name__ == "__main__":
    results = test_prime_collision_hypothesis()
    
    with open('prime_collision_test.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("Results saved to prime_collision_test.json")
