"""
Beal Conjecture Verification Kernel

Stage 1: Mechanization

This kernel provides deterministic verification for Beal Conjecture:
- Equation checking: a^x + b^y = c^z
- GCD computation: gcd(a,b,c)
- Prime factorization
- Solution validation
"""

from typing import Tuple, List, Dict, Any, Optional
import math
from functools import reduce


class BealKernel:
    """
    Deterministic verification kernel for Beal Conjecture
    
    Philosophy: Kernel is authority, not speculation.
    """
    
    def __init__(self):
        self.solutions_found = []
        self.near_misses = []
    
    def gcd(self, a: int, b: int) -> int:
        """Compute GCD using Euclidean algorithm"""
        while b:
            a, b = b, a % b
        return a
    
    def gcd_multiple(self, *numbers: int) -> int:
        """Compute GCD of multiple numbers"""
        return reduce(self.gcd, numbers)
    
    def prime_factors(self, n: int) -> List[int]:
        """Get prime factors of n"""
        factors = []
        d = 2
        while d * d <= n:
            while n % d == 0:
                if not factors or factors[-1] != d:
                    factors.append(d)
                n //= d
            d += 1
        if n > 1:
            factors.append(n)
        return factors
    
    def verify_equation(self, a: int, b: int, c: int, x: int, y: int, z: int) -> bool:
        """
        Verify if a^x + b^y = c^z
        
        Returns True if equation holds, False otherwise.
        """
        try:
            lhs = a**x + b**y
            rhs = c**z
            return lhs == rhs
        except OverflowError:
            # For very large numbers, use modular arithmetic check first
            return False
    
    def check_exponent_constraint(self, x: int, y: int, z: int) -> bool:
        """Check if exponents satisfy x, y, z > 2"""
        return x > 2 and y > 2 and z > 2
    
    def is_counterexample(self, a: int, b: int, c: int, x: int, y: int, z: int) -> bool:
        """
        Check if (a,b,c,x,y,z) is a counterexample to Beal Conjecture
        
        A counterexample must satisfy:
        1. a^x + b^y = c^z
        2. x, y, z > 2
        3. gcd(a,b,c) = 1
        """
        # Check exponent constraint
        if not self.check_exponent_constraint(x, y, z):
            return False
        
        # Check equation
        if not self.verify_equation(a, b, c, x, y, z):
            return False
        
        # Check coprimality
        common_gcd = self.gcd_multiple(a, b, c)
        return common_gcd == 1
    
    def is_solution(self, a: int, b: int, c: int, x: int, y: int, z: int) -> Dict[str, Any]:
        """
        Check if (a,b,c,x,y,z) is a solution (not necessarily counterexample)
        
        Returns detailed analysis.
        """
        result = {
            "equation_holds": False,
            "exponents_valid": False,
            "gcd": None,
            "is_counterexample": False,
            "prime_factors_a": [],
            "prime_factors_b": [],
            "prime_factors_c": [],
            "common_factors": []
        }
        
        # Check exponents
        result["exponents_valid"] = self.check_exponent_constraint(x, y, z)
        
        # Check equation
        result["equation_holds"] = self.verify_equation(a, b, c, x, y, z)
        
        if result["equation_holds"]:
            # Compute GCD
            result["gcd"] = self.gcd_multiple(a, b, c)
            
            # Get prime factors
            result["prime_factors_a"] = self.prime_factors(a)
            result["prime_factors_b"] = self.prime_factors(b)
            result["prime_factors_c"] = self.prime_factors(c)
            
            # Find common factors
            common = set(result["prime_factors_a"]) & set(result["prime_factors_b"]) & set(result["prime_factors_c"])
            result["common_factors"] = sorted(list(common))
            
            # Check if counterexample
            result["is_counterexample"] = (result["exponents_valid"] and result["gcd"] == 1)
        
        return result
    
    def search_solutions(self, 
                        a_max: int = 100, 
                        b_max: int = 100, 
                        c_max: int = 100,
                        x_max: int = 10,
                        y_max: int = 10,
                        z_max: int = 10) -> Dict[str, Any]:
        """
        Systematic search for solutions
        
        This is EXHAUSTIVE within bounds, not heuristic.
        """
        solutions = []
        counterexamples = []
        total_checked = 0
        
        for x in range(3, x_max + 1):
            for y in range(3, y_max + 1):
                for z in range(3, z_max + 1):
                    for a in range(1, a_max + 1):
                        for b in range(1, b_max + 1):
                            # Compute c from equation
                            lhs = a**x + b**y
                            
                            # Check if lhs is a perfect z-th power
                            c_candidate = round(lhs ** (1/z))
                            
                            # Verify with small tolerance
                            for c in [c_candidate - 1, c_candidate, c_candidate + 1]:
                                if c <= 0 or c > c_max:
                                    continue
                                
                                total_checked += 1
                                
                                if self.verify_equation(a, b, c, x, y, z):
                                    analysis = self.is_solution(a, b, c, x, y, z)
                                    solutions.append({
                                        "a": a, "b": b, "c": c,
                                        "x": x, "y": y, "z": z,
                                        "gcd": analysis["gcd"],
                                        "common_factors": analysis["common_factors"]
                                    })
                                    
                                    if analysis["is_counterexample"]:
                                        counterexamples.append((a, b, c, x, y, z))
        
        return {
            "total_checked": total_checked,
            "solutions_found": len(solutions),
            "counterexamples_found": len(counterexamples),
            "solutions": solutions,
            "counterexamples": counterexamples
        }
    
    def analyze_near_misses(self, 
                           a_max: int = 50,
                           tolerance: int = 10) -> List[Dict[str, Any]]:
        """
        Find "near-miss" solutions where a^x + b^y ≈ c^z
        
        This helps identify patterns in almost-solutions.
        """
        near_misses = []
        
        for x in range(3, 8):
            for y in range(3, 8):
                for z in range(3, 8):
                    for a in range(2, a_max):
                        for b in range(2, a_max):
                            lhs = a**x + b**y
                            c_approx = round(lhs ** (1/z))
                            
                            for c in range(max(2, c_approx - 2), c_approx + 3):
                                rhs = c**z
                                diff = abs(lhs - rhs)
                                
                                if 0 < diff <= tolerance:
                                    gcd_val = self.gcd_multiple(a, b, c)
                                    near_misses.append({
                                        "a": a, "b": b, "c": c,
                                        "x": x, "y": y, "z": z,
                                        "lhs": lhs,
                                        "rhs": rhs,
                                        "diff": diff,
                                        "gcd": gcd_val
                                    })
        
        return near_misses


# Example usage
if __name__ == "__main__":
    kernel = BealKernel()
    
    print("="*70)
    print("BEAL CONJECTURE VERIFICATION KERNEL")
    print("="*70)
    print()
    
    # Test known solution: 3^3 + 6^3 = 3^5 (gcd = 3)
    print("[Test 1] Known Solution (gcd > 1)")
    print("-"*70)
    result = kernel.is_solution(3, 6, 3, 3, 3, 5)
    print(f"Equation holds: {result['equation_holds']}")
    print(f"GCD: {result['gcd']}")
    print(f"Common factors: {result['common_factors']}")
    print(f"Is counterexample: {result['is_counterexample']}")
    print()
    
    # Search for solutions in small range
    print("[Test 2] Systematic Search (small range)")
    print("-"*70)
    search_result = kernel.search_solutions(a_max=20, b_max=20, c_max=30, 
                                           x_max=5, y_max=5, z_max=5)
    print(f"Total checked: {search_result['total_checked']}")
    print(f"Solutions found: {search_result['solutions_found']}")
    print(f"Counterexamples found: {search_result['counterexamples_found']}")
    
    if search_result['solutions']:
        print("\nSolutions:")
        for sol in search_result['solutions'][:5]:
            print(f"  {sol['a']}^{sol['x']} + {sol['b']}^{sol['y']} = {sol['c']}^{sol['z']}, gcd={sol['gcd']}")
    
    print()
    print("="*70)
    print("KERNEL STATUS: ACTIVE")
    print("Ready for systematic falsification experiments.")
    print("="*70)
