"""
Analyze the gcd=8,16 solutions to understand the pattern
"""

from beal_kernel_enhanced import BealKernelEnhanced

def analyze_counterexamples():
    """
    Analyze the solutions found with gcd=8,16
    """
    print("="*70)
    print("ANALYZING LEMMA B3 COUNTEREXAMPLES")
    print("="*70)
    print()
    
    kernel = BealKernelEnhanced()
    
    # The solutions found
    solutions = [
        (32, 8, 16, 3, 5, 4),   # gcd=8
        (8, 32, 16, 5, 3, 4),   # gcd=8
        (32, 16, 128, 4, 5, 3), # gcd=16
        (64, 16, 32, 4, 6, 5),  # gcd=16
        (16, 32, 128, 5, 4, 3), # gcd=16
        (16, 64, 32, 6, 4, 5),  # gcd=16
    ]
    
    print("Analyzing each solution:")
    print()
    
    for a, b, c, x, y, z in solutions:
        print(f"Solution: {a}^{x} + {b}^{y} = {c}^{z}")
        
        # Check equation
        lhs = a**x + b**y
        rhs = c**z
        print(f"  LHS = {lhs}, RHS = {rhs}, Equal: {lhs == rhs}")
        
        # Analyze structure
        analysis = kernel.analyze_gcd_structure(a, b, c)
        print(f"  gcd(a,b,c) = {analysis['gcd']}")
        print(f"  v₂(gcd) = {analysis['v2_gcd']}")
        print(f"  Odd part = {analysis['odd_part_gcd']}")
        
        # Check if a,b,c are all powers of 2
        def is_power_of_2(n):
            return n > 0 and (n & (n-1)) == 0
        
        a_pow2 = is_power_of_2(a)
        b_pow2 = is_power_of_2(b)
        c_pow2 = is_power_of_2(c)
        
        print(f"  a={a} is power of 2: {a_pow2}")
        print(f"  b={b} is power of 2: {b_pow2}")
        print(f"  c={c} is power of 2: {c_pow2}")
        
        if a_pow2 and b_pow2 and c_pow2:
            # Express as powers of 2
            import math
            a_exp = int(math.log2(a))
            b_exp = int(math.log2(b))
            c_exp = int(math.log2(c))
            
            print(f"  a = 2^{a_exp}, b = 2^{b_exp}, c = 2^{c_exp}")
            print(f"  Equation: (2^{a_exp})^{x} + (2^{b_exp})^{y} = (2^{c_exp})^{z}")
            print(f"  Simplifies: 2^{a_exp*x} + 2^{b_exp*y} = 2^{c_exp*z}")
            
            # Check if this is a valid 2-power equation
            lhs_exp_1 = a_exp * x
            lhs_exp_2 = b_exp * y
            rhs_exp = c_exp * z
            
            print(f"  2^{lhs_exp_1} + 2^{lhs_exp_2} = 2^{rhs_exp}")
            
            # For this to work, need min(lhs_exp_1, lhs_exp_2) + 1 = rhs_exp
            # (since 2^n + 2^m = 2^(min(n,m)+1) if n=m, otherwise dominated by max)
            
            if lhs_exp_1 == lhs_exp_2:
                expected_rhs = lhs_exp_1 + 1
                print(f"  Since exponents equal: 2·2^{lhs_exp_1} = 2^{expected_rhs}")
                print(f"  Expected rhs_exp: {expected_rhs}, Actual: {rhs_exp}")
                print(f"  Match: {expected_rhs == rhs_exp}")
        
        print()
    
    print("="*70)
    print("KEY OBSERVATION")
    print("="*70)
    print()
    print("ALL counterexamples have a,b,c as PURE POWERS OF 2")
    print()
    print("Pattern: 2^α·x + 2^β·y = 2^γ·z")
    print("This is a SPECIAL CASE where all bases share the SAME prime (2)")
    print()
    print("Refined Lemma B3:")
    print("  If a,b,c are NOT all powers of the same prime,")
    print("  then gcd = 2^k is impossible for a≠b")
    print()
    print("Exception: When a,b,c are all powers of same prime p,")
    print("  the equation reduces to p-power arithmetic")
    print("  (similar to symmetric a=b case)")
    print()
    print("="*70)


if __name__ == "__main__":
    analyze_counterexamples()
