"""
Beal Conjecture - Adversarial Testing Kernel

Extended testing suite to validate Prime Collision Necessity theorem:
1. Generate 10k+ random coprime attempts with high exponents
2. Check valuation debt Δ_p for all primes p ≤ 100
3. Test boundary cases (minimal exponents, large primes, symmetric)
4. Test p | x subcases where prime divides exponent
"""

from beal_kernel import BealKernel
from valuation_debt_kernel import p_adic_valuation, prime_factors_set, compute_valuation_debt
import json
import random
from collections import defaultdict


def generate_coprime_attempts(count=10000, a_max=1000, x_min=3, x_max=20):
    """
    Generate random coprime (a,b) pairs with high exponents
    
    Returns list of (a, b, x, y) tuples where gcd(a,b) = 1
    """
    attempts = []
    kernel = BealKernel()
    
    while len(attempts) < count:
        a = random.randint(2, a_max)
        b = random.randint(2, a_max)
        
        # Ensure coprimality
        if kernel.gcd(a, b) != 1:
            continue
        
        x = random.randint(x_min, x_max)
        y = random.randint(x_min, x_max)
        
        attempts.append((a, b, x, y))
    
    return attempts


def check_all_primes(a, b, c, x, y, z, p_max=100):
    """
    Compute Δ_p for all primes p ≤ p_max
    
    Returns dict: {prime: Δ_p}
    Flag any Δ_p = 0 as potential counterexample
    """
    # Generate primes up to p_max using sieve
    def sieve_of_eratosthenes(limit):
        is_prime = [True] * (limit + 1)
        is_prime[0] = is_prime[1] = False
        
        for i in range(2, int(limit**0.5) + 1):
            if is_prime[i]:
                for j in range(i*i, limit + 1, i):
                    is_prime[j] = False
        
        return [i for i in range(2, limit + 1) if is_prime[i]]
    
    primes = sieve_of_eratosthenes(p_max)
    
    sum_val = a**x + b**y
    
    debts = {}
    
    for p in primes:
        v_left = p_adic_valuation(sum_val, p)
        v_right = z * p_adic_valuation(c, p)
        
        delta = v_right - v_left
        
        debts[p] = {
            'v_left': v_left,
            'v_right': v_right,
            'delta': delta,
            'p_divides_a': p_adic_valuation(a, p) > 0,
            'p_divides_b': p_adic_valuation(b, p) > 0,
            'p_divides_c': p_adic_valuation(c, p) > 0
        }
    
    return debts


def test_boundary_cases():
    """
    Test boundary cases:
    - Minimal exponents (x=y=z=3)
    - Large primes
    - Symmetric cases (a=b, x=y=z)
    """
    kernel = BealKernel()
    results = {
        'minimal_exponents': [],
        'symmetric_cases': [],
        'large_primes': []
    }
    
    print("\n[Boundary Test 1] Minimal Exponents (x=y=z=3)")
    print("-" * 70)
    
    for a in range(2, 30):
        for b in range(a+1, 30):
            if kernel.gcd(a, b) != 1:
                continue
            
            sum_val = a**3 + b**3
            
            for z in [3]:
                c_approx = round(sum_val ** (1/z))
                
                for c in [c_approx - 1, c_approx, c_approx + 1]:
                    if c <= 0:
                        continue
                    
                    if kernel.verify_equation(a, b, c, 3, 3, z):
                        gcd_val = kernel.gcd_multiple(a, b, c)
                        
                        if gcd_val == 1:
                            debts = compute_valuation_debt(a, b, c, 3, 3, z)
                            results['minimal_exponents'].append({
                                'a': a, 'b': b, 'c': c,
                                'x': 3, 'y': 3, 'z': z,
                                'gcd': gcd_val,
                                'debts': debts
                            })
                            print(f"  ⚠️  Found: {a}^3 + {b}^3 = {c}^{z}, gcd=1")
    
    if not results['minimal_exponents']:
        print("  ✅ No coprime solutions found (as predicted)")
    
    print("\n[Boundary Test 2] Symmetric Cases (a=b, x=y=z)")
    print("-" * 70)
    
    for a in range(2, 20):
        for x in range(3, 8):
            sum_val = 2 * a**x
            
            for z in range(3, 8):
                c_approx = round(sum_val ** (1/z))
                
                for c in [c_approx - 1, c_approx, c_approx + 1]:
                    if c <= 0:
                        continue
                    
                    if kernel.verify_equation(a, a, c, x, x, z):
                        gcd_val = kernel.gcd_multiple(a, a, c)
                        
                        if gcd_val == 1:
                            debts = compute_valuation_debt(a, a, c, x, x, z)
                            results['symmetric_cases'].append({
                                'a': a, 'b': a, 'c': c,
                                'x': x, 'y': x, 'z': z,
                                'gcd': gcd_val,
                                'debts': debts
                            })
                            print(f"  ⚠️  Found: {a}^{x} + {a}^{x} = {c}^{z}, gcd=1")
    
    if not results['symmetric_cases']:
        print("  ✅ No coprime symmetric solutions found (as predicted)")
    
    return results


def test_p_divides_exponent():
    """
    Test cases where prime p divides exponent x or y
    Verify that valuation debt still holds (Δ_p > 0)
    """
    kernel = BealKernel()
    results = []
    
    print("\n[LTE Subcase Test] p | x or p | y")
    print("-" * 70)
    
    # Test small primes dividing exponents
    test_primes = [2, 3, 5]
    
    for p in test_primes:
        print(f"\n  Testing p={p} | x:")
        
        for a in range(2, 20):
            for b in range(a+1, 20):
                if kernel.gcd(a, b) != 1:
                    continue
                
                # Use exponent divisible by p
                x = p * 2  # e.g., 4, 6, 10
                y = 3
                
                if x <= 2 or y <= 2:
                    continue
                
                sum_val = a**x + b**y
                
                for z in range(3, 6):
                    c_approx = round(sum_val ** (1/z))
                    
                    for c in [c_approx - 1, c_approx, c_approx + 1]:
                        if c <= 0:
                            continue
                        
                        if kernel.verify_equation(a, b, c, x, y, z):
                            gcd_val = kernel.gcd_multiple(a, b, c)
                            
                            if gcd_val == 1:
                                debts = compute_valuation_debt(a, b, c, x, y, z)
                                
                                # Check if Δ_p > 0 for p | x
                                if p in debts and debts[p]['delta'] > 0:
                                    status = "✅ Δ_p > 0 (as predicted)"
                                elif p in debts and debts[p]['delta'] == 0:
                                    status = "❌ Δ_p = 0 (COUNTEREXAMPLE!)"
                                else:
                                    status = "⚠️  p not in debt list"
                                
                                results.append({
                                    'a': a, 'b': b, 'c': c,
                                    'x': x, 'y': y, 'z': z,
                                    'p': p,
                                    'gcd': gcd_val,
                                    'debts': debts,
                                    'status': status
                                })
                                
                                print(f"    {a}^{x} + {b}^{y} = {c}^{z}, gcd=1, {status}")
    
    if not results:
        print("  ✅ No coprime solutions found with p | x (as predicted)")
    
    return results


def run_adversarial_suite():
    """
    Main adversarial testing suite
    """
    print("=" * 70)
    print("ADVERSARIAL TESTING KERNEL - BEAL CONJECTURE")
    print("=" * 70)
    print()
    
    kernel = BealKernel()
    
    # Phase 1: High-exponent coprime attempts
    print("[Phase 1] Generating 10,000 Random Coprime Attempts")
    print("-" * 70)
    
    attempts = generate_coprime_attempts(count=10000, a_max=500, x_min=3, x_max=15)
    print(f"Generated {len(attempts)} coprime (a,b,x,y) tuples")
    
    counterexamples = []
    tested = 0
    
    for i, (a, b, x, y) in enumerate(attempts):
        if i % 1000 == 0:
            print(f"  Progress: {i}/{len(attempts)} tested...")
        
        sum_val = a**x + b**y
        
        # Check if sum is a perfect power
        for z in range(3, 10):
            c_approx = round(sum_val ** (1/z))
            
            if c_approx <= 0:
                continue
            
            # Check exact match
            if c_approx**z == sum_val:
                tested += 1
                gcd_val = kernel.gcd_multiple(a, b, c_approx)
                
                if gcd_val == 1:
                    # Potential counterexample! Check all primes
                    debts = check_all_primes(a, b, c_approx, x, y, z, p_max=100)
                    
                    # Check if any Δ_p = 0
                    zero_debts = [p for p, d in debts.items() if d['delta'] == 0]
                    
                    if zero_debts:
                        counterexamples.append({
                            'a': a, 'b': b, 'c': c_approx,
                            'x': x, 'y': y, 'z': z,
                            'zero_debt_primes': zero_debts
                        })
                        print(f"\n  ❌ POTENTIAL COUNTEREXAMPLE: {a}^{x} + {b}^{y} = {c_approx}^{z}")
                        print(f"     Primes with Δ_p = 0: {zero_debts}")
    
    print(f"\n  Total perfect power matches tested: {tested}")
    print(f"  Counterexamples found: {len(counterexamples)}")
    
    if counterexamples:
        print("\n  ⚠️  WARNING: Counterexamples detected!")
    else:
        print("\n  ✅ No counterexamples found (theorem holds)")
    
    # Phase 2: Boundary cases
    print("\n[Phase 2] Boundary Case Testing")
    print("-" * 70)
    boundary_results = test_boundary_cases()
    
    # Phase 3: p | x testing
    print("\n[Phase 3] LTE Subcase Testing (p | x)")
    print("-" * 70)
    lte_results = test_p_divides_exponent()
    
    # Summary
    print("\n" + "=" * 70)
    print("ADVERSARIAL TESTING COMPLETE")
    print("=" * 70)
    print(f"Total coprime attempts: {len(attempts)}")
    print(f"Perfect power matches: {tested}")
    print(f"Counterexamples: {len(counterexamples)}")
    print(f"Boundary cases tested: {len(boundary_results['minimal_exponents']) + len(boundary_results['symmetric_cases'])}")
    print(f"LTE subcases tested: {len(lte_results)}")
    
    if len(counterexamples) == 0:
        print("\n✅ THEOREM VALIDATED: No coprime solutions found in 10k+ tests")
    else:
        print(f"\n❌ THEOREM FAILED: {len(counterexamples)} counterexamples detected")
    
    return {
        'total_attempts': len(attempts),
        'perfect_power_matches': tested,
        'counterexamples': counterexamples,
        'boundary_results': boundary_results,
        'lte_results': lte_results
    }


if __name__ == "__main__":
    results = run_adversarial_suite()
    
    with open('adversarial_results.json', 'w') as f:
        json.dump(results, f, indent=2)
    
    print("\nResults saved to adversarial_results.json")
