import pandas as _pd
import numpy as _np
import os as _os
import warnings as _warnings
import datetime as _datetime

from pathlib import Path as _Path

try:
    from darq.pulseprocessing import PulseReader as _PulseReader
    from darq.pulseprocessing import RawPulse as _RawPulse
except ImportError:
    _warnings.warn("DARQ is not installed. Please install the `darq` package to use the PAQS module.", Warning)

from ._core import append as _append, dtypes as _dtypes, multiprocessing as _multiprocessing


def find(folders: list) -> _np.ndarray:
    folders = _np.array(folders, ndmin = 1)
    paths_set = set()

    for folder in folders:
        if _os.path.exists(folder):
            for dirpath, _, filenames in _os.walk(_Path(folder)):
                if "Summary.txt" in filenames:
                    paths_set.add(dirpath)
                    
        else:
            _warnings.warn(f"`{folder}` does not exist.", Warning)

    paths = _np.array(list(paths_set))

    return paths


def repair(paths: list) -> None:
    paths = _np.array(paths, ndmin = 1)

    for path in paths:
        try:
            _ = _PulseReader(path)

        except:
            try:
                _ = _PulseReader(path, repair = True)

            except Exception as exception:
                _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    return None


def index(paths: list, multiprocessing: bool = False, processes: int = _os.cpu_count()) -> _pd.DataFrame:
    paths = _np.array(paths, ndmin = 1)

    fitfiles = _multiprocessing(function = _read_pixel, iterables = _read_pixel_iterable(paths), 
                                multiprocessing = multiprocessing, processes = processes,
                                desc = 'index', unit = 'pixels')

    fitfile = _append(fitfiles)
    fitfile = _dtypes(fitfile)

    return fitfile


def onlinefits(paths: list) -> _pd.DataFrame:
    paths = _np.array(paths, ndmin = 1)
    fitfile_list = []

    for path in paths:
        try:
            fitresults_path = _os.path.join(path, 'Fitresults')

            if _os.path.isdir(fitresults_path):
                files = _os.listdir(fitresults_path)

                for file in files:
                    file_path = _os.path.join(fitresults_path, file)
                    onlinefit = _read_onlinefit(file_path)
                    if len(onlinefit) > 0:
                        fitfile_list.append(onlinefit)

                    else:
                        _warnings.warn(f"`{file_path}` is empty.", Warning)

                if not files:
                    _warnings.warn(f"`{fitresults_path}` is empty.", Warning)

            else:
                _warnings.warn(f"`{fitresults_path}` does not exist.", Warning)

        except Exception as exception:
            _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    fitfile = _append(fitfile_list)
    fitfile = _convert_onlinefit(fitfile)
    fitfile = _dtypes(fitfile)

    return fitfile


def summary(paths: list) -> _pd.DataFrame:
    paths = _np.array(paths, ndmin = 1)
    summary_list = []

    for path in paths:
        try:
            summary_data = _read_summary(path)
            if len(summary_data) > 0:
                summary_list.append(summary_data)

            else:
                _warnings.warn(f"`{path}` summary is empty.", Warning)

        except Exception as exception:
            _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    summary = _pd.DataFrame(summary_list)
    summary = _convert_summary(summary)

    return summary


def settings(paths: list) -> _pd.DataFrame:
    paths = _np.array(paths, ndmin = 1)
    settings_list = []

    for path in paths:
        try:
            settings = _read_settings(path)
            if len(settings) > 0:
                settings_list.extend(settings)

            else:
                _warnings.warn(f"`{path}` settings are empty.", Warning)

        except Exception as exception:
            _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    settings = _pd.DataFrame(settings_list)
    settings = _convert_settings(settings)

    return settings


def log(paths: list) -> _pd.DataFrame: 
    paths = _np.array(paths, ndmin = 1)
    log_list = []

    for path in paths:
        try:
            log = _read_log(path)
            if len(log) > 0:
                log_list.append(log)

            else:
                _warnings.warn(f"`{path}` log is empty.", Warning)

        except Exception as exception:
            _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    log = _pd.concat(log_list)
    log = _convert_log(log)

    return log


def _pulses(path: str, channel: str, polarity: str):
    try:
        pr = _PulseReader(path, channel, polarity)
        pr.mode = 'header'

    except Exception as exception:
        _warnings.warn(f"Failed to process `{path}`, `{channel}`, `{polarity}`: {exception}", Warning)
    
    for pulse in pr.pulses:
        try:
            yield pulse

        except Exception as exception:
            _warnings.warn(f"Failed to yield pulse: {exception}", Warning)


def _read_pixel_iterable(paths: list):
    for path in paths:
        try:
            pr = _PulseReader(path)

            for channel in pr.summaryInfo.columns:
                for polarity in pr.summaryInfo.index:
                    if pr.summaryInfo[channel][polarity] > 0:
                        yield path, channel, polarity

        except Exception as exception:
            _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)


def _read_pixel(path: str, channel: str, polarity: str) -> _pd.DataFrame:
    data_list = []

    try:
        for pulse in _pulses(path, channel, polarity):
            try:
                path_pulse = pulse.headerData['path']
                path_split = _os.path.normpath(path_pulse).split(_os.path.sep)

                data_pulse = {'folder': _os.path.normpath(_os.path.sep.join(path_split[:-5])),
                              'measurement': str(path_split[-5]),
                              'channel': str(path_split[-4]),
                              'polarity': str(path_split[-3]),
                              'signal': int(_os.path.splitext(path_split[-1])[0][1:]) - 1,
                              'timestamp': int(pulse.headerData['Timestamp'])}

                data_list.append(data_pulse)

            except Exception as exception:
                _warnings.warn(f"Failed to read pulse: {exception}", Warning)

    except Exception as exception:
        _warnings.warn(f"Failed to process `{path}`, `{channel}`, `{polarity}`: {exception}", Warning)

    fitfile = _pd.DataFrame(data_list)

    return fitfile


def _read_onlinefit(path: str) -> _pd.DataFrame:
    path_split = _os.path.normpath(path).split(_os.path.sep)
    filename, extension = _os.path.splitext(path_split[-1])

    onlinefit = _pd.DataFrame()

    try:
        if extension == '.csv':
            onlinefit = _pd.read_csv(path, delimiter=", ", index_col=False, engine='python')
        elif extension == '.fit':
            onlinefit = _pd.read_csv(path, delimiter="\t", skiprows=25)
            
        else:
            raise NotImplementedError(f"Extension {extension} not supported.")

        pixel_number = int(filename[5:])
        channel_number = int((pixel_number + 1) / 2)

        onlinefit['folder'] = str(_os.path.join(*path_split[:-3]))
        onlinefit['measurement'] = str(path_split[-3])
        onlinefit['channel'] = 'ADC' + str(channel_number)

    except Exception as exception:
        _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    return onlinefit


def _read_summary(path: str) -> dict:
    pr = _PulseReader(path)

    folder, measurement = _os.path.split(path)
    start, start_timestamp = _convert_time(pr.summaryHeader['Start time'])
    stop, stop_timestamp = _convert_time(pr.summaryHeader['Stop time'])

    if isinstance(start_timestamp, _datetime.datetime):
        start = start_timestamp

    if isinstance(stop_timestamp, _datetime.datetime):
        stop = stop_timestamp

    summary = {'folder': folder,
               'measurement': measurement,
               'start': start,
               'stop': stop,
               'duration': stop - start,
               'counts': int(pr.summaryHeader['Total number of saved traces']),
               'size': _get_size(path),
               'file_version': pr.summaryHeader['FileVersion'],
               'paqs_version': pr.summaryHeader['version'],
               'corrupt': _os.path.exists(_os.path.join(path, "SummaryRepaired.txt"))}

    return summary


def _read_settings(path: str) -> list:
    pr = _PulseReader(path)

    settings_list = []

    for channel in pr.scopeSettings:
        for polarity in pr.scopeSettings[channel]:

            folder, measurement = _os.path.split(path)
            settings = {'folder': folder,
                        'measurement': measurement,
                        'channel': channel,
                        'polarity': polarity}

            settings.update(pr.scopeSettings[channel][polarity])

            settings_list.append(settings)

    return settings_list


def _read_log(path: str) -> _pd.DataFrame:
    filepath = _os.path.join(path, 'Log.txt')
    folder, measurement = _os.path.split(path)

    log_list = []

    try:
        with open(filepath, 'r') as file:
            lines = file.readlines()

        for line in lines:
            if line[-1:] == '\n':
                line = line[:-1]
            time_string, comment_string = line.split(': ', maxsplit=1)

            time, timestamp = _convert_time(time_string)

            if isinstance(timestamp, _datetime.datetime):
                time = timestamp

            comment, timestamp = _convert_comment(comment_string)

            if isinstance(timestamp, _datetime.datetime):
                time = timestamp

            lineDict = {'folder': folder,
                        'measurement': measurement,
                        'time': time,
                        'comment': comment}

            log_list.append(lineDict)

    except Exception as exception:
        _warnings.warn(f"Failed to process `{path}`: {exception}", Warning)

    log = _pd.DataFrame(log_list)

    return log


def _convert_onlinefit(fitfile: _pd.DataFrame) -> _pd.DataFrame:
    fitfile.rename(columns={'Polarity': 'polarity', 'Timestamp(ADC)': 'timestamp'}, inplace=True)
    fitfile['polarity'] = fitfile['polarity'].replace({1: 'POSP', 0: 'BASE', -1: 'NEGP'})

    dropList = ['PulsNo']

    for column in fitfile.columns:
        if column.startswith('Tinfo ch. '):
            dropList.append(column)

    fitfile.drop(dropList, axis=1, inplace=True, errors='ignore')

    columns_dict = {'rel. Amplitude': 'amplitude',
                    'Chi2(ds)': 'chi2',
                    'Pretrigger-Offset': 'offset',
                    'Signal height (V)': 'height',
                    'Total area': 'area',
                    'Temperature info': 'temperature_info',
                    'Temperature info uncertainty': 'temperature_info_std',
                    'CommentID': 'comment',
                    'Relative pulse onset': 'onset',
                    'AverageCountStatus': 'rate',
                    'ADC temperature': 'temperature_adc',
                    'muonTrig [us]': 'muontrigger',
                    'muonTrig_prev [us]': 'muontrigger_previous',
                    'muonTrig_post [us]': 'muontrigger_post'}

    fitfile.rename(columns=columns_dict, inplace=True)

    return fitfile


def _convert_summary(summary: _pd.DataFrame) -> _pd.DataFrame:
    summary.sort_values('start', inplace=True)
    summary.reset_index(drop=True, inplace=True)

    dtype_dict = {'folder': str,
                  'measurement': str,
                  'counts' : int,
                  'size' : int,
                  'file_version': str,
                  'paqs_version': str,
                  'corrupt': bool}
    summary = summary.astype(dtype_dict, errors='ignore')

    category_list = ['folder', 
                     'measurement', 
                     'file_version', 
                     'paqs_version']
    summary = summary.astype({key : 'category' for key in category_list}, errors='ignore')

    return summary


def _convert_settings(settings: _pd.DataFrame) -> _pd.DataFrame:
    columns_dict = {'Sampling rate': 'samplingrate',
                    'Oversampling': 'oversampling', 
                    'Sample length': 'samples',
                    'Pretrigger delay': 'pretrigger', 
                    'Sample length (fast trace)': 'samples_fast', 
                    'Pretrigger delay (fast trace)': 'pretrigger_fast',
                    'Voltage range': 'voltagerange',
                    'Bit range': 'coderange', 
                    'y offset': 'offset',
                    'Input termination': 'termination',
                    'Trigger threshold': 'threshold',
                    'Trigger on/off': 'trigger'}
    settings.rename(columns=columns_dict, inplace=True, errors='ignore')

    dtype_dict = {'folder': str,
                  'measurement': str,
                  'channel': str, 
                  'polarity': str,
                  'samplingrate': float,
                  'oversampling': int,
                  'samples': int,
                  'pretrigger': int,
                  'samples_fast': int, 
                  'pretrigger_fast': int,
                  'voltagerange': float,
                  'coderange': int, 
                  'offset': int,
                  'termination': int,
                  'threshold': int, 
                  'trigger': bool}
    settings = settings.astype(dtype_dict, errors='ignore')

    category_list = ['folder', 
                     'measurement', 
                     'channel', 
                     'polarity', 
                     'samplingrate', 
                     'oversampling', 
                     'samples', 
                     'pretrigger', 
                     'samples_fast', 
                     'pretrigger_fast', 
                     'voltagerange', 
                     'coderange', 
                     'offset', 
                     'termination', 
                     'threshold']
    settings = settings.astype({key : 'category' for key in category_list}, errors='ignore')

    return settings


def _convert_log(log: _pd.DataFrame) -> _pd.DataFrame:
    log.sort_values('time', inplace=True)
    log.reset_index(drop=True, inplace=True)

    dtype_dict = {'folder': str,
                  'measurement': str,
                  'comment': str}
    log = log.astype(dtype_dict, errors='ignore')

    category_list = ['folder', 
                     'measurement', 
                     'comment']
    log = log.astype({key : 'category' for key in category_list}, errors='ignore')

    return log


def _convert_time(string: str) -> tuple:
    time_split = string.split(', ', maxsplit=1)

    if len(time_split) == 2:
        time_string, timestamp_string = time_split
    else:
        time_string = time_split[0]
        timestamp_string = ''

    time = _datetime.datetime.strptime(time_string, '%H:%M:%S on %d.%m.%Y')

    if timestamp_string.startswith('UNIX time '):
        timestamp = _datetime.datetime.fromtimestamp(int(timestamp_string.split()[-1]))
    else:
        timestamp = None

    return time, timestamp


def _convert_comment(string: str) -> tuple:
    string_split = string.split('; ', maxsplit=1)

    if len(string_split) == 1:
        timestamp = None
    else:
        if string_split[1].startswith('Unix time [ms]: '):
            try:
                timestamp = _datetime.datetime.fromtimestamp(float(string_split[1].split()[-1]) / 1e3)
            except Exception as exception:
                _warnings.warn(f"Failed to convert string `{string_split[1]}`: {exception}", Warning)
                timestamp = None

    comment = string_split[0]

    return comment, timestamp


def _get_size(path: str) -> int:
    size = 0

    for dirpath, _, filenames in _os.walk(_Path(path)):
        for filename in filenames:
            filepath = _os.path.join(dirpath, filename)
            if not _os.path.islink(filepath):
                size += _os.path.getsize(filepath)

    return size