# create dummy functions to generate phreeqc input-string from the function parameters


phreeqify.solution <- function(n = 1, units = "mg/L", ...){
  # gerenerate phreeqc string for SOLUTION block
  # default solution number n=1, units="mg/L"
  # all arguments are passed as named arguments (e.g. Ca = 1) with default phreeqc syntax
  # defaults are from phreeqc (i.e. temperature = 25, pH = 7, pressure = 1 atm, pe = 4,  density = 1 kg/L, alkalinity = 0, water = 1.0 kg)
  # see phreeqc manual for details
  # example: phreeqify.solution(Ca=1, Mg=2, Cl=3) 

  args <- list(...)
  phr_str <-  c(paste0("SOLUTION ", n),
                paste0("units ", units))
  
  for (i in 1:length(args)){
    phr_str <- c(phr_str, paste(names(args)[[i]], args[[i]]))
  }
                
return(phr_str)
  }


phreeqify.equi <- function(n = 1, si = NULL, ...){
  # TODO: allow ppmv, % and or Pa as input for gas phases
  
  # gerenerate phreeqc string for EQUILIBRIUM_PHASES block
  # default equilibrium phase number n=1, 
  # default target saturation index si=0
  # all arguments are passed as named arguments (e.g. goethite = 100) with default phreeqc syntax
  # target si different from 0 either provided as a single value to be applied to all phases, or a named list or vector of values for each phases
  # note for gasses the target si is given in log atmosphere; e.g. si = list("CO2(g)" = -3.5)
  # see phreeqc manual for details
  
  
  args <- list(...)
  
  if (is.null(si)){
    si_list <- as.list(rep(0, length(args)))  # default target SI is 0
    names(si_list) <- names(args)
  } else if(is.null(names(si))){
    if (length(si) == 1){
      si_list <- as.list(rep(si, length(args)))  # apply target SI to all phases
      names(si_list) <- names(args)
    } else {
      warning("I'm confused: you provided more than 1 SI value but not named them. I applied the First si value to all phases. Different SI's for different phases can be provided with a named vector or list only.")
      si_list <- as.list(rep(si[1], length(args))) # apply first value of si vector as target SI to all phases
      names(si_list) <- names(args)
    }
  } else {
    si_list <- as.list(rep(0, length(args)))  # default target SI is 0
    names(si_list) <- names(args)
    si_list[names(si)] <- si
  }
  
  phr_str <-  c(paste("USE SOLUTION", n), # TODO: check if this is correct or needed 
                paste("EQUILIBRIUM_PHASES", n))
  
  for (i in 1:length(args)){
    phr_str <- c(phr_str, paste(names(args)[[i]], si_list[[i]],args[[i]]))
  }
  
  return(phr_str)
}


phreeqify.exchange <- function(n = 1, solution = 1, ...){
  # gerenerate phreeqc string for EXCHANGE block
  # default exchange number n=1, 
  # equilibrates by defauls with the solution with the same number n
  # default behaviour is the implicit way of defining the exchange complex, i.e. provide total CEC (=X in phreeqc) and equilibrate this with a predifined solution
  # total CEC can be provided as X = ... (phreeqc default) or as CEC = ...
  # when solution = NULL the exchange complex has to be defined explicitely
  # see phreeqc manual for details
  
  args <- list(...)
  if (is.element("CEC", names(args))){
    names(args)[which(names(args)=="CEC")] <- "X"
  }
  
  
  phr_str <-  c(paste("EXCHANGE", n))
  
  # when solution = NULL, the exchange complex has to be defined explicitely 
  if(!is.null(solution)){
    phr_str <- c(phr_str, paste("-equilibrate with solution", solution))
  }
  
  for (i in 1:length(args)){
    phr_str <- c(phr_str, paste(names(args)[[i]], args[[i]]))
  }
  
  return(phr_str)
}


phreeqify.kinetics.weathering <- function(n = 1, massfraction, rockmass, SSA, mineral_data){
  # gerenerate phreeqc string for KINETICS block of the Palandra and Kharaka/Kelland/Shrinking spere weathering model
  # default kinetics number n=1
  # massfraction: named vector of mass fractions of minerals in the rock
  # mineral_data: data frame with mineral properties (density, ...)
  # Wr_data: data frame with weathering rate parameters (k_acid, k_neutr, k_base, Ea_acid, Ea_neutr, Ea_base, n_acid, n_base)

  
  m0 <- (massfraction/mineral_data[names(massfraction), "molarmass"])*(rockmass)   # [mol] initial mineral in total volume
  afrac <- relativemass2relativesurface(massfraction, SSA, mineral_data) # [-] relative surface area of individual minerals
  
  phr_str <-  paste("KINETICS", n)
  
  for (mineral in names(massfraction)){
    phr_str_i <-  c(mineral,
                    # paste("-formula", mineral_data[mineral, "atomicformula"]),
                    paste("-m ",     m0[mineral]),
                    paste("-m0",  m0[mineral]), 
                    "-tol      1e-008",
                    paste("-parms", SSA, rockmass, afrac[mineral])) 
    
    phr_str <- c(phr_str, phr_str_i)
  }
  
  return(phr_str)
}

phreeqify.rates.weathering <- function(n = 1, massfraction, Wr_data){
  # gerenerate phreeqc string for RATES block of the Palandra and Kharaka/Kelland/Shrinking sphere weathering model
  # default rates number n=1
  # massfraction: named vector of mass fractions of minerals in the rock
  # Wr_data: data frame with weathering rate parameters (k_acid, k_neutr, k_base, Ea_acid, Ea_neutr, Ea_base, n_acid, n_base)
  
  phr_str <- paste("RATES", n)
  
  for (mineral in names(massfraction)){ 
    phr_str_i <- c(mineral,
                "-start",  
                "10 REM Rate law from Palandri and Kharaka (2004)",
                "20 REM Affinity ON",
                "30 moles = 0",
                paste0("40 IF(M <= 0 OR SI('", mineral, "') > 0) THEN GOTO 270"), 
                "50 SSA = PARM(1)",
                "60 rockmass = PARM(2)",
                "80 afrac = PARM(3)", 
                "90 a0 = SSA*rockmass*afrac",
                "110 dif_temp = 1/TK-1/298.15",
                paste ("120 k_acid =", Wr_data[mineral, "k_acid"]), paste ("130 eapp_acid =", Wr_data[mineral, "Ea_acid"]), paste ("140 n_acid =", Wr_data[mineral, "n_acid"]),
                paste ("150 k_neut =", Wr_data[mineral, "k_neutr"]), paste ("160 eapp_neut =", Wr_data[mineral, "Ea_neutr"]), paste ("170 k_base =", Wr_data[mineral, "k_base"]),
                paste ("180 eapp_base =", Wr_data[mineral, "Ea_base"]),  paste ("190 n_base =", Wr_data[mineral, "n_base"]),
                "200 hplus = MOL('H+')",
                "210 r_acid = k_acid*EXP((-eapp_acid/8.314e-3)*dif_temp)*(hplus^n_acid)",
                "220 r_neut = k_neut*EXP((-eapp_neut/8.314e-3)*dif_temp)",
                "230 r_base = k_base*EXP((-eapp_base/8.314e-3)*dif_temp)*(hplus^n_base)",
                "240 r_all = r_acid+r_neut+r_base",
                paste0("250 rate = (a0)*(M/M0)^0.67*r_all*(1-SR('", mineral,"'))"), 
                "260 moles = rate*TIME",
                "270 SAVE moles",
                "-end")
    phr_str <- c(phr_str, phr_str_i)
  }
  
  return(phr_str)
}


phreeqify.kinetics.soc <- function(n=1, state_init, all_pars){
  # gerenerate phreeqc string for KINETICS block of the SOC model
  # default kinetics number n=1
  # state_init: named vector of initial state variables
  # Met_pars, Str_pars, Slow_pars, Pass_pars, DOCf_pars: named vectors of parameters associated with the different pools in the SOC model

  # intermediate parameter vectors since a single BASIC function can handle max 10 parameter values
  Met_par_names <- c("I", "litterfrac", "Tmet", "temp", "moist")
  Met_pars <- all_pars[Met_par_names]
  names(Met_pars) <- Met_par_names
  
  
  Str_par_names <- c("I", "litterfrac", "Tstr", "LC")
  Str_pars <- all_pars[Str_par_names]
  names(Str_pars) <- Str_par_names
  
  Slow_par_names <- c("ss", "as", "Tstr", "TDOCf", "Ts", "LC")
  Slow_pars <- all_pars[Slow_par_names]
  names(Slow_pars) <- Slow_par_names
  
  Pass_par_names <- c("ap", "sp", "Ts", "TDOCf", "Tp")
  Pass_pars <- all_pars[Pass_par_names]
  names(Pass_pars) <- Pass_par_names
  
  DOCf_par_names <- c("ma", "stra", "sa", "pa", "Tmet", "Tstr", "Ts", "Tp", "LC", "TDOCf")
  DOCf_pars <- all_pars[DOCf_par_names]
  names(DOCf_pars) <- DOCf_par_names
  
  DOCa_par_names <- c("soil_C")
  DOCa_pars <- all_pars[DOCa_par_names]
  names(DOCa_pars) <- DOCa_par_names
  
  phr_str <- c(paste("KINETICS", n), 
                    "Met",
                    "-formula H 0.0",
                    paste("-m0",state_init["Met"]), 
                    paste("-parms", paste(Met_pars, collapse=" ")), 
                    
                    "Str",
                    "-formula H 0.0",
                    paste("-m0",state_init["Str"]),
                    paste("-parms", paste(Str_pars, collapse=" ")),
                    
                    "Slow",
                    "-formula H 0.0",
                    paste("-m0",state_init["S"]),
                    paste("-parms", paste(Slow_pars, collapse=" ")),
                    
                    "Pass",
                    "-formula H 0.0",
                    paste("-m0",state_init["P"]),
                    paste("-parms", paste(Pass_pars, collapse=" ")),
                    
                    "SumDOC",
                    "-formula H 0.0",
                    paste("-m0",state_init["SumDOC"]),
                    paste("-parms", paste(DOCf_pars, collapse=" ")),
                    "-step 86400 in 24",
                    "-tol      1e-008")
  
  return(phr_str)
   
}


phreeqify.rates.soc <- function(all_pars){ #} Q10, V_cell, total_porosity, clay, soilmass, area){
  # gerenerate phreeqc string for RATES block of the SOC model
  # Met_pars, Str_pars, Slow_pars, Pass_pars, DOCf_pars: named vectors of parameters associated with the different pools in the SOC model
  # all_pars: named vector of all parameters in the model, only Q10, V_cell, total_porosity, clay, soilmass, area are used
  
  # intermediate parameter vectors since a single BASIC function can handle max 10 parameter values
  Met_par_names <- c("I", "litterfrac", "Tmet", "temp", "moist")
  Str_par_names <- c("I", "litterfrac", "Tstr", "LC")
  Slow_par_names <- c("ss", "as", "Tstr", "TDOCf", "Ts", "LC")
  Pass_par_names <- c("ap", "sp", "Ts", "TDOCf", "Tp")
  DOCf_par_names <- c("ma", "stra", "sa", "pa", "Tmet", "Tstr", "Ts", "Tp", "LC", "TDOCf")
  DOCa_par_names <- c("soil_C")
  
  
  Met_par_names <- c("I", "litterfrac", "Tmet", "temp", "moist")
  Met_pars <- all_pars[Met_par_names]
  names(Met_pars) <- Met_par_names
  
  
  Str_par_names <- c("I", "litterfrac", "Tstr", "LC")
  Str_pars <- all_pars[Str_par_names]
  names(Str_pars) <- Str_par_names
  
  Slow_par_names <- c("ss", "as", "Tstr", "TDOCf", "Ts", "LC")
  Slow_pars <- all_pars[Slow_par_names]
  names(Slow_pars) <- Slow_par_names
  
  Pass_par_names <- c("ap", "sp", "Ts", "TDOCf", "Tp")
  Pass_pars <- all_pars[Pass_par_names]
  names(Pass_pars) <- Pass_par_names
  
  DOCf_par_names <- c("ma", "stra", "sa", "pa", "Tmet", "Tstr", "Ts", "Tp", "LC", "TDOCf")
  DOCf_pars <- all_pars[DOCf_par_names]
  names(DOCf_pars) <- DOCf_par_names
  
  DOCa_par_names <- c("soil_C")
  DOCa_pars <- all_pars[DOCa_par_names]
  names(DOCa_pars) <- DOCa_par_names

  with(as.list(all_pars), {
    phr_str <- c("RATES",
                 
                 "Met",
                 "-start",
                 "5 Met = M",
                 paste0("10 I = parm(", match("I", names(Met_pars)),")"),
                 paste0("20 litterfrac = parm(", match("litterfrac", names(Met_pars)),")"),
                 paste0("30 Tmet = parm(", match("Tmet", names(Met_pars)),")"),
                 paste0("40 temp = parm(", match("temp", names(Met_pars)),")"),
                 paste0("41  basalt_ctrl =", 1) ,
                 paste0("55 temp_ctrl = basalt_ctrl*",Q10,"^(((TK-273.15 - 10) )/ 10) "),
                 paste0("61 moist =    TOT('water')/",V_cell),
                 paste0("62 moist_normalized = moist /", total_porosity),
                 "65 moist_ctrl = 1", #set to 1 as moisture was not limiting in this study according to gompertz relationship
                 "70 if (moist_ctrl>1) then moist_ctrl=1",
                 "75 if (moist_ctrl<0.25) then moist_ctrl=0.25",
                 "76 pH_sim = -LA('H+')",
                 "77 pH_ctrl = (1.046353/(1+exp((-(pH_sim-5.3453)/0.8643))))",
                 "78 if (pH_ctrl>1) then pH_ctrl=1",
                 "80 Met_rate = I*litterfrac-(1/Tmet)*Met*temp_ctrl*moist_ctrl*pH_ctrl",
                 "90 PUT(temp_ctrl, 1)",
                 "95 PUT(temp, 10)",
                 "97 PUT(moist, 11)",
                 "98 PUT(I,12)",
                 "100 PUT(moist_ctrl, 2)",
                 "110 PUT(Met_rate,3)",
                 "111 PUT(pH_ctrl ,20)",
                 
                 "120 SAVE - Met_rate * time",
                 "-end",
                 
                 
                 
                 "Str",
                 "-start",
                 "5 Str = M",
                 paste0("10 I = parm(", match("I", names(Str_pars)),")"),
                 paste0("20 litterfrac = parm(", match("litterfrac", names(Str_pars)),")"),
                 paste0("30 Tstr = parm(", match("Tstr", names(Str_pars)),")"),
                 paste0("35 LC = parm(", match("LC", names(Str_pars)),")"),
                 "40 temp_ctrl = GET(1)",
                 "41 pH_ctrl = GET(20)",
                 "50 moist_ctrl = GET(2)",
                 "60 Str_rate = I*(1-litterfrac)-(1/Tstr)*Str*exp( -3.0*LC)*temp_ctrl*moist_ctrl*pH_ctrl",
                 "70 PUT(Str_rate,4)",
                 "75 PUT(exp(-3.0*LC), 16)",
                 "80 SAVE - Str_rate * time",
                 "-end",
                 
                 "Slow",
                 "-start",
                 "5 Slow = M",
                 "10 SumDOC = KIN('SumDoc')",
                 "20 Str = KIN('Str')",
                 paste0("30 ss = parm(", match("ss", names(Slow_pars)),")"),
                 paste0("31 clay =", clay,"+ (EQUI('Chlorite')*555.79+ 366.57*EQUI('CaMontmorillonite'))/",soilmass*1000), 
                 "35 as = 1 - (.85-.68*clay) - ap ",
                 paste0("40 Tstr = parm(", match("Tstr", names(Slow_pars)),")"),
                 "41 pH_ctrl = GET(20)",
                 paste0("45 TDOCf = parm(", match("TDOCf", names(Slow_pars)),")"),
                 paste0("50 Ts = parm(", match("Ts", names(Slow_pars)),")"),
                 paste0("60 LC = parm(", match("LC", names(Slow_pars)),")"),
                 # compute the adsorption-desorption parameters
                 "130 soilC = (Met +  Str +  SumDOC + Slow + Pass)",
                 
                 #convert to g C / kg soil 
                 # SoilC = SoilC_tonha*0.22*10^(6)/10000/35 #g/kg 
                 paste0("131  soilC_g_kg = soilC*10^(6)*",area,"/10000/",soilmass) ,             
                 "140 b = exp(0.32*log10(soilC_g_kg) + 3.3)",
                 #mol goethite = mol Fe *gFe/mol Fe = g Fe *1000 = mg Fe / soilmass( in kg) = Fe in mg/kg
                 paste0("150 Fe = (EQUI('Goethite')+EQUI('Fe(OH)3(a)')+2*KIN('hematite'))*55.845*1000/",soilmass),
                 paste0("160 Al = (EQUI('Gibbsite')+EQUI('Al(OH)3(a)'))*26.98*1000/", soilmass),
                 "170 Qmax = exp(0.55*log(Fe + Al) + 5.7)",  #mg/kg
                 "180 k = exp(-0.64*log(Qmax)-1.68)",        #mg/kg?
                 paste0("181  SumDOC_mg_kg = SumDOC*10^(9)*",area,"/10000/",soilmass) ,
                 #idem conversion as above, tonC ha-1 --> gC/kg        
                 # SUMDOC was in g/kg here i think  (not divided by 1000),
                 #de rest in mg/kg denk ik !
                 "190 D = (1/k/1000 + Qmax/1000 -SumDOC_mg_kg)^2 + 4/k/1000*SumDOC_mg_kg", #(g/kg)^2
                 "195 DOCf_mg_kg = 0.5*(-(1/k/1000 + Qmax/1000 - SumDOC_mg_kg) + sqrt(D))", #g/kg 
                 paste0("196  DOCf = DOCf_mg_kg*10000*",soilmass,"/10^(9)/",area) ,
                 #convert terug naar ton C ha-1
                 "200 temp_ctrl = GET(1)",
                 "210 moist_ctrl = GET(2)",
                 "220 Slow_rate = ss*(1/Tstr)*Str*exp( -3.0*LC)*LC*temp_ctrl*moist_ctrl*pH_ctrl + as*DOCf*(1/TDOCf)*temp_ctrl*moist_ctrl*pH_ctrl - Slow*(1/Ts)*temp_ctrl*moist_ctrl*pH_ctrl",
                 "230 PUT(b, 3)",
                 "235 PUT(k, 4)",
                 "237 PUT(Qmax,5)",
                 "240 SAVE - Slow_rate * time",
                 "-end",
                 
                 "Pass",
                 "-start",
                 "5 Pass = M",
                 "10 SumDOC = KIN('SumDoc')",
                 "20 Slow = KIN('Slow')",
                 paste0("30 ap = parm(", match("ap", names(Pass_pars)),")"),
                 paste0("35 sp = parm(", match("sp", names(Pass_pars)),")"),
                 paste0("40 TDOCf = parm(", match("TDOCf", names(Pass_pars)),")"),
                 "41 pH_ctrl = GET(20)",
                 paste0("45 Ts = parm(", match("Ts", names(Pass_pars)),")"),
                 paste0("50 Tp = parm(", match("Tp", names(Pass_pars)),")"),
                 # compute the adsorption-desorption parameters
                 "130 soilC = (Met +  Str +  SumDOC + Slow + Pass)",
                 
                 paste0("131  soilC_g_kg = soilC*10^(6)*",area,"/10000/",soilmass) ,           
                 "140 b = exp(0.32*log10(soilC_g_kg) + 3.3)",
                 #mol goethite = mol Fe *gFe/mol Fe = g Fe *1000 = mg Fe / soilmass( in kg) = Fe in mg/kg
                 paste0("150 Fe = (EQUI('Goethite')+EQUI('Fe(OH)3(a)'))*55.845*1000/",soilmass),
                 paste0("160 Al = (EQUI('Gibbsite')+EQUI('Al(OH)3(a)'))*26.98*1000/", soilmass),
                 "170 Qmax = exp(0.55*log(Fe + Al) + 5.7)",  #mg/kg
                 "180 k = exp(-0.64*log(Qmax)-1.68)",        #mg/kg
                 paste0("181  SumDOC_mg_kg = SumDOC*10^(9)*",area,"/10000/",soilmass) ,
                 #idem conversie als hierboven, tonC ha-1 --> gC/kg        
                 # SUMDOC stond in g/kg hier denk ik (niet gedeeld door 1000), de rest in mg/kg denk ik !
                 "190 D = (1/k/1000 + Qmax/1000 -SumDOC_mg_kg)^2 + 4/k/1000*SumDOC_mg_kg", #(g/kg)^2
                 "195 DOCf_mg_kg = 0.5*(-(1/k/1000 + Qmax/1000 - SumDOC_mg_kg) + sqrt(D))", #g/kg       paste0("196  DOCf = DOCf_g_kg*10000*",soilmass,"/10^(6)/",area) ,
                 #convert terug naar ton C ha-1
                 "200 temp_ctrl = GET(1)",
                 "210 moist_ctrl = GET(2)",
                 "220 Pass_rate = ap*DOCf*(1/TDOCf)*temp_ctrl*moist_ctrl*pH_ctrl + sp*Slow*(1/Ts)*temp_ctrl*moist_ctrl*pH_ctrl - Pass*(1/Tp)*temp_ctrl*moist_ctrl*pH_ctrl",
                 "240 SAVE - Pass_rate * time",
                 "-end",
                 
                 "SumDoc",
                 "-start",
                 "5 SumDOC = M",
                 "10 Met = KIN('Met')",
                 "20 Str = KIN('Str')",
                 "30 Slow = KIN('Slow')",
                 "40 Pass = KIN('Pass')",
                 "41 pH_ctrl = GET(20)",
                 paste0("50 ma = parm(", match("ma", names(DOCf_pars)),")"),
                 paste0("55 stra = parm(", match("stra", names(DOCf_pars)),")"),
                 paste0("60 sa = parm(", match("sa", names(DOCf_pars)),")"),
                 paste0("70 pa = parm(", match("pa", names(DOCf_pars)),")"),
                 paste0("80 Tmet = parm(", match("Tmet", names(DOCf_pars)),")"),
                 paste0("90 Tstr = parm(", match("Tstr", names(DOCf_pars)),")"),
                 paste0("100 Ts = parm(", match("Ts", names(DOCf_pars)),")"),
                 paste0("110 Tp = parm(", match("Tp", names(DOCf_pars)),")"),
                 paste0("120 LC = parm(", match("LC", names(DOCf_pars)),")"),
                 paste0("125 TDOCf = parm(", match("TDOCf", names(DOCf_pars)),")"),
                 # compute the adsorption-desorption parameters
                 "130 soilC = (Met +  Str +  SumDOC + Slow + Pass)",
                 paste0("131  soilC_g_kg = soilC*10^(6)*",area,"/10000/",soilmass) ,           
                 "140 b = exp(0.32*log10(soilC_g_kg) + 3.3)",
                 #mol goethite = mol Fe *gFe/mol Fe = g Fe *1000 = mg Fe / soilmass( in kg) = Fe in mg/kg
                 paste0("150 Fe = (EQUI('Goethite')+EQUI('Fe(OH)3(a)'))*55.845*1000/",soilmass),
                 paste0("160 Al = (EQUI('Gibbsite')+EQUI('Al(OH)3(a)'))*26.98*1000/", soilmass),
                 "170 Qmax = exp(0.55*log(Fe + Al) + 5.7)",  #mg/kg
                 "180 k = exp(-0.64*log(Qmax)-1.68)",        #mg/kg
                 paste0("181  SumDOC_mg_kg = SumDOC*10^(9)*",area,"/10000/",soilmass) ,
                 #idem conversie als hierboven, tonC ha-1 --> gC/kg        
                 # SUMDOC stond in g/kg hier denk ik (niet gedeeld door 1000), de rest in mg/kg denk ik !
                 "190 D = (1/k/1000 + Qmax/1000 -SumDOC_mg_kg)^2 + 4/k/1000*SumDOC_mg_kg", #(g/kg)^2
                 "195 DOCf_mg_kg = 0.5*(-(1/k/1000 + Qmax/1000 - SumDOC_mg_kg) + sqrt(D))", #g/kg    
                 paste0("196  DOCf = DOCf_mg_kg*10000*",soilmass,"/10^(9)/",area) ,
                 "197 DOCa_mg_kg = Qmax/1000*DOCf_mg_kg/(1/k/1000+DOCf_mg_kg)",
                 #(g/kg)^2 / g/kg ? outcome moet g/kg zijn
                 paste0("198  DOCa = DOCa_mg_kg*10000*",soilmass,"/10^(9)/",area) ,
                 "200 temp_ctrl = GET(1)",
                 "210 moist_ctrl = GET(2)",
                 "220 SumDOC_rate = ma*(1/Tmet)*Met*temp_ctrl*moist_ctrl*pH_ctrl + stra*(1/Tstr)*Str*exp( -3.0*LC)*(1.0-LC)*temp_ctrl*moist_ctrl*pH_ctrl  + sa*Slow*(1/Ts)*temp_ctrl*moist_ctrl*pH_ctrl + pa*Pass*(1/Tp)*temp_ctrl*moist_ctrl*pH_ctrl - DOCf*(1/TDOCf)*temp_ctrl*moist_ctrl*pH_ctrl",
                 "230 PUT(DOCa, 7)",
                 "235 PUT(DOCf, 8)",
                 "237 PUT(SoilC,9)",
                 "240 PUT(exp( -3.0*LC)*(1.0-LC),17)",
                 "250 SAVE - (SumDOC_rate) * time",
                 "-end"
    )
    return(phr_str)
    
  })
  
}

phreeqify.surf <- function(){
  # TODO: currently not surface complexation is implemented in the coupled EW/SOC model
  
}

phreeqify.output <- function(){
  # TODO: this can be made a function with arguments,  e.g. list of equilibrium phases names, list of kinetic reactants names, list of solution names, etc.
  # for the time being this will work
  
  phr_str <- c("SELECTED_OUTPUT", 
                     "-reset false",
                     "-temperature",
                     "-high_precision true",
                     "-time",
                     "-pH",
                     "-water",
                     "-molalities           CaX2  KX  MgX2  NaX FeX2 MnX2 SrX2 AlX3 Hfo_sOHCa+2 Hfo_wOCa+ Hfo_wOMg+ Hfo_s Hfo_w Hfo_wHCO3 Hfo_wCO3- Hfo_sOHCO3-", 
                     "-step true",
                     "-totals H2O Fe Al Ca Mg Si Alkalinity C(4)",
                     "-solution",
               "    -si calcite dolomite olivine forsterite chlorite Nesquehonite",
                     "-equilibrium_phases goethite gibbsite CO2(g) calcite SiO2(a) Fe(OH)3(a) Al(OH)3(a) Nesquehonite CaMontmorillonite Chlorite",
                     "-kinetic reactants Met Str Slow Pass Docf Doca SumDOC olivine plagioclase augite illite chlorite hematite forsterite",
                     "USER_PUNCH",
                     "-headings temp_ctrl moist_ctrl b k Qmax temp moist I DOCf DOCa soilC RE flux_raw pH_ctrl",
                     "-start",
                     "10 PUNCH GET(1)",
                     "20 PUNCH GET(2)",
                     "25 PUNCH GET(3)",
                     "27 PUNCH GET(4)",
                     "28 PUNCH GET(5)",
                     "30 PUNCH GET(10)",
                     "40 PUNCH GET(11)",
                     "50 PUNCH GET(12)",
                     "80 PUNCH GET(7)",
                     "85 PUNCH GET(8)",
                     "90 PUNCH GET(9)",
                     "110 PUNCH GET(14)",
                     "120 PUNCH GET(15)",
                     "130 PUNCH GET(20)",
               
                     "-end")
  return(phr_str)
}
