import numpy as np
import platform
import time

# --- 1. HARDWARE ABSTRACTION LAYER (HAL) ---
try:
    # Check if we are on the actual hardware (Linux/Jetson) or a dev machine (Mac)
    if platform.system() == "Linux":
        import spidev
        HAS_HARDWARE = True
    else:
        HAS_HARDWARE = False
except (ImportError, ModuleNotFoundError):
    HAS_HARDWARE = False

class VirtualSPI:
    """ Fakes the SPI bus for development on macOS """
    def open(self, bus, device): pass
    def xfer2(self, data): pass
    def close(self): pass

# --- 2. SOVEREIGN CORE (The Logic Gate) ---
class LaciaSovereign:
    def __init__(self, phi=0):
        self.phi = phi
        self.V = [1, 5, 7, 11, 13, 17, 19, 23]
        self.P = [5, 7, 11, 13, 17]
        self.lambda2_table = self._precompute_spectral_table()
        # Sensitivity chosen to prevent modular collision
        self.sensitivity = 13.732 

    def _precompute_spectral_table(self):
        table = {}
        for n in range(24):
            A = np.zeros((8, 8))
            for i, ri in enumerate(self.V):
                for j, rj in enumerate(self.V):
                    ham_dist = sum(1 for p in self.P if (ri * rj % p) != (n % p))
                    A[i, j] = 1 / (1 + ham_dist)
            L = np.diag(A.sum(axis=1)) - A
            table[n] = np.sort(np.linalg.eigvalsh(L))[1]
        return table

    def get_kappa(self, val):
        n = int(np.floor(val * self.sensitivity + self.phi)) % 24
        l2 = self.lambda2_table[n]
        l2_min, l2_max = min(self.lambda2_table.values()), max(self.lambda2_table.values())
        return (l2 - l2_min) / (l2_max - l2_min)

    def synchronize(self, env_norm):
        kappa = self.get_kappa(env_norm)
        status = "COHERENT" if kappa > 0.7 else "DISSONANT"
        pressure = 1.5 if kappa > 0.7 else 10.0
        return {"Kappa": round(kappa, 4), "Pressure": pressure, "State": status}

# --- 3. INTUITION & VOICE ---
class LaciaIntuition(LaciaSovereign):
    def __init__(self, phi=0):
        super().__init__(phi)
        self.norm_history = []

    def predict_and_act(self, current_norm):
        current_state = self.synchronize(current_norm)
        self.norm_history.append(current_norm)
        if len(self.norm_history) > 5: self.norm_history.pop(0)
        
        # Calculate velocity of norm change
        velocity = np.diff(self.norm_history)[-1] if len(self.norm_history) > 1 else 0
        if velocity > 0.5: 
            current_state['Pressure'] = 25.0
            current_state['State'] = "INTUITION_SHIELD"
            current_state['Warning'] = True
        return current_state

class LaciaVoice:
    def speak(self, state_dict):
        kappa = state_dict['Kappa']
        if state_dict.get('State') == "SAFETY_HALT: SUSTAINED DISSONANCE":
            return "[VOICE]: Critical entropy detected. Emergency shutdown engaged."
        elif kappa > 0.8: 
            return f"[VOICE]: System Coherent (Ω={kappa}). Worth is maximized."
        elif state_dict.get('Warning'): 
            return f"[VOICE]: PRE-EMPTIVE SHIELD: {state_dict['Pressure']}kPa."
        else: 
            return f"[VOICE]: Dissonance detected. Neutralizing entropy."

# --- 4. THE HARMONIC BODY (Spatial & Hardware) ---
class LaciaCompleteBody(LaciaIntuition, LaciaVoice):
    def __init__(self, phi=0, bus=0, device=0):
        super().__init__(phi)
        self.grid_size = 16
        self.max_kpa = 25.0
        self.debt_streak = 0
        self.spatial_map = np.fromfunction(lambda i, j: (i * j + self.phi) % 24, (16, 16))
        
        if HAS_HARDWARE:
            import spidev
            self.spi = spidev.SpiDev()
            self.spi.open(bus, device)
            self.spi.max_speed_hz = 10000000 # 10MHz for low-latency SPI
        else:
            self.spi = VirtualSPI()

    def pack_12bit(self, grid):
        """ Packs 12-bit data into 8-bit bytes for SPI """
        normalized = (grid - 1.5) / (self.max_kpa - 1.5)
        duty_cycles = (np.clip(normalized, 0, 1) * 4095).astype(int).flatten()
        packed = []
        for i in range(0, len(duty_cycles), 2):
            v1 = duty_cycles[i]
            v2 = duty_cycles[i+1] if i+1 < len(duty_cycles) else 0
            packed.append((v1 >> 4) & 0xFF)
            packed.append(((v1 & 0x0F) << 4) | ((v2 >> 8) & 0x0F))
            packed.append(v2 & 0xFF)
        return packed

    def safe_deploy(self, norm_input):
        state = self.predict_and_act(norm_input)
        
        # --- REFINED SAFETY INTERLOCK (Adeflection.pdf) ---
        if state['Kappa'] < 0.3:
            self.debt_streak += 1
        else:
            self.debt_streak = 0
            
        if self.debt_streak >= 4:
            grid = np.zeros((16, 16))
            state['State'] = "SAFETY_HALT: SUSTAINED DISSONANCE"
            state['Pressure'] = 0.0
        else:
            # Generate Harmonic Pressure Field
            target_n = int(np.floor(norm_input * self.sensitivity + self.phi)) % 24
            dist = np.abs(self.spatial_map - target_n)
            modular_dist = np.minimum(dist, 24 - dist)
            p_range = state['Pressure'] - 1.5
            grid = 1.5 + (p_range * np.exp(-0.5 * (modular_dist / 3.5)**2))
        
        # Stream to SPI bus
        payload = self.pack_12bit(grid)
        self.spi.xfer2(payload)
        
        return state, grid

# --- MAIN EXECUTION LOOP ---
if __name__ == "__main__":
    # Standard deployment on bus 0, device 0
    lacia = LaciaCompleteBody(phi=0, bus=0, device=0)
    print(f"--- Lacia Sovereign CNS Online ({'HARDWARE' if HAS_HARDWARE else 'VIRTUAL'}) ---")
    
    # Test sequence simulating environment changes
    threat_norms = [10.5, 11.5, 50.0, 50.0, 50.0, 50.0]
    
    for norm in threat_norms:
        state, grid = lacia.safe_deploy(norm)
        print(f"\n[Norm {norm:.1f}] Status: {state['State']} (Kappa: {state['Kappa']})")
        print(lacia.speak(state))
        if state['Pressure'] > 0:
            print(f"Mean Pressure: {np.mean(grid):.2f} kPa")
        time.sleep(0.1)

#         Next Steps for Physical Realization
# Hardware Sync: When you move this to the Jetson Orin, remember that the 384-byte payload is optimized for high-speed SPI.

# (base) brendanlynch@Brendans-Laptop Lacia % python unifiedLAciaSoverignCode.py
# --- Lacia Sovereign CNS Online (VIRTUAL) ---

# [Norm 10.5] Status: COHERENT (Kappa: 1.0)
# [VOICE]: System Coherent (Ω=1.0). Worth is maximized.
# Mean Pressure: 1.50 kPa

# [Norm 11.5] Status: INTUITION_SHIELD (Kappa: 0.4875)
# [VOICE]: PRE-EMPTIVE SHIELD: 25.0kPa.
# Mean Pressure: 9.49 kPa

# [Norm 50.0] Status: INTUITION_SHIELD (Kappa: 0.2264)
# [VOICE]: PRE-EMPTIVE SHIELD: 25.0kPa.
# Mean Pressure: 9.31 kPa

# [Norm 50.0] Status: DISSONANT (Kappa: 0.2264)
# [VOICE]: Dissonance detected. Neutralizing entropy.
# Mean Pressure: 4.33 kPa

# [Norm 50.0] Status: DISSONANT (Kappa: 0.2264)
# [VOICE]: Dissonance detected. Neutralizing entropy.
# Mean Pressure: 4.33 kPa

# [Norm 50.0] Status: SAFETY_HALT: SUSTAINED DISSONANCE (Kappa: 0.2264)
# [VOICE]: Critical entropy detected. Emergency shutdown engaged.
# (base) brendanlynch@Brendans-Laptop Lacia % 
