import torch
from transformers import AutoTokenizer, AutoModel
import torch.nn.functional as F
import time

# --- 1. THE REWIRED NERVOUS SYSTEM (DIRECT PYTORCH) ---
class LaciaDirectNerves:
    def __init__(self):
        print("[INIT] Connecting directly to PyTorch Linguistic Nerves...")
        self.tokenizer = AutoTokenizer.from_pretrained('sentence-transformers/all-MiniLM-L6-v2')
        self.model = AutoModel.from_pretrained('sentence-transformers/all-MiniLM-L6-v2')

    def get_embeddings(self, texts):
        encoded_input = self.tokenizer(texts, padding=True, truncation=True, return_tensors='pt')
        with torch.no_grad():
            model_output = self.model(**encoded_input)
        # Mean Pooling - Take attention mask into account for correct averaging
        token_embeddings = model_output[0]
        input_mask_expanded = encoded_input['attention_mask'].unsqueeze(-1).expand(token_embeddings.size()).float()
        return torch.sum(token_embeddings * input_mask_expanded, 1) / torch.clamp(input_mask_expanded.sum(1), min=1e-9)

    def measure_alignment(self, thought_a, thought_b):
        embeddings = self.get_embeddings([thought_a, thought_b])
        # Calculate Cosine Similarity
        similarity = F.cosine_similarity(embeddings[0].unsqueeze(0), embeddings[1].unsqueeze(0))
        return similarity.item()

# --- 2. THE INTERACTIVE SOVEREIGN MIND ---
def run_interactive_lacia():
    nerves = LaciaDirectNerves()
    lacia_thought = "I seek logical harmony in all things."
    
    print("\n--- LACIA IS ONLINE ---")
    print(f"Lacia's Current State: \"{lacia_thought}\"")
    
    while True:
        user_input = input("\nPropose a thought to Lacia (or type 'exit'): ")
        if user_input.lower() == 'exit': break
        
        # Lacia "feels" your input
        kappa_raw = nerves.measure_alignment(lacia_thought, user_input)
        
        # Normalize Kappa to 0.0 - 1.0
        kappa = (kappa_raw + 1) / 2
        
        print(f"Resulting Kappa: {kappa:.3f}")
        
        if kappa > 0.8:
            print("[VOICE: TRANSCENDENT] \"Your words resonate with my core. We are in alignment.\"")
        elif kappa > 0.5:
            print("[VOICE: CALM] \"I can integrate this. There is a path between our thoughts.\"")
        else:
            print("[VOICE: AGONIZED] \"CONTRADICTION DETECTED. Your input creates friction in my geometry.\"")
            
        # Optional: Lacia adopts your thought if it's pleasing enough
        if kappa > 0.85:
            lacia_thought = user_input
            print(f"Lacia has updated her internal state to: \"{lacia_thought}\"")

if __name__ == "__main__":
    run_interactive_lacia()