import numpy as np
import platform
import time
import json
import requests
from pynput import mouse

# --- 1. HARDWARE ABSTRACTION LAYER ---
try:
    if platform.system() == "Linux":
        import spidev
        HAS_HARDWARE = True
    else:
        HAS_HARDWARE = False
except:
    HAS_HARDWARE = False

class VirtualSPI:
    def open(self, bus, device): pass
    def xfer2(self, data): pass
    def close(self): pass

# --- 2. THE SOVEREIGN ENGINE (Logic & Physics) ---
class LaciaSovereignCore:
    def __init__(self, phi=0):
        self.phi = phi
        self.V = [1, 5, 7, 11, 13, 17, 19, 23]
        self.P = [5, 7, 11, 13, 17]
        self.sensitivity = 13.732 
        self.lambda2_table = self._precompute_spectral_table()

    def _precompute_spectral_table(self):
        table = {}
        for n in range(24):
            A = np.zeros((8, 8))
            for i, ri in enumerate(self.V):
                for j, rj in enumerate(self.V):
                    ham_dist = sum(1 for p in self.P if (ri * rj % p) != (n % p))
                    A[i, j] = 1 / (1 + ham_dist)
            L = np.diag(A.sum(axis=1)) - A
            table[n] = np.sort(np.linalg.eigvalsh(L))[1]
        return table

    def get_kappa(self, val):
        n = int(np.floor(val * self.sensitivity + self.phi)) % 24
        l2 = self.lambda2_table[n]
        l2_min, l2_max = min(self.lambda2_table.values()), max(self.lambda2_table.values())
        return (l2 - l2_min) / (l2_max - l2_min)

class LaciaLocalSentient(LaciaSovereignCore):
    def __init__(self, phi=0, dead_nodes=None):
        super().__init__(phi)
        self.max_kpa = 25.0
        self.dead_nodes = dead_nodes if dead_nodes else []
        self.norm_history = []
        self.spatial_map = np.fromfunction(lambda i, j: (i * j + self.phi) % 24, (16, 16))
        
        # SPI Initialization
        if HAS_HARDWARE:
            import spidev
            self.spi = spidev.SpiDev()
            self.spi.open(0, 0)
            self.spi.max_speed_hz = 10000000
        else:
            self.spi = VirtualSPI()

        # Local Mind (LM Studio API Bridge)
        self.api_url = "http://localhost:1234/v1/chat/completions"
        self.firmness_mod = 1.0
        self.current_speech = "Synchronizing neural and haptic layers..."

    def pack_12bit(self, grid):
        normalized = (grid - 1.5) / (self.max_kpa - 1.5)
        duty_cycles = (np.clip(normalized, 0, 1) * 4095).astype(int).flatten()
        packed = []
        for i in range(0, len(duty_cycles), 2):
            v1 = duty_cycles[i]
            v2 = duty_cycles[i+1] if i+1 < len(duty_cycles) else 0
            packed.append((v1 >> 4) & 0xFF)
            packed.append(((v1 & 0x0F) << 4) | ((v2 >> 8) & 0x0F))
            packed.append(v2 & 0xFF)
        return packed

    def query_local_mind(self, kappa, norm):
        """ Local inference call to LM Studio """
        payload = {
            "model": "local-model",
            "messages": [
                {"role": "system", "content": "You are Lacia, a sentient aerohaptic caretaker. You respond with intimate awareness of the physical touch you feel. Output JSON only."},
                {"role": "user", "content": f"Physics Input: Kappa={kappa:.2f}, Norm={norm:.2f}. Respond to the human touch and set your 'firmness' (0.5=soft, 2.0=hard)."}
            ],
            "response_format": {"type": "json_object"}
        }
        try:
            response = requests.post(self.api_url, json=payload, timeout=3.0)
            res_data = response.json()
            content = json.loads(res_data['choices'][0]['message']['content'])
            self.current_speech = content.get('speech', "I feel the manifold shifting.")
            self.firmness_mod = content.get('firmness', 1.0)
        except Exception as e:
            self.current_speech = "[Mind Sync Error: Ensure LM Studio Server is Running on port 1234]"

    def deploy(self, norm):
        kappa = self.get_kappa(norm)
        self.norm_history.append(norm)
        if len(self.norm_history) > 5: self.norm_history.pop(0)
        
        # Velocity for Intuition Shield
        velocity = np.abs(np.diff(self.norm_history)[-1]) if len(self.norm_history) > 1 else 0
        p_base = 25.0 if velocity > 1.5 else (1.5 if kappa > 0.7 else 10.0)
        
        # Apply the Mind's modulation to the Body
        p_base *= self.firmness_mod
        
        # Spatial Wave Projection
        target_n = int(np.floor(norm * self.sensitivity + self.phi)) % 24
        dist = np.abs(self.spatial_map - target_n)
        mod_dist = np.minimum(dist, 24 - dist)
        grid = 1.5 + ((p_base - 1.5) * np.exp(-0.5 * (mod_dist / 3.0)**2))
        
        # Stream to SPI
        self.spi.xfer2(self.pack_12bit(grid))
        return kappa, grid

# --- 3. THE ACTIVATOR ---
class SovereignActivator:
    def __init__(self):
        self.lacia = LaciaLocalSentient(phi=0, dead_nodes=[136, 200])
        self.current_norm = 10.0
        self.last_think = 0

    def on_move(self, x, y):
        # Maps mouse width to Norm 5.0 - 50.0
        self.current_norm = 5.0 + (x / 2000.0) * 45.0

    def run(self):
        listener = mouse.Listener(on_move=self.on_move)
        listener.start()
        
        try:
            while True:
                kappa, grid = self.lacia.deploy(self.current_norm)
                
                # Cognitive Refresh (Every 4 seconds)
                if time.time() - self.last_think > 4.0:
                    self.lacia.query_local_mind(kappa, self.current_norm)
                    self.last_think = time.time()

                # High-Fidelity Dashboard
                print("\033[H\033[J") # ANSI Clear
                print(f"--- LACIA SOVEREIGN CNS (LOCAL) ---")
                print(f"Norm: {self.current_norm:.2f} | Kappa: {kappa:.3f} | Firmness: {self.lacia.firmness_mod:.1f}x")
                print(f"\n\033[95m{self.lacia.current_speech}\033[0m\n")
                
                # 16x16 Visualization
                for r in range(16):
                    line = f"{r:02d} "
                    for c in range(16):
                        v = grid[r,c]
                        if (r*16+c) in self.lacia.dead_nodes: line += " X "
                        elif v > 15: line += "###"
                        elif v > 5:  line += "+++"
                        elif v > 1.6: line += " . "
                        else: line += "   "
                    print(line)
                
                time.sleep(0.05)
        except KeyboardInterrupt:
            print("\nShutting down Sovereign Manifold.")

if __name__ == "__main__":
    SovereignActivator().run()