#ifndef SYSCALL_H
#define SYSCALL_H

/**
 * basic fixed-size integer types (used for syscall prototypes).
 */
#include "arch_int_types.h"


/** 
 * For some reason syscall numbers vary across platforms.
 * This header is supposed to be generic, but we need this platform dependent mapping.
 * So we include an additional platform specific header ...
 * This header defines syscall numbers for syscalls that are supported by the platform.
 * If a syscall is not supported, the syscall number is just not defined.
 * This header will then make sure only defined syscalls are used.
 */
#include "arch_syscall.h"


/**
 * Generic syscall wrappers ([SYSCALL_n] for a syscall with n arguments)
 * Implementation is defined in architecture specific code [arch/<architecture>/syscall.S].
 * As programmer, it is convenient to use the syscall number as first argument.
 * However, on most platforms, argument registers match better if we pass the first syscall argument as first argument.
 * For instance, on loongarch64, this allows the assembly code to keep all arguments in their respective argument register 
 * and only move the syscall number to a different register before invoking a syscall.
 */

extern uint64_t __syscall_1(void* a0, int nr);
#define SYSCALL_1(nr, a0) __syscall_1((void*)(a0), nr)

extern uint64_t __syscall_2(void* a0, void* a1, int nr);
#define SYSCALL_2(nr, a0, a1) __syscall_2((void*)(a0), (void*)(a1), nr)

extern uint64_t __syscall_3(void* a0, void* a1, void* a2, int nr);
#define SYSCALL_3(nr, a0, a1, a2) __syscall_3((void*)(a0), (void*)(a1), (void*)(a2), nr)

extern uint64_t __syscall_4(void* a0, void* a1, void* a2, void* a3, int nr);
#define SYSCALL_4(nr, a0, a1, a2, a3) __syscall_4((void*)(a0), (void*)(a1), (void*)(a2), (void*)(a3), nr)

extern uint64_t __syscall_5(void* a0, void* a1, void* a2, void* a3, void* a4, int nr);
#define SYSCALL_5(nr, a0, a1, a2, a3, a4) __syscall_5((void*)(a0), (void*)(a1), (void*)(a2), (void*)(a3), (void*)(a4), nr)

extern uint64_t __syscall_6(void* a0, void* a1, void* a2, void* a3, void* a4, void* a5, int nr);
#define SYSCALL_6(nr, a0, a1, a2, a3, a4, a5) __syscall_6((void*)(a0), (void*)(a1), (void*)(a2), (void*)(a3), (void*)(a4), (void*)(a5), nr)


/**
 * Commonly used syscalls.
 * Only the prototypes are defined here, for arguments and struct definitions you may need to include additional headers depending on the syscall.
 * If a syscall is not defined by a platform, SYS_<syscall_name> is not defined.
 * Additionally, using an undefined syscall leads to a failing _Static_assert.
 */

// convenience macro to raise error if undefined syscall is used.
#define UNDEF_SYSCALL(x) _Static_assert(0, #x " is not defined by platform")

// convenience macro to build a function wrapping a syscall.
#define MAKE_SYSCALL_1(name, num, tr, t0, a0) static inline __attribute__((always_inline)) tr name(t0 a0) { return (tr) SYSCALL_1(num, (uint64_t) (a0)); }
#define MAKE_SYSCALL_2(name, num, tr, t0, a0, t1, a1) static inline __attribute__((always_inline)) tr name(t0 a0, t1 a1) { return (tr) SYSCALL_2(num, (uint64_t) (a0), (uint64_t) (a1)); }
#define MAKE_SYSCALL_3(name, num, tr, t0, a0, t1, a1, t2, a2) static inline __attribute__((always_inline)) tr name(t0 a0, t1 a1, t2 a2) { return (tr) SYSCALL_3(num, (uint64_t) (a0), (uint64_t) (a1), (uint64_t) (a2)); }
#define MAKE_SYSCALL_4(name, num, tr, t0, a0, t1, a1, t2, a2, t3, a3) static inline __attribute__((always_inline)) tr name(t0 a0, t1 a1, t2 a2, t3 a3) { return (tr) SYSCALL_4(num, (uint64_t) (a0), (uint64_t) (a1), (uint64_t) (a2), (uint64_t) (a3)); }
#define MAKE_SYSCALL_5(name, num, tr, t0, a0, t1, a1, t2, a2, t3, a3, t4, a4) static inline __attribute__((always_inline)) tr name(t0 a0, t1 a1, t2 a2, t3 a3, t4 a4) { return (tr) SYSCALL_5(num, (uint64_t) (a0), (uint64_t) (a1), (uint64_t) (a2), (uint64_t) (a3), (uint64_t) (a4)); }
#define MAKE_SYSCALL_6(name, num, tr, t0, a0, t1, a1, t2, a2, t3, a3, t4, a4, t5, a5) static inline __attribute__((always_inline)) tr name(t0 a0, t1 a1, t2 a2, t3 a3, t4 a4, t5 a5) { return (tr) SYSCALL_6(num, (uint64_t) (a0), (uint64_t) (a1), (uint64_t) (a2), (uint64_t) (a3), (uint64_t) (a4), (uint64_t) (a5)); }

#ifdef SYS_READ
    MAKE_SYSCALL_3(sys_read, SYS_READ, uint64_t, int, fd, void*, buf, uint64_t, count)
#else
    #define sys_read(fd, buf, count) UNDEF_SYSCALL(SYS_READ)
#endif /* SYS_READ */

#ifdef SYS_WRITE
    MAKE_SYSCALL_3(sys_write, SYS_WRITE, uint64_t, int, fd, void*, buf, uint64_t, count)
#else
    #define sys_write(fd, buf, count) UNDEF_SYSCALL(SYS_WRITE)
#endif /* SYS_WRITE */

#ifdef SYS_EXIT
    MAKE_SYSCALL_1(sys_exit, SYS_EXIT, void, int, status)
#else
    #define sys_exit(status) UNDEF_SYSCALL(SYS_EXIT)
#endif /* SYS_EXIT */

#ifdef SYS_OPENAT
    MAKE_SYSCALL_3(sys_openat, SYS_OPENAT, int, int, dir, char*, path, int, flags)
#else
    #define sys_openat(dir, path, flags) UNDEF_SYSCALL(SYS_OPENAT)
#endif /* SYS_OPENAT */

#ifdef SYS_CLOSE
    MAKE_SYSCALL_1(sys_close, SYS_CLOSE, int, int, fd)
#else
    #define sys_close(path, flags) UNDEF_SYSCALL(SYS_CLOSE)
#endif /* SYS_CLOSE */

#ifdef SYS_SETITIMER
    struct itimerval;
    MAKE_SYSCALL_3(sys_setitimer, SYS_SETITIMER, int, int, which, struct itimerval*, new_val, struct itimerval*, old_val)
#else
    #define sys_setitimer(which, new_val, old_val) UNDEF_SYSCALL(SYS_SETITIMER)
#endif /* SYS_SETITIMER */

#ifdef SYS_SIGALTSTACK
    struct sigstack;
    MAKE_SYSCALL_2(sys_sigaltstack, SYS_SIGALTSTACK, int, struct sigstack*, ss, struct sigstack*, old_ss)
#else
    #define sys_sigaltstack(ss, old_ss) UNDEF_SYSCALL(SYS_SIGALTSTACK)
#endif /* SYS_SIGALTSTACK */

#ifdef SYS_RT_SIGACTION
    struct sigaction;
    MAKE_SYSCALL_4(sys_rt_sigaction, SYS_RT_SIGACTION, int, int, signum, struct sigaction*, act, struct sigaction*, oldact, uint64_t, sigsetsize)
#else
    #define sys_rt_sigaction(signum, act, oldact, sigsetsize) UNDEF_SYSCALL(SYS_RT_SIGACTION)
#endif /* SYS_RT_SIGACTION */

#ifdef SYS_RT_SIGPROCMASK
    struct sigset;
    MAKE_SYSCALL_4(sys_rt_sigprocmask, SYS_RT_SIGPROCMASK, int, int, how, struct sigset*, set, struct sigset*, oldset, uint64_t, sigsetsize)
#else
    #define sys_rt_sigprocmask(how, set, oldset, sigsetsize) UNDEF_SYSCALL(SYS_RT_SIGPROCMASK)
#endif /* SYS_RT_SIGPROCMASK */

#ifdef SYS_MMAP
    MAKE_SYSCALL_6(sys_mmap, SYS_MMAP, void*, void*, addr, uint64_t, length, int, prot, int, flags, int, fd, uint64_t, offset)
#else
    #define sys_mmap(addr, length, prot, flags, fd, offset) UNDEF_SYSCALL(SYS_MMAP)
#endif /* SYS_MMAP */


#ifdef SYS_MUNMAP
    MAKE_SYSCALL_2(sys_munmap, SYS_MUNMAP, void, void*, addr, uint64_t, length)
#else
    #define sys_munmap(addr, length) UNDEF_SYSCALL(SYS_MUNMAP)
#endif /* SYS_MUNMAP */

#ifdef SYS_MPROTECT
    MAKE_SYSCALL_3(sys_mprotect, SYS_MPROTECT, int, void*, addr, uint64_t, length, int, prot)
#else
    #define sys_mprotect(addr, length, prot) UNDEF_SYSCALL(SYS_MPROTECT)
#endif /* SYS_MPROTECT */

#ifdef SYS_CLOCK_GETTIME
    struct timespec;
    MAKE_SYSCALL_2(sys_clock_gettime, SYS_CLOCK_GETTIME, int, int32_t, clockid, struct timespec*, tp);
#else
    #define sys_clock_gettime(clockid, tp) UNDEF_SYSCALL(SYS_CLOCK_GETTIME)
#endif /* SYS_CLOCK_GETTIME */


#ifdef SYS_SOCKET
    MAKE_SYSCALL_3(sys_socket, SYS_SOCKET, uint64_t, int, domain, int, type, int, protocol);
#else
    #define sys_socket(domain, type, protocol) UNDEF_SYSCALL(SYS_SOCKET)
#endif /* SYS_SOCKET */
    
#ifdef SYS_CONNECT
    MAKE_SYSCALL_3(sys_connect, SYS_CONNECT, uint64_t, int, socket_fd, void*, addr, uint64_t, addrlen);
#else
    #define sys_connect(socket_fd, addr, addrlen) UNDEF_SYSCALL(SYS_CONNECT)
#endif /* SYS_CONNECT */

#endif /* SYSCALL_H */
