#ifndef STDLIB_H
#define STDLIB_H


// fixed-size integer types (uint64_t, uint8_t, ...).
#include "arch_int_types.h"

// system call wrappers.
#include "syscall.h"


/**
 * varadict types, mostly used for string formatting functions.
 */
 
#define va_arg(v,l) __builtin_va_arg(v,l)
#define va_start(v,l) __builtin_va_start(v,l)
typedef __builtin_va_list va_list;

/*
 * definitions / global variables that are useful to make code more readable (instead of putting numbers everywhere).
 */
 
#define AT_FDCWD -100
#define MAP_SHARED 0x01
#define MAP_PRIVATE 0x02
#define MAP_FIXED 0x10
#define MAP_FIXED_NOREPLACE 0x100000
#define PROT_READ 0x1
#define PROT_WRITE 0x2
#define PROT_EXEC 0x4

#if defined(__mips64)
#define MAP_ANONYMOUS 0x800
#define MAP_POPULATE 0x10000
#else
#define MAP_ANONYMOUS 0x20
#define MAP_POPULATE 0x8000
#endif

#define stdin  0
#define stdout 1
#define stderr 2
#define NULL ((void*)0)

// open
#define O_RDONLY 0
#define O_WRONLY 1
#define O_RDWR   2
#define O_TRUNC  0x200
#define O_CREAT  0x40


/**
 * useful functions directly wrapping syscalls (and definitions required, but no types).
 * Other functions may also directly map to syscalls, but require additional types (structs) to be defined.
 * Such functions are defined in more specific header files (alarm.h, signal.h, ...).
 */

#define mmap(...) sys_mmap(__VA_ARGS__)
#define munmap(...) sys_munmap(__VA_ARGS__)
#define mprotect(...) sys_mprotect(__VA_ARGS__) 
#define read(...) sys_read(__VA_ARGS__)
#define write(...) sys_write(__VA_ARGS__)
#define exit(...) sys_exit(__VA_ARGS__)
#define openat(...) sys_openat(__VA_ARGS__)
#define open(...) openat(AT_FDCWD, __VA_ARGS__)
#define close(...) sys_close(__VA_ARGS__)

#define socket(...) sys_socket(__VA_ARGS__)
#define connect(...) sys_connect(__VA_ARGS__)

/**
 * generic useful library functions.
 * Some of them slightly differ from the "usual" implementation.
 * However, these functions also have slightly different names such that they aren't confused (e.g., memset8 vs. memset).
 */

uint64_t strlen(char* s);

// write until n bytes are actually written
uint64_t nwrite(int fd, uint8_t* buf, uint64_t n);

// read until n bytes are actually read
uint64_t nread(int fd, void* buf, uint64_t n);

// write string, so length = strlen(s)
uint64_t swrite(int fd, char* s);

// slightly different then normal memset: take uint8_t instead of int.
// This allows for different memset implementations filling with bigger types.
void* memset8(void* buffer, uint8_t value, uint64_t length);

void* memcpy(void* dest, void* src, uint64_t n);

// my spin at string formatting.
// Avoids buffer overflows by specifying the maximum buffer size (buf_size).
// Additionally allows to not write the first n characters that would be written (skip).
// This is done so format() can be called in a loop to format remaining characters after a buffer is filled and e.g., printed.
// And, finally, only support for very few formaters and no padding (yet):
// %u: unsigned 64-bit decimal (big endian)
// %d: signed 64-bit decimal (big endian)
// %x: unsigned 64-bit hexadecimal (big endian)
// %b: unsigned 64-bit binary (big endian)
// %s: a string
// Returns numbers of characters written to buf.
uint64_t format(char* buf, uint64_t buf_size, uint64_t skip, char* format, ...);

// write format string to file descriptor.
void writef(int fd, char* format, ...);

// queue write. will be written when flush() is called or buffer is full
// there is only one buffer so when a write to a different file descriptor is queued, the old queue is flushed.
// TODO: error?
void queue(int fd, uint8_t* buf, uint64_t n);

// flush buffer (returns total amount written including implicit flushes)
uint64_t flush(void);

// not the actual thing as it is restricted to formatters supported by format.
#define printf(format, ...) writef(stdout, format __VA_OPT__(,) __VA_ARGS__)

// string to unsigned 64-bit integer.
// the string is always assumed to be big-endian (highest significant digit first).
// if prefixed with 0x, string is interpreted as hexadecimal (0-9a-fA-F).
// if prefixed with 0b, string is interpreted as binary (0-1).
// if prefixed with 0o, the string is interpreted as octal (0-7).
// Otherwise, the string is interpreted as decimal (0-9).
// Once a character is encountered, that does not match the charset of the number system, parsing stops.
// This character does not have to be a nullbyte.
// A pointer to the first character after the parsed string is written to endptr, if endptr is not NULL.
// This method does not check for overflows.
// If the number does not fit into an uint64_t, the result is unspecified.
uint64_t strtou64(char* s, char** endptr);

/**
 * startup code
 */

void main(int argc, char** argv);
 
void start(uint64_t* sp);

#endif /* STDLIB_H */
