/** 
 * check the stdlib.h header for information on what those functions do :)
 */
#include "stdlib.h"

int write_fd = -1;
uint64_t write_offset = 0;
uint8_t write_buf[4096 * 4];
uint64_t total_written = 0;

uint64_t _flush(int implicit) {
    uint64_t ret = 0;
    if(write_offset) {
        total_written += nwrite(write_fd, write_buf, write_offset);
        write_offset = 0;
    }
    ret = total_written;
    if(!implicit) {
        total_written = 0;
    }
    return ret;
}

void queue(int fd, uint8_t* buf, uint64_t n) {
    uint64_t queued = 0;
    uint64_t to_queue = 0;
    if(write_offset != 0 && write_fd != fd) {
        _flush(0);
    }
    write_fd = fd;
    
    while(queued < n) {
        to_queue = sizeof(write_buf) - write_offset;
        if(to_queue > (n - queued)) {
            to_queue = n - queued;
        }
        memcpy(&write_buf[write_offset], &buf[queued], to_queue);
        queued += to_queue;
        write_offset += to_queue;
        
        if(write_offset == sizeof(write_buf)) {
            _flush(1);
        }
    }
}



// flush buffer
uint64_t flush(void) {
    return _flush(0);
}

uint64_t strlen(char* s) {
    uint64_t length;
    for(length = 0; s[length]; length ++);
    return length;
}

uint64_t nwrite(int fd, uint8_t* buf, uint64_t len) {
    uint64_t written = 0;
    // fix for write to address 0 as this for some reason fails ...
    if(len && buf == (uint8_t*)0) {
        uint8_t fix = *buf;
        written += write(fd, &fix, 1);
    }
    // TODO: handle errors ("negative" return values)
    // writef(stderr, "writef(%u, %u, %u)\n", fd, buf, len);
    while(written < len) {
        written += write(fd, buf + written, len - written);
    }
    return written;
}

uint64_t nread(int fd, void* _buf, uint64_t len) {
    uint8_t* buf = _buf;
    uint64_t amount_read = 0;

    // TODO: handle errors ("negative" return values)
    // writef(stderr, "writef(%u, %u, %u)\n", fd, buf, len);
    while(amount_read < len) {
        amount_read += read(fd, buf + amount_read, len - amount_read);
    }
    return amount_read;
}

// TODO: this could be a macro
uint64_t swrite(int fd, char* s){
    return nwrite(fd, (uint8_t*)s, strlen(s));
}

void* memset8(void* buffer, uint8_t value, uint64_t length) {
    uint8_t* _buffer = buffer;
    for(uint64_t i = 0; i < length; i++) {
        _buffer[i] = value;
    }
    return buffer;
}

// needs to be there otherwise gcc won't compile correctly
void* memset(void* buffer, int value, uint64_t length) {
    return memset8(buffer, (uint8_t) value, length);
}

void* memcpy(void* dest, void* src, uint64_t n) {
    uint8_t* _dest = (uint8_t*) dest;
    uint8_t* _src  = (uint8_t*) src;
    for(uint64_t i = 0; i < n; i++) {
        _dest[i] = _src[i];
    }
    return dest;
}


// helper function as passing varadic arguments is a pain
static uint64_t vformat(char* buf, uint64_t buf_size, uint64_t skip, char* format, va_list v) {
    // Simple number printing code would print numbers in little endian.
    // Thus, we use a temporary buffer to reverse numbers after using the simple code.
    // The buffer must be big enough to hold a 64-bit binary number represented in characters (=64 bytes).
    // There is no need to also hold a terminating null-byte, as the number writing code remembers the amount of characters.
    char num_buf[64];
    
    // amount of characters already written.
    // This is crucial to make sure we don't overflow buf.
    uint64_t written = 0;
    
    // 1 if the last character was a %, starting a formatter.
    int is_formatter = 0;
    
    // remember original format string pointer so we can print it on error
    char* orig_format = format;
    
    // helper to write or skip a character
    #define W(c) if(skip > 0) {skip --;} else {buf[written ++] = c;} 
    
    // iterate until the format string is fully consumed or the buffer is full
    while(*format && written < buf_size) {
        if(is_formatter) {
            switch(*format) {
                
                case 'u': {
                    // unsigned 64-bit integer as decimal (big endian)
                    uint64_t f_u = va_arg(v, uint64_t);
                    uint64_t f_u_len = 0;
                    do {
                        num_buf[f_u_len ++] = '0' + (f_u % 10);
                        f_u /= 10;
                    } while(f_u);
                    
                    // number is now in num_buf, but reversed, so reverse it back while copying.
                    while(f_u_len && written < buf_size) {
                        f_u_len --;
                        W(num_buf[f_u_len]);
                    }
                    break;
                }
                
                case 'd': {
                    // signed 64-bit integer as decimal (big endian)
                    int64_t f_d = va_arg(v, int64_t);
                    
                    // we can assume at least one character of space still in the buffer here (otherwise the outer loop would have already terminated).
                    if(f_d < 0) {
                        W('-');
                        // WARNING: this breaks for the minimum number that can be written. 
                        // TODO: We should do an extra check for int64_t_min. 
                        f_d = -f_d;
                    }
                    
                    uint64_t f_d_len = 0;
                    do {
                        num_buf[f_d_len ++] = '0' + (f_d % 10);
                        f_d /= 10;
                    } while(f_d);
                    
                    // number is now in num_buf, but reversed, so reverse it back while copying.
                    while(f_d_len && written < buf_size) {
                        f_d_len --;
                        W(num_buf[f_d_len]);
                    }
                    break;
                }
                
                case 'x': {
                    // unsigned 64-bit integer as hexadecimal (big endian)
                    uint64_t f_x = va_arg(v, uint64_t);
                    uint64_t f_x_len = 0;
                    do {
                               
                        num_buf[f_x_len ++] = ((f_x % 16) > 9 ? ('a' - 10) : '0') + (f_x % 16);
                        f_x /= 16;
                    } while(f_x);
                    
                    // number is now in num_buf, but reversed, so reverse it back while copying.
                    while(f_x_len && written < buf_size) {
                        f_x_len --;
                        W(num_buf[f_x_len]);
                    }
                    break;
                }
                
                case 'b': {
                    // unsigned 64-bit integer as binary (big endian)
                    uint64_t f_b = va_arg(v, uint64_t);
                    uint64_t f_b_len = 0;
                    do {
                        num_buf[f_b_len ++] = '0' + (f_b % 2);
                        f_b /= 2;
                    } while(f_b);
                    
                    // number is now in num_buf, but reversed, so reverse it back while copying.
                    while(f_b_len && written < buf_size) {
                        f_b_len --;
                        W(num_buf[f_b_len]);
                    }
                    break;
                }
                
                case 's': {
                    // string
                    char * f_s = va_arg(v, char*);
                    while(*f_s && written < buf_size) {
                        W(*f_s);
                        f_s ++;
                    }
                    break;
                }
                
                default: {
                    // use of undefined formatter.
                    swrite(stderr, "Format string uses unknown formatter '");
                    write(stderr, format, 1);
                    swrite(stderr, "'\n");
                    swrite(stderr, "format string: ");
                    swrite(stderr, orig_format);
                    swrite(stderr, "\n");
                    exit(-1);
                }
            }
            
            // no longer inside a formatter
            is_formatter = 0;
        } else if(*format == '%') {
            // start of formatter
            is_formatter = 1;
        } else {
            // easy case: just write the current character.
            W(*format);
        }
        
        // consumed one character of format string.
        format ++;
    }
    
    
    if(is_formatter && !*format) {
        // stray % at end of format string, this is not allowed and could be a serious issue!
        swrite(stderr, "Stray % at end of format string!\n");
        exit(-1);
    }
    
    return written;
}


uint64_t format(char* buf, uint64_t buf_size, uint64_t skip, char* format, ...) {
    va_list v;
    va_start(v, format);
    
    return vformat(buf, buf_size, skip, format, v);
}

void writef(int fd, char* format, ...) {
    va_list v;
    va_start(v, format);
    
    // internal buffer used for printing
    char buf[128];
    
    // amount of characters to skip
    uint64_t skip = 0;
    
    // amount of characters written in current iteration
    uint64_t written = 0;
    
    // this whole thing only works because C gurantees that sizeof(char) = 1
    do {
        written = vformat(buf, sizeof(buf), skip, format, v);
        write(fd, buf, written);
        skip += written;
    } while(written == sizeof(buf));
}

uint64_t strtou64(char* s, char** endptr) {
    uint64_t base = 10;
    uint64_t result = 0;
    if(s[0] && s[1] && s[0] == '0') {
        if(s[1] == 'x'){
            base = 16;
            s += 2;
        }
        else if(s[1] == 'o') {
            base = 8;
            s += 2;
        }
        else if(s[1] == 'b') {
            base = 2;
            s += 2;
        }
    }
    
    char cur;
    int cur_add = 0;
    while(1) {
        cur = *s;
        
        switch(base) {
            case 2:
            case 8:
            case 10: {
                if(cur < '0' || cur >= '0' + base) {
                    cur_add = -1;
                } else {
                    cur_add = cur - '0';
                }
                break;
            }
            
            case 16: {
                if(cur >= '0' && cur <= '9') {
                    cur_add = cur - '0';
                } else if(cur >= 'a' && cur <= 'f') {
                    cur_add = 10 + cur - 'a';
                } else if(cur >= 'A' && cur <= 'F') {
                    cur_add = 10 + cur - 'A';
                } else {
                    cur_add = -1;
                }
                break;
            }
            
            default: {
                writef(stderr, "ERROR: strtou64: got base %u\n", base);
                exit(-1);
                break;
            }
            
        }
        
        if(cur_add == -1) {
            break;
        }
        
        result *= base;
        result += cur_add;
        
        s ++;
    }
    
    if(endptr) {
        *endptr = s;
    }
    return result;
}

void start(uint64_t* sp) {
    main(sp[0], (char**) &sp[1]);
    exit(0);
}
