#ifndef ARCH_LOADER_H
#define ARCH_LOADER_H

#include <stdlib.h>
#include <signal.h>

#define PAGE_SIZE (4096)

#include "difffuzz_regs.h"

struct __attribute__((packed)) register_state {
    // program counter is a special register on risv64
    uint64_t pc;
    struct regs regs;
};

// at this point, everything is still mapped as normal, but additional mappings are already present and initialized.
// this method should check all registers for architectural limitations and exit with an error if some limitation is violated.
// Some architectures (like loongarch64) require additional instructions located in the mapping containing the PC to correctly setup all state.
// Those architectures should also check for those requirements.
void arch_check_registers(struct register_state* registers_before);

// at this point, there is a single mapping from the loader that is mapped as read only (except for the signal handler stack).
// this method should restore the full register state stored in the argument, including the program counter.
// This method does obviously not return.
void arch_setup_registers(struct register_state* registers_before);

// save register state of crashing instruction.
// arguments match signal handler arguments. 
// Some registers may not be exposed to the signal handler.
// Luckily, they are usually also not polluted by the kernel. Thus, we can read them using inline assembly.
void arch_store_registers(int signo, struct siginfo* info, void* ucontext, struct register_state* state);

// get address associated with crash.
// For SIGSEGV (and similar), this should be the accessed address (in some cases it is not recoverable and will sadly be NULL), for SIGILL, this should be the address of the instruction.
uint64_t arch_get_crash_address(int signo, struct siginfo* info, void* ucontext);

// wrapper for main function:
// changes the stack pointer to provided value and then calls wrapped_main with provided argc and argv.
void wrap_main(int argc, char** argv, void* stack);

#endif /* ARCH_LOADER_H */
