#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create general controlled vocabularies, properties, etc.

"""

from pybis import Openbis
from schema_helpers import *


MAX_NUM_ELEMENTS = 10  # Maximum number of elements in a sample
MAX_NUM_DETECTORS = 6  # Maximum number of detectors SEM/(S)TEM


# openBIS codes for the objects (also used in other modules)

code_experimental_step = "EXPERIMENTAL_STEP"
code_point_meas = "POINT_MEASUREMENT"
code_compute_resource = "COMPUTE_RESOURCE"


def create_general_properties(oBis: Openbis):
    """Creates general Controlled Vocabularies, Property Types & Object Types.

    Protocols that are closely related and are a set of complex of activities
    are better defined in a separate file.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """

    # # #########################################################################
    # #  Controlled Vocabularies
    # # #########################################################################

    # Types of experiments / modalities
    terms_exp_type = [
        {
            "code": "AFM",
            "label": "Atomic Force Microscopy",
            "description": "Atomic Force Microscopy",
        },
        {
            "code": "APT",
            "label": "Atom Probe Tomography",
            "description": "Atom Probe Tomography",
        },
        {
            "code": "ARCMELT",
            "label": "Arc Melting",
            "description": "Arc Melting",
        },
        {
            "code": "BSCG",
            "label": "Bridgman-Stockbarger crystal growth",
            "description": "Bridgman-Stockbarger crystal growth",
        },
        {
            "code": "COLDROLL",
            "label": "Cold Rolling",
            "description": "Cold Rolling",
        },
        {
            "code": "COMP",
            "label": "Computation",
            "description": "Computation",
        },
        {
            "code": "CZOCHRALSKI",
            "label": "Czochralski crystal growth",
            "description": "Czochralski crystal growth",
        },
        {
            "code": "DSC",
            "label": "Differential Scanning Calorimetry",
            "description": "Differential Scanning Calorimetry",
        },
        {
            "code": "EBSD_ANALYSIS",
            "label": "EBSD Analysis",
            "description": "EBSD Post-processing",
        },
        {
            "code": "EPMA",
            "label": "Electron probe micro analysis",
            "description": "Electron probe micro analysis",
        },
        {
            "code": "ERDA",
            "label": "Elastic recoil detection analysis",
            "description": "Elastic recoil detection analysis",
        },
        {
            "code": "EXTRUSION",
            "label": "Extrusion",
            "description": "Extrusion",
        },
        {
            "code": "HOTROLL",
            "label": "Hot Rolling",
            "description": "Hot Rolling",
        },
        {
            "code": "HYD_CHARG_GAS",
            "label": "Gaseous Hydrogen Charging",
            "description": "Gaseous Hydrogen Charging",
        },
        {
            "code": "METALLO",
            "label": "Metallographic Preparation",
            "description": "Metallographic Preparation",
        },
        {
            "code": "OCP",
            "label": "Open Circuit Potential",
            "description": "Open Circuit Potential",
        },
        {
            "code": "OM",
            "label": "Optical Microscopy",
            "description": "Optical Microscopy",
        },
        {
            "code": "PDP",
            "label": "Potentiodynamic polarization",
            "description": "Potentiodynamic polarization",
        },
        {
            "code": "SIET",
            "label": "Scanning ion-selective electrode technique",
            "description": "Scanning ion-selective electrode technique",
        },
        {
            "code": "STA",
            "label": "Slip Trace Analysis",
            "description": "Slip Trace Analysis",
        },
        {
            "code": "SVET",
            "label": "Scanning vibrating electrode technique",
            "description": "Scanning vibrating electrode technique",
        },
        {
            "code": "TEM",
            "label": "Transmission Electron Microscopy",
            "description": "Transmission Electron Microscopy",
        },
        {
            "code": "XCT",
            "label": "X-ray Computed Tomography",
            "description": "X-ray Computed Tomography",
        },
        {
            "code": "XPS",
            "label": "X-ray photoelectron spectroscopy",
            "description": "X-ray photoelectron spectroscopy",
        },
        {
            "code": "XRD",
            "label": "X-ray Diffraction",
            "description": "X-ray Diffraction",
        },
        {
            "code": "XRD_ANALYSIS",
            "label": "XRD Analysis",
            "description": "XRD Analysis",
        },
    ]
    voc_exp_type = oBis.new_vocabulary(
        code="EXP_TYPE_VOCAB",
        description="Experiment Type Vocabulary",
        terms=terms_exp_type,
    )
    register_controlled_vocabulary(oBis, voc_exp_type, terms_exp_type)

    # Measured / Calculated Quantities
    terms_quantity_type = [
        {
            "code": "HARDNESS",
            "label": "Hardness",
            "description": "https://w3id.org/pmd/co/PMD_0000773,https://www.materials.fraunhofer.de/ontologies/BWMD_ontology/domain#BWMD_00634",
        },
        {
            "code": "VOLUME_FRACTION",
            "label": "Volume Fraction",
            "description": "https://w3id.org/pmd/co/VolumeFraction",
        },
        {
            "code": "POISSON_RATIO",
            "label": "Poisson Ratio",
            "description": "https://www.materials.fraunhofer.de/ontologies/BWMD_ontology/domain#BWMD_00409",
        },
        {
            "code": "YOUNG_MODULUS",
            "label": "Young's Modulus",
            "description": "https://www.materials.fraunhofer.de/ontologies/BWMD_ontology/domain#BWMD_00471,http://matonto.org/ontologies/matonto#YoungsModulus",
        },
        {
            "code": "SHEAR_MODULUS",
            "label": "Shear Modulus",
            "description": "https://www.materials.fraunhofer.de/ontologies/BWMD_ontology/domain#BWMD_00619,http://matonto.org/ontologies/matonto#ShearModulus",
        },
        {
            "code": "BULK_MODULUS",
            "label": "Bulk Modulus",
            "description": "http://matonto.org/ontologies/matonto#YoungsModulus",
        },
        {
            "code": "FINAL_TOTAL_ENERGY",
            "label": "Final Total Energy",
            "description": "http://purls.helmholtz-metadaten.de/asmo/TotalEnergy",
        },
        {
            "code": "FINAL_VOLUME",
            "label": "Final Volume",
            "description": "http://purls.helmholtz-metadaten.de/asmo/Volume",
        },
        {
            "code": "TOTAL_MAGNETIC_MOMENT",
            "label": "Total Magnetic Moment",
            "description": "http://purls.helmholtz-metadaten.de/asmo/TotalMagneticMoment",
        },
        {
            "code": "PRESSURE",
            "label": "Pressure",
            "description": "http://purls.helmholtz-metadaten.de/asmo/Pressure",
        },
        {
            "code": "TEMPERATURE",
            "label": "Temperature",
            "description": "http://purls.helmholtz-metadaten.de/asmo/Temperature",
        },
        {
            "code": "FORMATION_ENERGY",
            "label": "Formation Energy",
            "description": "http://purls.helmholtz-metadaten.de/asmo/FormationEnergy",
        },
        {
            "code": "GRAIN_BOUNDARY_ENERGY",
            "label": "Grain Boundary Energy",
            "description": "http://purls.helmholtz-metadaten.de/cdos/pldo/GrainBoundaryEnergy",
        },
    ]
    voc_quantity_type = oBis.new_vocabulary(
        code="QUANTITY_VOCAB",
        description="Type of (measured / calculated) physical quantity",
        terms=terms_quantity_type,
    )
    register_controlled_vocabulary(oBis, voc_quantity_type, terms_quantity_type)

    # Notebook type
    terms_nb_type = [
        {"code": "PYTHON", "label": "Python", "description": "Python notebook"},
        {"code": "MATLAB", "label": "Matlab", "description": "Matlab notebook"},
        {"code": "OCTAVE", "label": "Octave", "description": "Octave notebook"},
    ]
    voc_nb_type = oBis.new_vocabulary(
        code="NOTEBOOK_TYPE_VOCAB",
        description="Type of notebook",
        terms=terms_nb_type,
    )
    register_controlled_vocabulary(oBis, voc_nb_type, terms_nb_type)

    # General Simulation types
    terms_sim_type = [
        {
            "code": "DFT",
            "label": "DFT",
            "description": "DFT simulation: http://purls.helmholtz-metadaten.de/asmo/DensityFunctionalTheory",
        },
        {"code": "ATOM", "label": "Atomistic", "description": "Atomistic simulation"},
        {"code": "EBSD", "label": "EBSD", "description": "EBSD simulation"},
        {
            "code": "STEM",
            "label": "STEM",
            "description": "STEM multi-slice image simulation",
        },
    ]
    voc_sim_type = oBis.new_vocabulary(
        code="SIMULATION_TYPE",
        description="Type of simulation",
        terms=terms_sim_type,
    )
    register_controlled_vocabulary(oBis, voc_sim_type, terms_sim_type)

    # EBSD Simulation types
    terms_ebsd_sim = [
        {
            "code": "MC",
            "label": "Monte-Carlo",
            "description": "Monte-Carlo simulation for EBSD",
        },
        {
            "code": "MASTER",
            "label": "Master",
            "description": "EBSD Master pattern simulation",
        },
        {
            "code": "SCREEN",
            "label": "Screen",
            "description": "EBSD Screen pattern simulation",
        },
    ]
    voc_ebsd_sim_type = oBis.new_vocabulary(
        code="EBSDSIM_TYPE",
        description="Type of the EBSD Simulation",
        terms=terms_ebsd_sim,
    )
    register_controlled_vocabulary(oBis, voc_ebsd_sim_type, terms_ebsd_sim)

    # Atomistic Simulation types
    terms_atom_sim = [
        {
            "code": "ATOM-MC",
            "label": "Monte-Carlo (Atomistic)",
            "description": "Monte-Carlo Atomistic simulation: http://purls.helmholtz-metadaten.de/asmo/MonteCarloMethod",
        },
        {
            "code": "ATOM-MD",
            "label": "Molecular Dynamics (Atomistic)",
            "description": "Molecular Dynamics (Atomistic) simulation: http://purls.helmholtz-metadaten.de/asmo/MolecularDynamics",
        },
        {
            "code": "ATOM-MS",
            "label": "Molecular Statics (Atomistic)",
            "description": "Molecular Statics (Atomistic) simulation: http://purls.helmholtz-metadaten.de/asmo/MolecularStatics",
        },
        {
            "code": "ATOM-SG",
            "label": "Sample Generation (Atomistic)",
            "description": "Sample Generation (Atomistic) simulation",
        },
    ]
    voc_atom_sim_type = oBis.new_vocabulary(
        code="ATOMSIM_TYPE",
        description="Type of the Atomistic Simulation",
        terms=terms_atom_sim,
    )
    register_controlled_vocabulary(oBis, voc_atom_sim_type, terms_atom_sim)

    # Euler Angle conventions
    terms_euler_angle = [
        {"code": "HKL", "label": "hkl", "description": "hkl"},
        {"code": "TSL", "label": "tsl", "description": "tsl"},
    ]
    voc_euler_angle = oBis.new_vocabulary(
        code="EULERANGLE_CONVENTION",
        description="Convention for Euler angles",
        terms=terms_euler_angle,
    )
    register_controlled_vocabulary(oBis, voc_euler_angle, terms_euler_angle)

    # Deformation Mode
    terms_deform_mode = [
        {"code": "BEND", "label": "Bending", "description": "Bending test"},
        {
            "code": "COMPRESS",
            "label": "Compression",
            "description": "Compression test",
        },
        {
            "code": "TENSILE",
            "label": "Tensile",
            "description": "Tensile test",
        },
    ]
    voc_deform_mode = oBis.new_vocabulary(
        code="DEFORMATION_MODE",
        description="Deformation mode for mechanical testing",
        terms=terms_deform_mode,
    )
    register_controlled_vocabulary(oBis, voc_deform_mode, terms_deform_mode)

    # Manufacturer
    terms_manufacturer = [
        {"code": "ZEISS", "label": "Zeiss", "description": "Zeiss"},
        {"code": "TESCAN", "label": "Tescan", "description": "Tescan"},
        {
            "code": "FEI",
            "label": "FEI / Thermo Fisher Scientific",
            "description": "FEI / ThermoFisher",
        },
        {"code": "JEOL", "label": "JEOL", "description": "JEOL"},
        {"code": "HITACHI", "label": "Hitachi", "description": "Hitachi"},
        {"code": "LEICA", "label": "Leica", "description": "Leica"},
        {"code": "BRUKER", "label": "Bruker", "description": "Bruker"},
        {"code": "NIKON", "label": "Nikon", "description": "Nikon"},
        {
            "code": "OXFORD",
            "label": "Oxford Instruments",
            "description": "Oxford Instruments",
        },
        {"code": "KLA", "label": "KLA", "description": "KLA"},
        {"code": "FEMTOTOOLS", "label": "FemtoTools", "description": "FemtoTools"},
        {
            "code": "PANALYTICAL",
            "label": "Malvern Panalytical",
            "description": "Malvern Panalytical",
        },
        {"code": "PERKINELMER", "label": "PerkinElmer", "description": "PerkinElmer"},
        {"code": "CAMECA", "label": "Cameca", "description": "Cameca"},
    ]
    voc_manufacturer = oBis.new_vocabulary(
        code="MANUFACTURER_VOCAB",
        description="Manufacturer",
        terms=terms_manufacturer,
    )
    register_controlled_vocabulary(oBis, voc_manufacturer, terms_manufacturer)

    # Tool used for computation
    terms_compute_type = [
        {"code": "WORKSTATION", "label": "Workstation", "description": "Workstation"},
        {"code": "HPC", "label": "HPC cluster", "description": "HPC cluster"},
        {"code": "HPC_NODE", "label": "HPC node", "description": "HPC node"},
    ]
    voc_compute_type = oBis.new_vocabulary(
        code="COMPUTE_RESOURCE_VOCAB",
        description="Type of Compute Resource",
        terms=terms_compute_type,
    )
    register_controlled_vocabulary(oBis, voc_compute_type, terms_compute_type)

    terms_operating_system_vocab = [
        {
            "code": "CENTOS_8",
            "label": "CentOS 8",
            "description": "CentOS website // CentOS Webseite: ",
        },
        {
            "code": "CENTOS_7",
            "label": "CentOS 7",
            "description": "CentOS website // CentOS Webseite: ",
        },
        {
            "code": "DEBIAN_10_BUSTER",
            "label": "Debian GNU/Linux 10 (buster)",
            "description": "Debian website // Debian Webseite: https://www.debian.org/releases/buster/",
        },
        {
            "code": "DEBIAN_11_BULLSEYE",
            "label": "Debian GNU/Linux 11 (bullseye)",
            "description": "Debian website // Debian Webseite: https://www.debian.org/releases/bullseye/",
        },
        {
            "code": "DEBIAN_12_BOOKWORM",
            "label": "Debian GNU/Linux 12 (bookworm)",
            "description": "Debian website // Debian Webseite: https://www.debian.org/releases/bookworm/",
        },
        {
            "code": "DEBIAN_13_TRIXIE",
            "label": "Debian GNU/Linux 13 (trixie)",
            "description": "Debian website // Debian Webseite: https://www.debian.org/releases/trixie/",
        },
        {
            "code": "RHEL_8_0_OOTPA",
            "label": "Red Hat Enterprise Linux 8.0 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.0_release_notes/index",
        },
        {
            "code": "RHEL_8_1_OOTPA",
            "label": "Red Hat Enterprise Linux 8.1 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.1_release_notes/index",
        },
        {
            "code": "RHEL_8_2_OOTPA",
            "label": "Red Hat Enterprise Linux 8.2 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.2_release_notes/index",
        },
        {
            "code": "RHEL_8_3_OOTPA",
            "label": "Red Hat Enterprise Linux 8.3 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.3_release_notes/index",
        },
        {
            "code": "RHEL_8_4_OOTPA",
            "label": "Red Hat Enterprise Linux 8.4 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.4_release_notes/index",
        },
        {
            "code": "RHEL_8_5_OOTPA",
            "label": "Red Hat Enterprise Linux 8.5 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.5_release_notes/index",
        },
        {
            "code": "RHEL_8_6_OOTPA",
            "label": "Red Hat Enterprise Linux 8.6 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.6_release_notes/index",
        },
        {
            "code": "RHEL_8_7_OOTPA",
            "label": "Red Hat Enterprise Linux 8.7 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.7_release_notes/index",
        },
        {
            "code": "RHEL_8_8_OOTPA",
            "label": "Red Hat Enterprise Linux 8.8 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.8_release_notes/index",
        },
        {
            "code": "RHEL_8_9_OOTPA",
            "label": "Red Hat Enterprise Linux 8.9 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.9_release_notes/index",
        },
        {
            "code": "RHEL_8_10_OOTPA",
            "label": "Red Hat Enterprise Linux 8.10 (Ootpa)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/8/html-single/8.10_release_notes/index",
        },
        {
            "code": "RHEL_9_0_PLOW",
            "label": "Red Hat Enterprise Linux 9.0 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.0_release_notes/index",
        },
        {
            "code": "RHEL_9_1_PLOW",
            "label": "Red Hat Enterprise Linux 9.1 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.1_release_notes/index",
        },
        {
            "code": "RHEL_9_2_PLOW",
            "label": "Red Hat Enterprise Linux 9.2 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.2_release_notes/index",
        },
        {
            "code": "RHEL_9_3_PLOW",
            "label": "Red Hat Enterprise Linux 9.3 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.3_release_notes/index",
        },
        {
            "code": "RHEL_9_4_PLOW",
            "label": "Red Hat Enterprise Linux 9.4 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.4_release_notes/index",
        },
        {
            "code": "RHEL_9_5_PLOW",
            "label": "Red Hat Enterprise Linux 9.5 (Plow)",
            "description": "RHEL website // RHEL Webseite: https://docs.redhat.com/en/documentation/red_hat_enterprise_linux/9/html-single/9.5_release_notes/index",
        },
        {
            "code": "ROCKY_8",
            "label": "Rocky Linux 8",
            "description": "Rocky Linux website // Rocky Linux Webseite: ",
        },
        {
            "code": "ROCKY_9",
            "label": "Rocky Linux 9",
            "description": "Rocky Linux website // Rocky Linux Webseite: ",
        },
        {
            "code": "SUSE_SLES_15_SP5",
            "label": "SUSE Linux Enterprise Server 15 SP5",
            "description": "SLES website // SLESWebseite: https://documentation.suse.com/de-de/sles/15-SP5/",
        },
        {
            "code": "UBUNTU_18_04",
            "label": "Ubuntu 18.04",
            "description": "Ubuntu website // Ubuntu Webseite: https://releases.ubuntu.com/bionic/",
        },
        {
            "code": "UBUNTU_20_04",
            "label": "Ubuntu 20.04",
            "description": "Ubuntu website // Ubuntu Webseite: https://releases.ubuntu.com/focal/",
        },
        {
            "code": "UBUNTU_22_04_4",
            "label": "Ubuntu 22.04.4 LTS",
            "description": "Ubuntu website // Ubuntu Webseite: https://releases.ubuntu.com/jammy/",
        },
        {
            "code": "UBUNTU_22_04",
            "label": "Ubuntu 22.04",
            "description": "Ubuntu website // Ubuntu Webseite: https://releases.ubuntu.com/jammy/",
        },
        {
            "code": "UBUNTU_24_04",
            "label": "Ubuntu 24.04",
            "description": "Ubuntu website // Ubuntu Webseite: https://releases.ubuntu.com/oracular/",
        },
        {
            "code": "WINDOWS_7_ENT",
            "label": "Windows 7 Enterprise",
            "description": "Windows website // Windows Webseite: https://www.microsoft.com/en-us/microsoft-365/windows/windows-11-enterprise",
        },
        {
            "code": "WINDOWS_10_EDU",
            "label": "Windows 10 Education",
            "description": "Windows website // Windows Webseite: ",
        },
        {
            "code": "WINDOWS_10_ENT",
            "label": "Windows 10 Enterprise",
            "description": "Windows website // Windows Webseite: https://learn.microsoft.com/en-us/windows/release-health/release-information",
        },
        {
            "code": "WINDOWS_11_EDU",
            "label": "Windows 11 Education",
            "description": "Windows website // Windows Webseite: ",
        },
        {
            "code": "WINDOWS_11_ENT",
            "label": "Windows 11 Enterprise",
            "description": "Windows website // Windows Webseite: https://www.microsoft.com/en-us/microsoft-365/windows/windows-11-enterprise",
        },
        {
            "code": "WINDOWS_11_PRO",
            "label": "Windows 11 Pro",
            "description": "Windows website // Windows Webseite: ",
        },
    ]
    voc_operating_system_vocab = oBis.new_vocabulary(
        code="OPERATING_SYSTEM_VOCAB",
        description="Operating System (OS) // Betriebssystem",
        terms=terms_operating_system_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_operating_system_vocab, terms_operating_system_vocab
    )

    terms_queuing_system_vocab = [
        {
            "code": "OPEN_PBS",
            "label": "openPBS",
            "description": "openPBS documentation // openPBS Dokumentation: https://openpbs.atlassian.net/wiki/spaces/PBSPro/pages/5537831/User+Documentation",
        },
        {
            "code": "ORACLE_GE",
            "label": "Oracle Grid Engine",
            "description": "Oracle Grid Engine documentation // Oracle Grid Engine Dokumentation: https://gridscheduler.sourceforge.net/documentation.html",
        },
        {
            "code": "PBS_PRO",
            "label": "PBS Pro",
            "description": "PBS Pro documentation // PBS Pro Dokumentation: https://openpbs.atlassian.net/wiki/spaces/PBSPro/pages/5537831/User+Documentation",
        },
        {
            "code": "SLURM",
            "label": "SLURM",
            "description": "SLURM documentation // SLURM Dokumentation: https://slurm.schedmd.com/documentation.html",
        },
        {
            "code": "TORQUE",
            "label": "TORQUE",
            "description": "TORQUE documentation // TORQUE Dokumentation: https://support.adaptivecomputing.com/torque-resource-manager-documentation/",
        },
    ]
    voc_queuing_system_vocab = oBis.new_vocabulary(
        code="QUEUING_SYSTEM_VOCAB",
        description="Queuing System used by HPC // Von HPC verwendetes Warteschlangensystem",
        terms=terms_queuing_system_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_queuing_system_vocab, terms_queuing_system_vocab
    )

    # #########################################################################
    # Plugins
    # #########################################################################

    date_validation_plugin_1 = oBis.new_plugin(
        name="DEFAULT_EXPERIMENT.date_range_validation",
        pluginType="ENTITY_VALIDATION",
        script=open("jython_scripts/date_range_validation.py", "r").read(),
    )
    register_plugin(oBis, date_validation_plugin_1)

    date_validation_plugin_2 = oBis.new_plugin(
        name="EXPERIMENTAL_STEP.date_range_validation",
        pluginType="ENTITY_VALIDATION",
        script=open("jython_scripts/date_range_validation.py", "r").read(),
    )
    register_plugin(oBis, date_validation_plugin_2)

    dynamic_property_plugin_1 = oBis.new_plugin(
        name="ENTITY_DATASET_COUNT",
        pluginType="DYNAMIC_PROPERTY",
        script=open("jython_scripts/DatasetCount.py", "r").read(),
    )
    dynamic_property_plugin_1.description = (
        "Count the number of datasets linked to the entity"
    )
    register_plugin(oBis, dynamic_property_plugin_1)

    # #########################################################################
    # Property Types
    # #########################################################################

    pt_finished_flag = oBis.new_property_type(
        code="FINISHED_FLAG",
        label="Experiment completed",
        description="Marks the experiment as finished",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_finished_flag)

    pt_start_date = oBis.new_property_type(
        code="START_DATE",
        label="Start date",
        description="Start date",
        dataType="TIMESTAMP",
    )
    register_property_type(oBis, pt_start_date)

    pt_end_date = oBis.new_property_type(
        code="END_DATE",
        label="End date",
        description="End date",
        dataType="TIMESTAMP",
    )
    register_property_type(oBis, pt_end_date)

    pt_goals = oBis.new_property_type(
        code="EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS",
        label="Experimental goals",
        description="Goals of the experiment",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_goals)

    pt_description = oBis.new_property_type(
        code="EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION",
        label="Experimental description",
        description="Description of the experiment",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_description)

    pt_results = oBis.new_property_type(
        code="EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS",
        label="Experimental results",
        description="Summary of experimental results",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_results)

    pt_spreadsheet = oBis.new_property_type(
        code="EXPERIMENTAL_STEP.SPREADSHEET",
        label="Spreadsheet",
        description="Multi purpose Spreadsheet",
        dataType="XML",
    )
    register_property_type(oBis, pt_spreadsheet)

    pt_references = oBis.new_property_type(
        code="REFERENCE",
        label="References",
        description="Useful references // Nützliche Referenzen",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_references)

    pt_publication = oBis.new_property_type(
        code="PUBLICATION",
        label="Publication",
        description="Own publication where this entity is referenced",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_publication)

    pt_notes = oBis.new_property_type(
        code="NOTES",
        label="Notes",
        description="Notes // Notizen",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_notes)

    pt_goals = oBis.new_property_type(
        code="DEFAULT_EXPERIMENT.EXPERIMENTAL_GOALS",
        label="Goals",
        description="Goals of the experiment",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_goals)

    pt_description = oBis.new_property_type(
        code="DEFAULT_EXPERIMENT.EXPERIMENTAL_DESCRIPTION",
        label="Description",
        description="Description of the experiment",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_description)

    pt_results = oBis.new_property_type(
        code="DEFAULT_EXPERIMENT.EXPERIMENTAL_RESULTS",
        label="Results",
        description="Summary of experimental results",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_results)

    pt_grant = oBis.new_property_type(
        code="DEFAULT_EXPERIMENT.GRANT",
        label="Grant",
        description="Grant",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_grant)

    # Presigned URL to download files from S3
    # NOT extracted by the parser, periodically created by an automatic script
    pt_download_link = oBis.new_property_type(
        code="S3_DOWNLOAD_LINK",
        label="Download Link",
        description="Download URL for dataset stored in S3/Coscine",
        dataType="HYPERLINK",
    )
    register_property_type(oBis, pt_download_link)

    # Institution
    pt_institution = oBis.new_property_type(
        code="INSTITUTION",
        label="Institution",
        description="Company/Institute/Lab where the data was created",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_institution)

    # Notebook type
    pt_nb_type = oBis.new_property_type(
        code="NOTEBOOK_TYPE",
        label="Notebook type",
        description="Type of notebook",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_nb_type.code,
    )
    register_property_type(oBis, pt_nb_type)

    # Comments
    pt_comments = oBis.new_property_type(
        code="COMMENTS",
        label="Comments",
        description="Comments",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_comments)

    # Description - Short description
    pt_description = oBis.new_property_type(
        code="DESCRIPTION",
        label="Description",
        description="Further details",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_description)

    # Description - Short description Multiline
    pt_description_multiline = oBis.new_property_type(
        code="DESCRIPTION_MULTILINE",
        label="Description",
        description="Short description and/or purpose // Kurzbeschreibung und/oder Zweck",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_description_multiline)

    # Conceptual Dictionary - Pyiron
    pt_pyiron_concept_dict = oBis.new_property_type(
        code="PYIRON_CONCEPTUAL_DICTIONARY",
        label="Conceptual Dictionary",
        description="Conceptual dictionary associated with pyiron // Begriffswörterbuch zu pyiron",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_pyiron_concept_dict)

    # Workflow manager - Pyiron
    pt_workflow_manager = oBis.new_property_type(
        code="WORKFLOW_MANAGER",
        label="Workflow Manager",
        description="Workflow manager // Workflow-Manager",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_workflow_manager)

    # Year of purchase
    pt_year_of_purchase = oBis.new_property_type(
        code="YEAR_OF_PURCHASE",
        label="Year of Purchase",
        description="Year of Purchase",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_year_of_purchase)

    # Company / Manufacturer / Supplier
    pt_company = oBis.new_property_type(
        code="PRODUCT.COMPANY",
        label="Company",
        description="Company",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_company)

    # Address
    pt_address = oBis.new_property_type(
        code="ADDRESS",
        label="Address",
        description="Address",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_address)

    # Time
    pt_time = oBis.new_property_type(
        code="TIME",
        label="Time",
        description="Time",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_time)

    # Date
    pt_date = oBis.new_property_type(
        code="DATE",
        label="Date",
        description="Date (iso-format)",
        dataType="DATE",
    )
    register_property_type(oBis, pt_date)

    # Datetime
    pt_datetime = oBis.new_property_type(
        code="DATETIME",
        label="DateTime",
        description="DateTime",
        dataType="TIMESTAMP",
    )
    register_property_type(oBis, pt_datetime)

    # Datetime End
    pt_datetime_end = oBis.new_property_type(
        code="DATETIME_END",
        label="DateTime End",
        description="DateTime (End)",
        dataType="TIMESTAMP",
    )
    register_property_type(oBis, pt_datetime_end)

    # Device Name
    pt_device = oBis.new_property_type(
        code="DEVICE",
        label="Device",
        description="Generic Device Name",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_device)

    # Device Model
    pt_device_model = oBis.new_property_type(
        code="DEVICEMODEL",
        label="Device Model",
        description="Generic Device Model",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_device_model)

    # Manufacturer
    pt_device_manufacturer = oBis.new_property_type(
        code="DEVICE_MANUFACTURER",
        label="Device Manufacturer",
        description="Manufacturer of the device",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_manufacturer.code,
    )
    register_property_type(oBis, pt_device_manufacturer)

    # Device full name (incl. model code)
    pt_device_name = oBis.new_property_type(
        code="DEVICE_NAME",
        label="Device Name",
        description="Name of the device",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_device_name)

    # Generic location
    pt_location = oBis.new_property_type(
        code="LOCATION",
        label="Location",
        description="Generic location (RWTH-IMM:E01, virtual, processed)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_location)

    # Specific location
    pt_sublocation = oBis.new_property_type(
        code="SUBLOCATION",
        label="Sublocation",
        description="Specific location (desiccator, workstation, etc.)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_sublocation)

    # Jobs submitted to a cluster are usually identified with a unique JobID
    pt_job_id = oBis.new_property_type(
        code="CLUSTER_JOBID",
        label="JobID",
        description="Unique job identifier on compute cluster",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_job_id)

    # Number of cores used in a job
    pt_job_ncores = oBis.new_property_type(
        code="NCORES",
        label="Number of Cores",
        description="Number of cores used // Anzahl der Kerne",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_job_ncores)

    # Number of GPUs used in a job
    pt_job_ngpus = oBis.new_property_type(
        code="NGPUS",
        label="Number of GPUs",
        description="Number of GPUs used // Anzahl der GPUs",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_job_ngpus)

    # Number of threads used in a job
    pt_job_nthreads = oBis.new_property_type(
        code="NTHREADS",
        label="Number of Threads",
        description="Number of Threads used // Anzahl der Threads",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_job_nthreads)

    # Number of nodes used in a job
    pt_job_nnodes = oBis.new_property_type(
        code="NNODES",
        label="Number of Nodes",
        description="Number of nodes used by job",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_job_nnodes)

    # Job run time
    pt_job_run_time = oBis.new_property_type(
        code="JOB_RUN_TIME_SEC",
        label="Run Time (s)",
        description="Run Time [s]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_job_run_time)

    # Requested run time
    pt_job_run_time_req = oBis.new_property_type(
        code="JOB_REQ_RUN_TIME",
        label="Run Time (Requested)",
        description="Requested Run Time",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_job_run_time_req)

    # General Simulation type
    pt_sim_type = oBis.new_property_type(
        code="SIMULATION_TYPE",
        label="General simulation type",
        description="Type of simulation (DFT/Atomistic/EBSD)",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_sim_type.code,
    )
    register_property_type(oBis, pt_sim_type)

    # EBSD Simulation type
    pt_ebsd_sim_type = oBis.new_property_type(
        code="EBSDSIM_TYPE",
        label="EBSD Simulation Type",
        description="Type of the EBSD Simulation",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_ebsd_sim_type.code,
    )
    register_property_type(oBis, pt_ebsd_sim_type)

    # Atomistic Simulation type
    pt_atom_sim_type = oBis.new_property_type(
        code="ATOMSIM_TYPE",
        label="Atomistic Simulation Type",
        description="Type of the Atomistic Simulation",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_atom_sim_type.code,
    )
    register_property_type(oBis, pt_atom_sim_type)

    # Flag for STEM images
    pt_is_stem = oBis.new_property_type(
        code="IS_STEM",
        label="Is STEM",
        description="Flag for STEM images",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_is_stem)

    # SEM/(S)TEM detectors
    for i in range(1, 1 + MAX_NUM_DETECTORS):
        pt_detector_name = oBis.new_property_type(
            code=f"DETECTOR{i}_NAME",
            label=f"Detector{i}",
            description=f"Detector{i}",
            dataType="VARCHAR",
        )
        register_property_type(oBis, pt_detector_name)

        pt_detector_type = oBis.new_property_type(
            code=f"DETECTOR{i}_TYPE",
            label=f"Detector{i} Type",
            description=f"Type of Detector{i}",
            dataType="VARCHAR",
        )
        register_property_type(oBis, pt_detector_type)

        pt_detector_gain = oBis.new_property_type(
            code=f"DETECTOR{i}_GAIN",
            label=f"Detector{i} Gain",
            description=f"Detector{i}Gain",
            dataType="REAL",
        )
        register_property_type(oBis, pt_detector_gain)

        pt_detector_offset = oBis.new_property_type(
            code=f"DETECTOR{i}_OFFSET",
            label=f"Detector{i} Offset",
            description=f"Detector{i} Offset",
            dataType="REAL",
        )
        register_property_type(oBis, pt_detector_offset)

    # Beam current - SEM / EBSD SIM
    pt_beam_current = oBis.new_property_type(
        code="BEAM_CURRENT_NA",
        label="Beam Current (nA)",
        description="Electron Beam Current in nA: https://w3id.org/pmd/mo/BeamCurrent,http://qudt.org/vocab/unit/NanoA",
        dataType="REAL",
    )
    register_property_type(oBis, pt_beam_current)

    # Electron Beam Dwell Time - SEM / EBSD SIM
    pt_dwell_time = oBis.new_property_type(
        code="DWELL_TIME",
        label="Dwell Time (μs)",
        description="Electron Beam Dwell Time in μs: https://w3id.org/pmd/mo/DwellTime,http://qudt.org/vocab/unit/MicroSEC",
        dataType="REAL",
    )
    register_property_type(oBis, pt_dwell_time)

    # Euler angle convention
    pt_eulerangle_convention = oBis.new_property_type(
        code="EULERANGLE_CONVENTION",
        label="Euler angle convention",
        description="Convention for Euler angles",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_euler_angle.code,
    )
    register_property_type(oBis, pt_eulerangle_convention)

    # Generic (sample) weight
    pt_weight_g = oBis.new_property_type(
        code="WEIGHT_G",
        label="Weight (g)",
        description="Weight [g]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_weight_g)

    # Sample temperature
    pt_sample_temperature = oBis.new_property_type(
        code="SAMPLE_TEMP",
        label="Sample Temperature (°C)",
        description="Sample Temperature [°C]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sample_temperature)

    # Room temperature
    pt_room_temperature = oBis.new_property_type(
        code="ROOM_TEMP",
        label="Room Temperature (°C)",
        description="Room temperature in which the experiment was performed",
        dataType="REAL",
    )
    register_property_type(oBis, pt_room_temperature)

    # Room Relative Humidity - SEM
    pt_room_rel_humidity = oBis.new_property_type(
        code="ROOM_REL_HUMIDITY",
        label="Room relative humidity (%)",
        description="Relative humidity of the room where the experiment "
        "was performed",
        dataType="REAL",
    )
    register_property_type(oBis, pt_room_rel_humidity)

    # Gas used in Experiment - Casting / SEM
    pt_gas_exp = oBis.new_property_type(
        code="GAS_EXP",
        label="Gas in experiment",
        description="Gas used in experimental setup",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_gas_exp)

    # Pre tilt angle - EBSD EXP
    pt_pretilt_angle = oBis.new_property_type(
        code="PRETILT_ANGLE",
        label="Pre-tilt angle (deg)",
        description="Pre-tilt angle due to the use of a sample holder "
        "(if applicable)",
        dataType="REAL",
    )
    register_property_type(oBis, pt_pretilt_angle)

    # Point Measurement (value +/- stat + syst_up - syst_down)

    pt_point_measurment_value = oBis.new_property_type(
        code="VALUE",
        label="Measured Value",
        description="Measured Value",
        dataType="REAL",
    )
    register_property_type(oBis, pt_point_measurment_value)

    pt_statistical_uncertainty = oBis.new_property_type(
        code="UNCERTAINTY_STAT",
        label="Statistical uncertainty",
        description="Statistical uncertainty",
        dataType="REAL",
    )
    register_property_type(oBis, pt_statistical_uncertainty)

    pt_upper_systematic_uncertainty = oBis.new_property_type(
        code="UNCERTAINTY_SYST_UP",
        label="Upper systematic uncertainty",
        description="Upper systematic uncertainty",
        dataType="REAL",
    )
    register_property_type(oBis, pt_upper_systematic_uncertainty)

    pt_lower_systematic_uncertainty = oBis.new_property_type(
        code="UNCERTAINTY_SYST_LOW",
        label="Lower systematic uncertainty",
        description="Lower systematic uncertainty",
        dataType="REAL",
    )
    register_property_type(oBis, pt_lower_systematic_uncertainty)

    # Point Measurement Quantity
    pt_point_measurment_quantity = oBis.new_property_type(
        code="PHYSICAL_QUANTITY",
        label="Physical Quantity",
        description="Measured / Calculated Physical Quantity",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_quantity_type.code,
    )
    register_property_type(oBis, pt_point_measurment_quantity)

    # Point Measurement Unit
    pt_point_measurment_unit = oBis.new_property_type(
        code="UNIT",
        label="Measurement Unit",
        description="Measurement Unit",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_point_measurment_unit)

    # Generic voltage
    pt_voltage = oBis.new_property_type(
        code="VOLTAGE",
        label="Voltage (V)",
        description="Voltage in volts",
        dataType="REAL",
    )
    register_property_type(oBis, pt_voltage)

    # Generic Temperature
    pt_temperature = oBis.new_property_type(
        code="TEMPERATURE",
        label="Temperature (K)",
        description="Temperature in Kelvin",
        dataType="REAL",
    )
    register_property_type(oBis, pt_temperature)

    # Generic minimum temperature
    pt_temperature_min = oBis.new_property_type(
        code="TEMPERATURE_MIN",
        label="Min. Temperature (°C)",
        description="Minimum Temperature [°C]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_temperature_min)

    # Generic maximum temperature
    pt_temperature_max = oBis.new_property_type(
        code="TEMPERATURE_MAX",
        label="Max. Temperature (°C)",
        description="Maximum Temperature [°C]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_temperature_max)

    # Generic maximum weight allowed by device / instrument
    pt_weightmax = oBis.new_property_type(
        code="WEIGHT_MAX",
        label="Max. Weight (g)",
        description="Maximum Weight [g]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_weightmax)

    # Generic maximum force allowed by device / instrument
    pt_force_max = oBis.new_property_type(
        code="FORCE_MAX",
        label="Max. Force (kN)",
        description="Maximum Force [kN]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_force_max)

    # Generic maximum power allowed by device / instrument
    pt_power_max = oBis.new_property_type(
        code="POWER_MAX",
        label="Max. Power (W)",
        description="Maximum Power [W]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_power_max)

    # Maximum sample geometry allowed by device / instrument
    pt_sample_geometry_max = oBis.new_property_type(
        code="SAMPLE_GEOMETRY_MAX",
        label="Max. Sample Geometry (mm mm mm)",
        description="Maximum Sample Geometry [mm mm mm]",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_sample_geometry_max)

    # Minimum sampling rate
    pt_sampling_rate_min = oBis.new_property_type(
        code="SAMPLING_RATE_MIN",
        label="Min. Sampling Rate (Hz)",
        description="Minimum Sampling Rate [Hz]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sampling_rate_min)

    # Maximum sampling rate
    pt_sampling_rate_max = oBis.new_property_type(
        code="SAMPLING_RATE_MAX",
        label="Max. Sampling Rate (Hz)",
        description="Maximum Sampling Rate [Hz]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sampling_rate_max)

    # Sampling rate
    pt_sampling_rate = oBis.new_property_type(
        code="SAMPLING_RATE_HZ",
        label="Sampling Rate (Hz)",
        description="Sampling Rate [Hz]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sampling_rate)

    # Internal ID - some internal number used in the organisation
    # e.g. "Stoffnummer" in DaMaRIS used by RWTH for chemicals
    pt_internal_id = oBis.new_property_type(
        code="ID_INTERNAL",
        label="Internal ID",
        description="Internal ID number used within an organisation",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_internal_id)

    # Alternative names
    pt_alternative_name = oBis.new_property_type(
        code="ALT_NAME",
        label="Alternative names",
        description="Aliases or synonyms",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_alternative_name)

    # Calibration date
    pt_calibration_date = oBis.new_property_type(
        code="CALIBRATION_DATE",
        label="Calibration Date",
        description="Calibration Date",
        dataType="TIMESTAMP",
    )
    register_property_type(oBis, pt_calibration_date)

    # Fabrication Date
    pt_fabrication_date = oBis.new_property_type(
        code="FABRICATION_DATE",
        label="Fabrication Date",
        description="Fabrication Date",
        dataType="DATE",
    )
    register_property_type(oBis, pt_fabrication_date)

    # Output filename - EBSD SIM
    pt_output_filename = oBis.new_property_type(
        code="OUTPUT_FILENAME",
        label="Output Filename",
        description="Output Filename",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_output_filename)

    # Crystal structure filename - EBSD SIM
    pt_crystal_structure_filename = oBis.new_property_type(
        code="CRYSTAL_STRUCTURE_FILENAME",
        label="Filename of the crystal structure",
        description="Filename holding details of the crystal structure",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_crystal_structure_filename)

    # User Name - EBSD SIM / SEM / STEM / Pyiron
    pt_username = oBis.new_property_type(
        code="USER_NAME",
        label="User Name",
        description="User Name",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_username)

    # Data Size X
    pt_data_size_x = oBis.new_property_type(
        code="DATA_SIZE_X",
        label="DataSize X",
        description="Number of columns in data array",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_data_size_x)

    # Data Size Y
    pt_data_size_y = oBis.new_property_type(
        code="DATA_SIZE_Y",
        label="DataSize Y",
        description="Number of rows in data array",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_data_size_y)

    # Number of frames
    pt_number_of_frames = oBis.new_property_type(
        code="NUMBER_OF_FRAMES",
        label="Number Of Frames",
        description="Number of frames in image array",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_frames)

    # Number of channels
    pt_number_of_channels = oBis.new_property_type(
        code="NUMBER_OF_CHANNELS",
        label="Number Of Channels",
        description="Number of channels in image array (1=GRAYSCALE,3=RGB)",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_channels)

    # Number of datasets
    # In the case of STEM, each dataset corresponds to a detector
    pt_number_of_datasets = oBis.new_property_type(
        code="NUMBER_OF_DATASETS",
        label="Number Of Datasets",
        description="Number of datasets (number of detectors for STEM data)",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_datasets)

    # Bit depth
    pt_bit_depth = oBis.new_property_type(
        code="BIT_DEPTH",
        label="Bit Depth",
        description="Bit Depth",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_bit_depth)

    # Datatype (uint8, float32, etc.)
    pt_datatype = oBis.new_property_type(
        code="DATA_TYPE",
        label="Data Type",
        description="Data Type (uint8, float32, etc.)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_datatype)

    # Flag for linear LUT
    pt_linear_lut = oBis.new_property_type(
        code="LINEAR_LUT",
        label="Linear LUT",
        description="True if pixel values are mapped to grayscale (not RGB)",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_linear_lut)

    # Instrument (free-text) - Metallography
    # Whenever more than one instrument is used, a number of steps per instrument
    # should be given (as indicated by the hint)
    pt_instrument = oBis.new_property_type(
        code="INSTRUMENT",
        label="Instrument",
        description="Instruments (NAME1:NUM_STEPS1+NAME2:NUM_STEPS2+NAME3:NUM_STEPS3)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_instrument)

    # Deformation Mode
    pt_deformation_mode = oBis.new_property_type(
        code="DEFORMATION_MODE",
        label="Deformation Mode",
        description="Deformation Mode",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_deform_mode.code,
    )
    register_property_type(oBis, pt_deformation_mode)

    # Electrolyte (free-text)
    pt_electrolyte = oBis.new_property_type(
        code="ELECTROLYTE",
        label="Electrolyte",
        description="Electrolyte used in Electropolishing / Electrochemistry experiment",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_electrolyte)

    # Experiment Type
    pt_exp_type = oBis.new_property_type(
        code="EXP_TYPE",
        label="Experiment Type",
        description="Experiment Type",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_exp_type.code,
    )
    register_property_type(oBis, pt_exp_type)

    # Compute Resource
    pt_compute_type = oBis.new_property_type(
        code="COMPUTE_RESOURCE_TYPE",
        label="Compute Resource Type",
        description="Type of compute resource",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_compute_type.code,
    )
    register_property_type(oBis, pt_compute_type)

    # Operating System
    pt_operating_system = oBis.new_property_type(
        code="OPERATING_SYSTEM",
        label="Operating System",
        description="Operating System (OS) // Betriebssystem",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_operating_system_vocab.code,
    )
    register_property_type(oBis, pt_operating_system)

    # Queuing System used by HPC
    pt_hpc_queue_type = oBis.new_property_type(
        code="QUEUING_SYSTEM",
        label="Queuing System",
        description="Queuing System used by HPC // Warteschlangensystem des HPCs",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_queuing_system_vocab.code,
    )
    register_property_type(oBis, pt_hpc_queue_type)

    # HPC Queue
    pt_hpc_job_queue = oBis.new_property_type(
        code="HPC_JOB_QUEUE",
        label="HPC Job Queue",
        description="HPC queue used // Verwendete HPC-Warteschlange",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_hpc_job_queue)

    # Job ID in HPC
    pt_hpc_job_id = oBis.new_property_type(
        code="HPC_JOB_ID",
        label="HPC Job ID",
        description="Job ID in the HPC queue // Job-ID in der HPC-Warteschlange",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_hpc_job_id)

    # Tool used for managing the python environment
    pt_env_tool = oBis.new_property_type(
        code="ENV_TOOL",
        label="Environment Tool",
        description="Tool used for managing the environment (eg. conda, poetry, etc.)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_env_tool)

    # Hash (helps identify file by content)
    pt_md5_hash = oBis.new_property_type(
        code="MD5_HASH",
        label="Hash",
        description="MD5 Hash of file",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_md5_hash)

    # Image Content
    pt_image_content = oBis.new_property_type(
        code="IMAGE_CONTENT",
        label="Image Content",
        description="Image Content",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_image_content)

    # Number of datasets registered to an Object
    # LinkedData does not show up in ELN-LIMS GUI if multiple files are registered.
    pt_num_datasets = oBis.new_property_type(
        code="NUM_DATASETS",
        label="Number of datasets",
        description="Number of datasets linked to the object",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_num_datasets)

    # Location of measurement in the sample
    pt_measurement_position = oBis.new_property_type(
        code="MEASUREMENT_POSITION",
        label="Measurement Position",
        description="Position of the measurement on the sample",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_measurement_position)

    # #########################################################################
    # Collection
    # #########################################################################
    try:
        col = oBis.get_experiment_type("DEFAULT_EXPERIMENT")
    except Exception:
        col = oBis.new_collection_type(
            code="DEFAULT_EXPERIMENT",
            validationPlugin=date_validation_plugin_1.name,
        )
        col.save()

    sec2props_default_experiment = {
        "General": [
            ("$NAME", 1, None),
            ("$DEFAULT_OBJECT_TYPE", 0, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("DEFAULT_EXPERIMENT.GRANT", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Experimental Details": [
            ("DEFAULT_EXPERIMENT.EXPERIMENTAL_GOALS", 0, None),
            ("DEFAULT_EXPERIMENT.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("DEFAULT_EXPERIMENT.EXPERIMENTAL_RESULTS", 0, None),
        ],
        "References": [
            ("REFERENCE", 0, None),
            ("PUBLICATION", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            # ("$XMLCOMMENTS", 0, None),
            # ('$ANNOTATIONS_STATE', 0, None),
        ],
    }
    assign_property_types(col, sec2props_default_experiment)

    # #########################################################################
    # Objects
    # #########################################################################

    # Experimental Step - Generic
    obj_experimental_step = oBis.new_object_type(
        code=code_experimental_step,
        generatedCodePrefix="EXP",
        autoGeneratedCode=True,
        validationPlugin=date_validation_plugin_2.name,
    )
    register_object_type(oBis, obj_experimental_step)

    obj_experimental_step = oBis.get_object_type(code_experimental_step)
    obj_experimental_step.description = "Experimental Step"
    register_object_type(oBis, obj_experimental_step)

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props_experimental_step = {
        "General": [
            ("$NAME", 1, None),
            ("EXP_TYPE", 0, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Experimental Details": [
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS", 0, None),
            ("EXPERIMENTAL_STEP.SPREADSHEET", 0, None),
        ],
        "References": [
            ("REFERENCE", 0, None),
            ("PUBLICATION", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            # ("$XMLCOMMENTS", 0, None),
            # ('$ANNOTATIONS_STATE', 0, None),
        ],
    }
    assign_property_types(obj_experimental_step, sec2props_experimental_step)

    # Point Measurement
    obj_point_meas = oBis.new_object_type(
        code=code_point_meas,
        generatedCodePrefix=code_point_meas,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_point_meas)

    obj_point_meas = oBis.get_object_type(code_point_meas)
    obj_point_meas.description = "Measurement of a single quantity"
    register_object_type(oBis, obj_point_meas)

    obj_point_meas = oBis.get_object_type(code_point_meas)

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props_point_meas = {
        "General": [
            ("$NAME", 1, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("COMMENTS", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Measurement Details": [
            ("PHYSICAL_QUANTITY", 0, None),
            ("VALUE", 1, None),
            ("UNIT", 1, None),
            ("UNCERTAINTY_STAT", 0, None),
            ("UNCERTAINTY_SYST_UP", 0, None),
            ("UNCERTAINTY_SYST_LOW", 0, None),
        ],
    }
    assign_property_types(obj_point_meas, sec2props_point_meas)

    # Define Compute Resource Object Type
    obj_compute_resource = oBis.new_object_type(
        code=code_compute_resource,
        generatedCodePrefix=code_compute_resource,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_compute_resource)
    
    obj_compute_resource = oBis.get_object_type(code_compute_resource)
    obj_compute_resource.description = (
        "Compute resource used for performing computations / simulations"
    )
    register_object_type(oBis, obj_compute_resource)
    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 1, None),
            ("ALT_NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
        ],
        "Technical Information": [
            ("COMPUTE_RESOURCE_TYPE", 0, None),
            ("OPERATING_SYSTEM", 0, None),
        ],
        "Technical Information - HPC": [
            ("QUEUING_SYSTEM", 0, None),
        ],
        "External Documentation": [
            ("REFERENCE", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_compute_resource, sec2props)

    # #########################################################################
    # Dataset types
    # #########################################################################

    # Notebook Type
    ds_notebook = oBis.get_dataset_type("ANALYSIS_NOTEBOOK")
    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props = {
        "General info": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("NOTEBOOK_TYPE", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            ("COMMENTS", 0, None),
            ("$XMLCOMMENTS", 0, None),
            ("$HISTORY_ID", 0, None),
        ],
    }
    assign_property_types(ds_notebook, sec2props)

    # Publication Data
    ds_pub_data = oBis.get_dataset_type("PUBLICATION_DATA")
    ds_pub_data.description = "Bibliography management data"
    ds_pub_data.save()

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props = {
        "General info": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            ("COMMENTS", 0, None),
            ("$XMLCOMMENTS", 0, None),
        ],
    }
    assign_property_types(ds_pub_data, sec2props)

    # Processed Data
    ds_processed_data = oBis.get_dataset_type("PROCESSED_DATA")
    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props = {
        "General info": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            ("COMMENTS", 0, None),
            ("$XMLCOMMENTS", 0, None),
        ],
    }
    assign_property_types(ds_processed_data, sec2props)

    # Raw Data
    ds_raw_data = oBis.get_dataset_type("RAW_DATA")
    ds_raw_data.description = "Raw Data (Use domain-specific datasetTypes instead)"
    ds_raw_data.save()

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props = {
        "General info": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),
        ],
        "Comments": [
            ("NOTES", 0, None),
            ("COMMENTS", 0, None),
            ("$XMLCOMMENTS", 0, None),
        ],
    }
    assign_property_types(ds_raw_data, sec2props)

    # # Analyzed Data
    # ds_analyzed_data = oBis.get_dataset_type('ANALYZED_DATA')
    # # Define display order
    # # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    # sec2props = {
    #     'General info': [
    #         ('$NAME', 1, None), ('DATE', 0, None),
    #         ('S3_DOWNLOAD_LINK', 0, None),
    #     ],
    #     'Comments': [
    #         ('NOTES', 0, None), ('COMMENTS', 0, None),
    #         ('$XMLCOMMENTS', 0, None),
    #     ],
    # }
    # assign_property_types(ds_analyzed_data, sec2props)
