#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for X-ray diffraction

"""


from pybis import Openbis
from schema_helpers import *


# openBIS codes for the objects (also used in other modules)

code_ds_texture_data = "XRD-TEXT_DATA"
code_ds_theta_scan_data = "XRD-THETA_DATA"
code_ds_laue_data = "XRAY-LAUE_DATA"


def create_xray_diffraction_schema(oBis: Openbis):
    """Creates X-ray Diffraction Entity Types.

        We create "empty" types so that data can be uploaded to the correct place.
        Once parsers are developed, we update the properties.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """
    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    terms_xray_source = [
        {"code": "AG", "label": "Silver", "description": "Silver"},
        {"code": "CO", "label": "Cobalt", "description": "Cobalt"},
        {"code": "CR", "label": "Chromium", "description": "Chromium"},
        {"code": "CU", "label": "Copper", "description": "Copper"},
        {"code": "FE", "label": "Iron", "description": "Iron"},
        {"code": "MO", "label": "Molybdenum", "description": "Molybdenum"},
    ]
    voc_xray_source = oBis.new_vocabulary(
        code="XRAY_SOURCE_VOCAB",
        description="X-ray Source Vocab",
        terms=terms_xray_source,
    )
    register_controlled_vocabulary(oBis, voc_xray_source, terms_xray_source)

    # #########################################################################
    # Property Types
    # #########################################################################

    pt_xray_source = oBis.new_property_type(
        code="XRAY_SOURCE",
        label="X-ray Source",
        description="X-ray Source / Anode / Target",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_xray_source.code,
    )
    register_property_type(oBis, pt_xray_source)

    # pt_CHANGEME = oBis.new_property_type(
    #     code = '',
    #     label = '',
    #     description = '',
    #     dataType = 'VARCHAR',
    # )
    # register_property_type(oBis, pt_CHANGEME)

    # #########################################################################
    # Objects
    # #########################################################################

    # code_obj_CHANGEME = ''
    # obj_CHANGEME = oBis.new_object_type(
    #     code = code_obj_CHANGEME,
    #     generatedCodePrefix = '',
    #     autoGeneratedCode = True,
    # )
    # register_object_type(oBis, obj_CHANGEME)

    # obj_CHANGEME = oBis.get_object_type(code_obj_CHANGEME)
    # obj_CHANGEME.description = ''
    # register_object_type(oBis, obj_CHANGEME)

    # obj_CHANGEME = oBis.get_object_type(code_obj_CHANGEME)

    # # Define display order
    # sec2props = {
    #     'General': [
    #         ('$NAME', 1, None), ('$SHOW_IN_PROJECT_OVERVIEW', 0, None),
    #         ('FINISHED_FLAG', 0, None), ('START_DATE', 0, None), ('END_DATE', 0, None),
    #         ('COMMENTS', 0, None),
    #     ],
    #     'Section Name': [
    #         ('PROPERTY_CODE', is_mandatory, plugin),
    #     ],
    # }
    # assign_property_types(obj_CHANGEME, sec2props)

    # #########################################################################
    # Dataset Types
    # #########################################################################

    # Texture Analysis

    ds_texture_data = oBis.new_dataset_type(
        code=code_ds_texture_data,
        description="Data from Texture Analysis (XRD - X-ray Diffraction)",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_texture_data)

    ds_texture_data = oBis.get_dataset_type(code_ds_texture_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_texture_data, sec2props)

    # Theta Scan

    ds_theta_scan_data = oBis.new_dataset_type(
        code=code_ds_theta_scan_data,
        description="Data from Theta Scan (XRD - X-ray Diffraction)",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_theta_scan_data)

    ds_theta_scan_data = oBis.get_dataset_type(code_ds_theta_scan_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_theta_scan_data, sec2props)

    # Laue Diffraction

    ds_laue_data = oBis.new_dataset_type(
        code=code_ds_laue_data,
        description="Data from Laue Diffraction",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_laue_data)

    ds_laue_data = oBis.get_dataset_type(code_ds_laue_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_laue_data, sec2props)
