#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.


    _summary_ : Create metadata schema for thin film synthesis

"""


from pybis import Openbis
from schema_helpers import *


# openBIS codes for the objects and datasets (also used in other modules)

code_obj_film_synthesis = "THIN_FILM_SYNTHESIS"


def create_thin_film_synthesis_schema(oBis: Openbis):
    """Creates Thin Film Synthesis Object Types.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """

    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    # Vacuum Level

    terms_vacuum = [
        {
            "code": "HV",
            "label": "High Vacuum - HV",
            "description": "High Vacuum",
        },
        {
            "code": "LV",
            "label": "Low Vacuum - LV",
            "description": "Low Vacuum",
        },
        {
            "code": "MV",
            "label": "Medium Vacuum - MV",
            "description": "Medium Vacuum",
        },
        {
            "code": "UHV",
            "label": "Ultra Vacuum - UHV",
            "description": "Ultra High Vacuum",
        },
        {
            "code": "XHV",
            "label": "Extreme  High Vacuum - XHV",
            "description": "Extreme High Vacuum",
        },
    ]
    voc_vacuum = oBis.new_vocabulary(
        code="VACUUM_TYPE_VOCAB",
        description="Type of vacuum",
        terms=terms_vacuum,
    )
    register_controlled_vocabulary(oBis, voc_vacuum, terms_vacuum)

    # #########################################################################
    # Property Types
    # #########################################################################

    # Vacuum Type

    pt_vacuum_type = oBis.new_property_type(
        code="DEPOSITION_VACUUM",
        label="Vacuum",
        description="Vacuum Type",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_vacuum.code,
    )
    register_property_type(oBis, pt_vacuum_type)

    # Deposition time
    pt_deposition_duration = oBis.new_property_type(
        code="DEPOSITION_DURATION",
        label="t dep (h)",
        description="Deposition time in hours",
        dataType="REAL",
    )
    register_property_type(oBis, pt_deposition_duration)

    # Deposition Temperature

    pt_deposition_temperature = oBis.new_property_type(
        code="DEPOSITION_TEMPERATURE",
        label="T dep (°C)",
        description="Deposition Temperature in °C",
        dataType="REAL",
    )
    register_property_type(oBis, pt_deposition_temperature)

    # Deposition Rate (nm/min)

    pt_deposition_rate = oBis.new_property_type(
        code="DEPOSITION_RATE",
        label="Deposition Rate (nm/min)",
        description="Deposition Rate (nm/min)",
        dataType="REAL",
    )
    register_property_type(oBis, pt_deposition_rate)

    # Base Pressure

    pt_base_pressure = oBis.new_property_type(
        code="BASE_PRESSURE_MBAR",
        label="P base (mbar)",
        description="Base pressure in mbar",
        dataType="REAL",
    )
    register_property_type(oBis, pt_base_pressure)

    # Sputtering Targets

    pt_sputtering_target = oBis.new_property_type(
        code="SPUTTERING_TARGET_STR",
        label="Target (s)",
        description="Comma-separated list of targets (e.g. Cu, Re, Ti)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_sputtering_target)

    # #########################################################################
    # Objects
    # #########################################################################

    # Experiment
    obj_film_synthesis = oBis.new_object_type(
        code=code_obj_film_synthesis,
        generatedCodePrefix="FILM_SYNTH",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_film_synthesis)

    obj_film_synthesis = oBis.get_object_type(code_obj_film_synthesis)
    obj_film_synthesis.description = "Thin Film Synthesis Experiment"
    register_object_type(oBis, obj_film_synthesis)

    obj_film_synthesis = oBis.get_object_type(code_obj_film_synthesis)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("COMMENTS", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Deposition Parameters": [
            ("SPUTTERING_TARGET_STR", 0, None),
            ("DEPOSITION_RATE", 0, None),
            ("DEPOSITION_VACUUM", 0, None),
            ("BASE_PRESSURE_MBAR", 0, None),
            ("DEPOSITION_DURATION", 0, None),
            ("DEPOSITION_TEMPERATURE", 0, None),
        ],
        "Experimental Details": [
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS", 0, None),
            ("EXPERIMENTAL_STEP.SPREADSHEET", 0, None),
        ],
    }
    assign_property_types(obj_film_synthesis, sec2props)
