#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""

    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for mechanical testing

"""

from pybis import Openbis
from schema_helpers import *


# openBIS codes for the objects and datasets (also used in other modules)

code_obj_mech_protocol = "MECH_TEST_PROTOCOL"
code_obj_mech_exp = "MECH_EXP"
code_ds_tensile_test_data = "TENS_TEST_DATA"
code_ds_compression_test_data = "COMP_TEST_DATA"
code_ds_bending_test_data = "BEND_TEST_DATA"


def create_tensile_schema(oBis: Openbis):
    """Creates Mechanical Testing Object Types and Dataset Types.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """
    # #########################################################################
    # Property Types
    # #########################################################################

    # Elongation at Fracture

    pt_tensile_elongation_fracture = oBis.new_property_type(
        code="TENSILE_ELONGATION_FRACTURE",
        label="Elongation at Fracture (%)",
        description="Elongation at Fracture (Fracture Strain) in %",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_elongation_fracture)

    # Yield Stress

    pt_tensile_yield_stress = oBis.new_property_type(
        code="TENSILE_YIELD_STRESS",
        label="Yield Stress (MPa)",
        description="Yield Stress [MPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_yield_stress)

    # Ultimate Tensile Stress

    pt_tensile_ultimate_stress = oBis.new_property_type(
        code="TENSILE_ULTIMATE_STRESS",
        label="Ultimate Tensile Stress (MPa)",
        description="Ultimate Tensile Stress [MPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_ultimate_stress)

    # Elastic Modulus

    pt_tensile_elastic_modulus = oBis.new_property_type(
        code="TENSILE_ELASTIC_MODULUS",
        label="Elastic Modulus (GPa)",
        description="Elastic Modulus [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_elastic_modulus)

    # Strain Rate

    pt_tensile_strain_rate = oBis.new_property_type(
        code="STRAIN_RATE_PER_S",
        label="Strain Rate (1/s)",
        description="Strain Rate [1/s]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_strain_rate)

    # Elongation

    pt_tensile_elongation = oBis.new_property_type(
        code="ELONGATION_PERCENT",
        label="Elongation (%)",
        description="Elongation / Strain in %",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_elongation)

    # TENSILE_P

    pt_tensile_p = oBis.new_property_type(
        code="TENSILE_P",
        label="Tensile P",
        description="Tensile P",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_p)

    # TENSILE_I

    pt_tensile_i = oBis.new_property_type(
        code="TENSILE_I",
        label="Tensile I",
        description="Tensile I",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_i)

    # TENSILE_D

    pt_tensile_d = oBis.new_property_type(
        code="TENSILE_D",
        label="Tensile D",
        description="Tensile D",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tensile_d)

    # Crosshead Speed

    pt_crosshead_speed = oBis.new_property_type(
        code="CROSSHEAD_SPEED_MM_PER_MIN",
        label="Crosshead Speed (mm/min)",
        description="Crosshead Speed (mm/min)",
        dataType="REAL",
    )
    register_property_type(oBis, pt_crosshead_speed)

    # #########################################################################
    # Objects
    # #########################################################################

    # You can reference a Mechanical Protocol as a parent of a Mechanical Experiment
    # Mechanical Testing Experiment could mean here (Tensile, Compression, Bending)

    # Mechanical experiment

    obj_mech_exp = oBis.new_object_type(
        code=code_obj_mech_exp,
        generatedCodePrefix="MECH_EXP",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_mech_exp)
    obj_mech_exp = oBis.get_object_type(code_obj_mech_exp)
    obj_mech_exp.description = "Mechanical Testing Experiment"
    register_object_type(oBis, obj_mech_exp)

    obj_mech_exp = oBis.get_object_type(code_obj_mech_exp)
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("COMMENTS", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Experimental Details": [
            ("DEFORMATION_MODE", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS", 0, None),
            ("TENSILE_ELONGATION_FRACTURE", 0, None),
            ("TENSILE_YIELD_STRESS", 0, None),
            ("TENSILE_ULTIMATE_STRESS", 0, None),
            ("TENSILE_ELASTIC_MODULUS", 0, None),
            ("TEMPERATURE", 0, None),
        ],
    }
    assign_property_types(obj_mech_exp, sec2props)

    # Mechanical experiment protocol

    obj_mech_protocol = oBis.new_object_type(
        code=code_obj_mech_protocol,
        generatedCodePrefix="MECH_PROT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_mech_protocol)
    obj_mech_protocol = oBis.get_object_type(code_obj_mech_protocol)
    obj_mech_protocol.description = "Protocol for mechanical testing"
    register_object_type(oBis, obj_mech_protocol)

    obj_mech_protocol = oBis.get_object_type(code_obj_mech_protocol)
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 1, None),
            ("DESCRIPTION", 0, None),
        ],
        "Experimental Details": [
            ("DEFORMATION_MODE", 0, None),
            ("ELONGATION_PERCENT", 0, None),
            ("STRAIN_RATE_PER_S", 0, None),
            ("TENSILE_P", 0, None),
            ("TENSILE_I", 0, None),
            ("TENSILE_D", 0, None),
            ("SAMPLING_RATE_HZ", 0, None),
            ("CROSSHEAD_SPEED_MM_PER_MIN", 0, None),
        ],
        "Comments": [
            ("COMMENTS", 0, None),
            # ("$XMLCOMMENTS", 0, None),
        ],
    }
    assign_property_types(obj_mech_protocol, sec2props)

    # #########################################################################
    # Dataset Types
    # #########################################################################

    ds_tens_test_data = oBis.new_dataset_type(
        code=code_ds_tensile_test_data,
        description="Data from Tensile Test Experiment",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_tens_test_data)

    ds_tens_test_data = oBis.get_dataset_type(code_ds_tensile_test_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_tens_test_data, sec2props)

    ds_comp_test_data = oBis.new_dataset_type(
        code=code_ds_compression_test_data,
        description="Data from Compression Test Experiment",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_comp_test_data)

    ds_comp_test_data = oBis.get_dataset_type(code_ds_compression_test_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_comp_test_data, sec2props)

    ds_bend_test_data = oBis.new_dataset_type(
        code=code_ds_bending_test_data,
        description="Data from Bending Test Experiment",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_bend_test_data)

    ds_bend_test_data = oBis.get_dataset_type(code_ds_bending_test_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin),
        # ],
    }
    assign_property_types(ds_bend_test_data, sec2props)
