#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2024 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for software

"""

from pybis import Openbis
from schema_helpers import *

# openBIS codes for the objects (also used in other modules)

code_obj_software = "SOFTWARE"


def create_software_schema(oBis: Openbis):
    """Creates Software Object Type.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """

    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    terms_prog_lang_type = [
        {"code": "C", "label": "C", "description": "C"},
        {"code": "CPLUSPLUS", "label": "C++", "description": "C++"},
        {
            "code": "CPLUSPLUS_FORTRAN",
            "label": "C++/Fortran",
            "description": "C++/Fortran",
        },
        {"code": "CSHARP", "label": "C#", "description": "C#"},
        {"code": "FORTRAN", "label": "Fortran", "description": "Fortran"},
        {"code": "JAVA", "label": "Java", "description": "Java"},
        {"code": "JAVASCRIPT", "label": "JavaScript", "description": "JavaScript"},
        {"code": "JULIA", "label": "Julia", "description": "Julia"},
        {"code": "MATLAB", "label": "Matlab", "description": "Matlab"},
        {"code": "PYTHON", "label": "Python", "description": "Python"},
        {"code": "R", "label": "R", "description": "R"},
        {"code": "RUST", "label": "Rust", "description": "Rust"},
    ]
    voc_prog_lang_type = oBis.new_vocabulary(
        code="PROG_LANG_VOCAB",
        description="Programming Languages",
        terms=terms_prog_lang_type,
    )
    register_controlled_vocabulary(
        oBis,
        voc_prog_lang_type,
        terms_prog_lang_type,
    )

    # #########################################################################
    # Property Types
    # #########################################################################

    # Software name

    pt_software_name = oBis.new_property_type(
        code="SOFTWARENAME",
        label="Software Name",
        description="Name of the software used for analysis or simulation",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_software_name)

    # Software Version

    pt_software_version = oBis.new_property_type(
        code="SOFTWAREVERSION",
        label="Software Version",
        description="Version of the software",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_software_version)

    # Release hash (e.g. if specific software version was used)

    pt_hash = oBis.new_property_type(
        code="COMMIT_HASH",
        label="Commit Hash",
        description="Commit hash from the version control system if no specific version number is available",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_hash)

    # Programming Language

    pt_prog_lang = oBis.new_property_type(
        code="PROG_LANG",
        label="Programming Language(s)",
        description="Programming Language(s)",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_prog_lang_type.code,
    )
    register_property_type(oBis, pt_prog_lang)

    # #########################################################################
    # Objects
    # #########################################################################

    obj_software = oBis.new_object_type(
        code=code_obj_software,
        generatedCodePrefix=code_obj_software,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_software)

    obj_software = oBis.get_object_type(code_obj_software)
    obj_software.description = (
        "Software used for simulation, experiment, analysis, etc."
    )
    register_object_type(oBis, obj_software)

    obj_software = oBis.get_object_type(code_obj_software)
    sec2pros_software = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("DESCRIPTION", 0, None),
            ("COMMENTS", 0, None),
        ],
        "Software Details": [
            ("SOFTWARENAME", 1, None),
            ("SOFTWAREVERSION", 0, None),
            ("COMMIT_HASH", 0, None),
            ("PROG_LANG", 0, None),
        ],
    }
    assign_property_types(obj_software, sec2pros_software)
