#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for atomistic and DFT simulations

"""


from pybis import Openbis
from schema_helpers import *


from .general_properties import MAX_NUM_ELEMENTS


# openBIS codes for the objects and datasets (also used in other modules)

code_obj_simulation_experiment = "SIMULATION_EXP"
code_obj_sim_artefact = "SIMULATION_ARTEFACT"
code_obj_interatomic_potential = "INTERATOMIC_POTENTIAL"
code_obj_pseudopotential = "PSEUDOPOTENTIAL"
code_obj_pyiron_generic = "PYIRON_JOB_GENERIC"
code_obj_pyiron_vasp = "PYIRON_JOB_VASP"
code_obj_pyiron_lammps = "PYIRON_JOB_LAMMPS"
code_obj_pyiron_murnaghan = "PYIRON_JOB_MURNAGHAN"
code_ds_crys_struct_data = "CRYS-STRUCT_DATA"
code_ds_sim_archive_data = "SIM_JOB"
code_ds_sim_artefact_data = "SIM_ARTEFACT"
code_ds_interatomic_potential_data = "INTERATOMIC_POTENTIAL_DATA"
code_ds_pyiron_concept_dict = "PYIRON_CONCEPT_DICT_DATA"
code_ds_pyiron_h5 = "PYIRON_HDF5"
code_ds_env_file = "COMP_ENV"


def create_simulation_schema(oBis: Openbis):
    """Creates Simulation-related Entity Types.

        We create "empty" types so that data can be uploaded to the correct place.
        Once parsers are developped, we update the properties.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """
    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    terms_file_source = [
        {"code": "ASE", "label": "ASE", "description": "file generated by ASE"},
        {
            "code": "ATOMSK",
            "label": "atomsk",
            "description": "file generated by atomsk",
        },
        {
            "code": "LAMMPS",
            "label": "lammps",
            "description": "file generated by lammps_write_data",
        },
        {
            "code": "OVITO",
            "label": "ovito",
            "description": "file generated by ovito software",
        },
        {
            "code": "PYIRON",
            "label": "pyiron",
            "description": "file generated by pyrion IDE",
        },
        {
            "code": "PYMATGEN",
            "label": "pymatgen",
            "description": "file generated by pymatgen",
        },
        {"code": "UNKNOWN", "label": "unknown", "description": "unknown"},
        {
            "code": "VESTA",
            "label": "vesta",
            "description": "file generated by VESTA software",
        },
    ]
    voc_file_source = oBis.new_vocabulary(
        code="FILE_SOURCE_VOCAB",
        description="Source",
        terms=terms_file_source,
    )
    register_controlled_vocabulary(oBis, voc_file_source, terms_file_source)

    terms_atom_kpoint_type_vocab = [
        {
            "code": "KPOINTS_MP",
            "label": "Monkhorst-Pack",
            "description": "Monkhorst-Pack method // Monkhorst-Pack Methode",
        },
        {
            "code": "KPOINTS_KSPACING",
            "label": "K-spacing",
            "description": "K-spacing method // K-spacing Methode",
        },
        {
            "code": "KPOINTS_MANUAL",
            "label": "Manually Defined",
            "description": "Manually defined // Manuell definiert",
        },
    ]
    voc_atom_kpoint_type_vocab = oBis.new_vocabulary(
        code="ATOM_KPOINT_TYPE_VOCAB",
        description="K-points specification type // K-Punkte-Spezifikation Typ",
        terms=terms_atom_kpoint_type_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_atom_kpoint_type_vocab, terms_atom_kpoint_type_vocab
    )

    terms_atom_potential_style_vocab = [
        {
            "code": "POT_STYLE_ACE",
            "label": "Atomic Cluster Expansion Potential",
            "description": "Atomic Cluster Expansion Potential",
        },
        {
            "code": "POT_STYLE_ADP",
            "label": "Angular Dependent Potential",
            "description": "Angular Dependent Potential",
        },
        {
            "code": "POT_STYLE_EAM",
            "label": "Embedded Atom Model Potential",
            "description": "Embedded atom model // Eingebettetes-Atom-Model",
        },
        {
            "code": "POT_STYLE_FF",
            "label": "Force Field Potential",
            "description": "Force field potential // Kraftfeld Potenzial",
        },
        {
            "code": "POT_STYLE_GRACE",
            "label": "Graph Atomic Cluster Expansion Potential",
            "description": "Graph Atomic Cluster Expansion Potential",
        },
        {
            "code": "POT_STYLE_LJ",
            "label": "Lennard-Jones Potential",
            "description": "Lennard-Jones potential // Lennard-Jones Potenzial",
        },
        {
            "code": "POT_STYLE_MEAM",
            "label": "Modified Embedded Atom Model Potential",
            "description": "Modifiead embedded atom model // Modifiziertes Eingebettetes-Atom-Model",
        },
        {
            "code": "POT_STYLE_MLIP",
            "label": "Machine Learning Potential",
            "description": "Machine learning potential // Machine-learning Potenzial",
        },
        {
            "code": "POT_STYLE_SW",
            "label": "Stillinger-Weber Potential",
            "description": "Stillinger-Weber potential // Stillinger-Weber Potenzial",
        },
        {
            "code": "POT_STYLE_UMLIP",
            "label": "Universal Machine Learning Potential",
            "description": "Universal machine learning potential // Universelles Machine-learning Potenzial",
        },
    ]
    voc_atom_potential_style_vocab = oBis.new_vocabulary(
        code="ATOM_POTENTIAL_STYLE_VOCAB",
        description="Interatomic Potential Style // Interatomares Potential Stil",
        terms=terms_atom_potential_style_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_atom_potential_style_vocab, terms_atom_potential_style_vocab
    )

    terms_atom_xc_functional_vocab = [
        {
            "code": "XC_FUNC_B3LYP",
            "label": "B3LYP",
            "description": "B3LYP functional // B3LYP Funktional",
        },
        {
            "code": "XC_FUNC_HSE03",
            "label": "HSE03",
            "description": "HSE03 functional // HSE03 Funktional",
        },
        {
            "code": "XC_FUNC_HSE06",
            "label": "HSE06",
            "description": "HSE06 functional // HSE06 Funktional",
        },
        {
            "code": "XC_FUNC_LDA",
            "label": "LDA",
            "description": "Local Density Approximation // Lokale Dichtenäherung",
        },
        {
            "code": "XC_FUNC_PBE",
            "label": "PBE",
            "description": "PBE-exchange-correlation functional // PBE-Austauschkorrelation Funktional",
        },
        {
            "code": "XC_FUNC_PBE_SOL",
            "label": "PBEsol",
            "description": "PBE functional revised for solids // PBE Funktional überarbeitet für Festkörper",
        },
        {
            "code": "XC_FUNC_SCAN0",
            "label": "SCAN0",
            "description": "SCAN0 functional // SCAN0 Funktional",
        },
    ]
    voc_atom_xc_functional_vocab = oBis.new_vocabulary(
        code="ATOM_XC_FUNCTIONAL_VOCAB",
        description="Exchange-correlation functional // Austausch-Korrelations-Funktional",
        terms=terms_atom_xc_functional_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_atom_xc_functional_vocab, terms_atom_xc_functional_vocab
    )

    terms_atomistic_calc_type_vocab = [
        {
            "code": "ATOM_CALC_MC",
            "label": "Monte-Carlo",
            "description": "Monte-Carlo calculation // Monte-Carlo Berechnung",
        },
        {
            "code": "ATOM_CALC_MD",
            "label": "Molecular Dynamics",
            "description": "Molecular dynamics (including ab initio) // Molekulardynamik (inklusive ab initio)",
        },
        {
            "code": "ATOM_CALC_NEB",
            "label": "NEB",
            "description": "Nudged Elastic Band method // Geschobene Elastische Band Methode",
        },
        {
            "code": "ATOM_CALC_SC",
            "label": "Self-consistent Field",
            "description": "Self-consistent field calculation // Selbstkonsistente Feldberechnung",
        },
        {
            "code": "ATOM_CALC_STRUC_OPT",
            "label": "Minimization/Structural-optimization",
            "description": "Minimization/structural-optimization // Minimierung/Strukturoptimierung",
        },
    ]
    voc_atomistic_calc_type_vocab = oBis.new_vocabulary(
        code="ATOMISTIC_CALC_TYPE_VOCAB",
        description="Type of calculation // Art der Berechnung",
        terms=terms_atomistic_calc_type_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_atomistic_calc_type_vocab, terms_atomistic_calc_type_vocab
    )

    terms_electronic_smearing_vocab = [
        {
            "code": "ELEC_SMEAR_FERMI",
            "label": "Fermi Smearing",
            "description": "Fermi smearing // Fermi-Schmiere",
        },
        {
            "code": "ELEC_SMEAR_FILE",
            "label": "Read from File",
            "description": "Read from file // Aus Datei gelesen",
        },
        {
            "code": "ELEC_SMEAR_GAUSS",
            "label": "Gaussian Smearing",
            "description": "Gaussian smearing // Gaussiani-Schmiere",
        },
        {
            "code": "ELEC_SMEAR_MP",
            "label": "Methfessel-Paxton Smearing",
            "description": "Methfessel-Paxton smearing // Methfessel-Paxton-Schmiere",
        },
        {
            "code": "ELEC_SMEAR_TET",
            "label": "Tetrahedron Method",
            "description": "Tetrahedron method // Tetrahedron-Verfahren",
        },
        {
            "code": "ELEC_SMEAR_TET_BL",
            "label": "Tetrahedron Method with Blöchl Corrections",
            "description": "Tetrahedron method with Blöchl corrections // Tetrahedron-Methode mit Blöchl-Korrekturen",
        },
    ]
    voc_electronic_smearing_vocab = oBis.new_vocabulary(
        code="ELECTRONIC_SMEARING_VOCAB",
        description="Partial occupancies // Teilbesetzungen",
        terms=terms_electronic_smearing_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_electronic_smearing_vocab, terms_electronic_smearing_vocab
    )

    terms_minimization_algo_vocab = [
        {
            "code": "MIN_ALGO_BLOCK_DVD",
            "label": "Blocked-Davidson",
            "description": "Blocked-Davidson iterative matrix diagonalization scheme // Iteratives Block-Davidson-Matrix-Diagonalisierungsverfahren",
        },
        {
            "code": "MIN_ALGO_CG",
            "label": "Conjugate Gradient",
            "description": "Conjugate Gradient // Konjugierte Gradientenverfahren",
        },
        {
            "code": "MIN_ALGO_DAMPED_MD",
            "label": "Simple Damped MD",
            "description": "Damped molecular dynamics // Gedämpfte Molekulardynamik",
        },
        {
            "code": "MIN_ALGO_EXCT_DIAG",
            "label": "Exact Diagonalization",
            "description": "Exact diagonalization // Exakte Diagonalisierung",
        },
        {
            "code": "MIN_ALGO_FIRE",
            "label": "FIRE",
            "description": "Fast inertial relaxation engine // Fast inertial relaxation engine",
        },
        {
            "code": "MIN_ALGO_HFTN",
            "label": "Hessian-free Truncated Newton Algorithm",
            "description": "Hessian-free truncated Newton algorithm // Hesse-freier abgeschnittener Newton-Algorithmus",
        },
        {
            "code": "MIN_ALGO_LBFGS",
            "label": "Limited-memory BFGS",
            "description": "Limited-memory Broyden-Fletcher-Goldfarb-Shanno (LBFGS) // Limited-memory Broyden-Fletcher-Goldfarb-Shanno (LBFGS)",
        },
        {
            "code": "MIN_ALGO_PREC_CG",
            "label": "Preconditioned Conjugate Gradient",
            "description": "Preconditioned conjugate gradient // Vorkonditioniertes konjugierte Gradientenverfahren",
        },
        {
            "code": "MIN_ALGO_RMM_DIIS",
            "label": "RMM-DIIS",
            "description": "Residual Minimization Method with Direct Inversion in the Iterative Subspace // Residuumminimierungsverfahren mit direkter Inversion im iterativen Unterraum",
        },
        {
            "code": "MIN_ALGO_STEEP_DESC",
            "label": "Steepest Descent",
            "description": "Steepest (Gradient) descent // Gradientenabstiegsverfahren",
        },
        {
            "code": "MIN_ALGO_QUICKMIN",
            "label": "Quickmin",
            "description": "Damped Molecular Dynamics according to Sheppard et al., J Chem Phys, 128, 134106 (2008) // Gedämpfte Molekulardynamik nach Sheppard et al., J Chem Phys, 128, 134106 (2008",
        },
        {
            "code": "MIN_ALGO_VASP_FAST",
            "label": "RMM-DIIS + Blocked-Davidson",
            "description": "Mixture of RMM-DIIS and Blocked Davidson (ALGO=FAST in VASP) // Mischung aus RMM-DIIS und Blocked Davidson (ALGO=FAST in VASP)",
        },
    ]
    voc_minimization_algo_vocab = oBis.new_vocabulary(
        code="MINIMIZATION_ALGO_VOCAB",
        description="Minimization Algorithm // Minimierungsalgorithmus",
        terms=terms_minimization_algo_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_minimization_algo_vocab, terms_minimization_algo_vocab
    )

    terms_thermodyn_ensemble_vocab = [
        {
            "code": "TD_ENSEMBLE_NVE",
            "label": "Microcanonical Ensemble",
            "description": "Microcanonical ensemble (NVE) // Mikrokanonisches Ensemble",
        },
        {
            "code": "TD_ENSEMBLE_GC",
            "label": "Grand Canonical Ensemble",
            "description": "Grand Canonical Ensemble (GC) // Großkanonisches Ensemble",
        },
        {
            "code": "TD_ENSEMBLE_ATOM_ENS_NVT",
            "label": "Canonical Ensemble",
            "description": "Canonical Ensemble (NVT) // Kanonisches Ensemble",
        },
        {
            "code": "TD_ENSEMBLE_NPH",
            "label": "Isoenthalpic-Isobaric Ensemble",
            "description": "Isoenthalpic-Isobaric Ensemble (NPH) // Isoenthalpisch-Isobarisches Ensemble",
        },
        {
            "code": "TD_ENSEMBLE_NPT",
            "label": "Isothermal-Isobaric Ensemble",
            "description": "Isothermal-Isobaric Ensemble (NPT) // Isothermisch-Isobarisches Ensemble",
        },
        {
            "code": "TD_ENSEMBLE_SGC",
            "label": "Semi-grand Canonical Ensemble",
            "description": "Semi-grand canonical Ensemble (SGC) // Semigroßes kanonisches Ensemble",
        },
    ]
    voc_thermodyn_ensemble_vocab = oBis.new_vocabulary(
        code="THERMODYN_ENSEMBLE_VOCAB",
        description="Atomistic Ensemble // Atomistische Ensemble",
        terms=terms_thermodyn_ensemble_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_thermodyn_ensemble_vocab, terms_thermodyn_ensemble_vocab
    )

    terms_murn_eqn_of_state_vocab = [
        {
            "code": "EOS_POLYNOMIAL",
            "label": "Polynomial",
            "description": "Polynomial // Polynom",
        },
        {
            "code": "EOS_MURNAGHAN",
            "label": "Murnaghan",
            "description": "Murnaghan equation // Murnaghan Gleichung",
        },
        {
            "code": "EOS_VINET",
            "label": "Vinet",
            "description": "Vinet equation // Vinet Gleichung",
        },
        {
            "code": "EOS_BIRCH_MURNAGHAN",
            "label": "Birch-Murnaghan",
            "description": "Birch-Murnaghan equation // Birch-Murnaghan Gleichung",
        },
    ]
    voc_murn_eqn_of_state_vocab = oBis.new_vocabulary(
        code="MURN_EQN_OF_STATE_VOCAB",
        description="Equation of state used for fit // Für das Fitting verwendete Zustandsgleichung",
        terms=terms_murn_eqn_of_state_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_murn_eqn_of_state_vocab, terms_murn_eqn_of_state_vocab
    )

    terms_pseudopot_functional_vocab = [
        {"code": "PSEUDOPOT_GGA", "label": "GGA", "description": "GGA"},
        {"code": "PSEUDOPOT_HYBRID", "label": "HYBRID", "description": "HYBRID"},
        {"code": "PSEUDOPOT_LDA", "label": "LDA", "description": "LDA"},
        {"code": "PSEUDOPOT_METAGGA", "label": "meta-GGA", "description": "meta-GGA"},
    ]
    voc_pseudopot_functional_vocab = oBis.new_vocabulary(
        code="PSEUDOPOT_FUNCTIONAL_VOCAB",
        description="Functional compatibility // Funktional-Kompatibilität",
        terms=terms_pseudopot_functional_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_pseudopot_functional_vocab, terms_pseudopot_functional_vocab
    )

    terms_pseudopot_type_vocab = [
        {
            "code": "PSEUDOPOT_AE",
            "label": "All-Electron (AE)",
            "description": "All-Electron",
        },
        {
            "code": "PSEUDOPOT_NC",
            "label": "Norm-conserving",
            "description": "Norm-conserving // Normerhaltend",
        },
        {
            "code": "PSEUDOPOT_PAW",
            "label": "Projector-augmented-wave method (PAW)",
            "description": "Projector-augmented-wave method (PAW) // Projektor-Augmented-Wave-Methode (PAW)",
        },
        {
            "code": "PSEUDOPOT_ULTRASOFT",
            "label": "Ultrasoft",
            "description": "Ultrasoft // Ultra-Weich",
        },
    ]
    voc_pseudopot_type_vocab = oBis.new_vocabulary(
        code="PSEUDOPOT_TYPE_VOCAB",
        description="Type of pseudopotential // Art des Pseudopotentials",
        terms=terms_pseudopot_type_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_pseudopot_type_vocab, terms_pseudopot_type_vocab
    )

    terms_lammps_unit = [
        {"code": "CGS", "label": "cgs", "description": "cgs"},
        {"code": "ELECTRON", "label": "electron", "description": "electron"},
        {"code": "LJ", "label": "lj", "description": "lj"},
        {"code": "METAL", "label": "metal", "description": "metal"},
        {"code": "MICRO", "label": "micro", "description": "micro"},
        {"code": "NANO", "label": "nano", "description": "nano"},
        {"code": "REAL", "label": "real", "description": "real"},
        {"code": "SI", "label": "si", "description": "si"},
    ]
    voc_lammps_unit = oBis.new_vocabulary(
        code="LAMMPS_UNITS_STYLE_VOCAB",
        description="LAMMPS units style",
        terms=terms_lammps_unit,
    )
    register_controlled_vocabulary(
        oBis,
        voc_lammps_unit,
        terms_lammps_unit,
    )

    terms_pyiron_job_type = [
        {"code": "CALPHY", "label": "Calphy", "description": "Calphy"},
        {
            "code": "ELASTICMATRIXJOB",
            "label": "ElasticMatrixJob",
            "description": "ElasticMatrixJob",
        },
        {
            "code": "PACEMAKERJOB",
            "label": "PacemakerJob",
            "description": "PacemakerJob",
        },
        {"code": "PHONOPYJOB", "label": "PhonopyJob", "description": "PhonopyJob"},
        {"code": "SPHINX", "label": "Sphinx", "description": "Sphinx"},
        {"code": "TABLEJOB", "label": "TableJob", "description": "TableJob"},
    ]
    voc_pyiron_job_type = oBis.new_vocabulary(
        code="PYIRON_JOB_TYPE_VOCAB",
        description="Pyiron Job Type Vocabulary",
        terms=terms_pyiron_job_type,
    )
    register_controlled_vocabulary(
        oBis,
        voc_pyiron_job_type,
        terms_pyiron_job_type,
    )

    # #########################################################################
    # Property Types
    # #########################################################################

    pt_file_source = oBis.new_property_type(
        code="FILE_SOURCE",
        label="File Source",
        description="Source of structure file",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_file_source.code,
    )
    register_property_type(oBis, pt_file_source)

    pt_number_of_atoms = oBis.new_property_type(
        code="NUMBER_OF_ATOMS",
        label="Number of Atoms",
        description="Total number of atoms contained in simulation box",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_atoms)

    pt_number_of_atom_types = oBis.new_property_type(
        code="NUMBER_OF_ATOM_TYPES",
        label="Number of Atom Types",
        description="Number of atom types contained in simulation box",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_atom_types)

    pt_list_of_atom_types = oBis.new_property_type(
        code="LIST_OF_ATOM_TYPES",
        label="List of Atom Types",
        description="List of atom types contained in simulation box",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_list_of_atom_types)

    pt_list_of_species = oBis.new_property_type(
        code="LIST_OF_SPECIES",
        label="List of Chemical Species",
        description="List of chemical species / elements, sorted alphabetically contained in simulation box",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_list_of_species)

    pt_length_a = oBis.new_property_type(
        code="BOX_LENGTH_A",
        label="Length A (Å)",
        description="Length of simulation box along the a axis in angstroms",
        dataType="REAL",
    )
    register_property_type(oBis, pt_length_a)

    pt_length_b = oBis.new_property_type(
        code="BOX_LENGTH_B",
        label="Length B (Å)",
        description="Length of simulation box along the b axis in angstroms",
        dataType="REAL",
    )
    register_property_type(oBis, pt_length_b)

    pt_length_c = oBis.new_property_type(
        code="BOX_LENGTH_C",
        label="Length C (Å)",
        description="Length of simulation box along the c axis in angstroms",
        dataType="REAL",
    )
    register_property_type(oBis, pt_length_c)

    pt_alpha = oBis.new_property_type(
        code="ANGLE_ALPHA",
        label="Alpha (deg)",
        description="Alpha angle (between b and c axes) in degrees",
        dataType="REAL",
    )
    register_property_type(oBis, pt_alpha)

    pt_beta = oBis.new_property_type(
        code="ANGLE_BETA",
        label="Beta (deg)",
        description="Beta angle (between a and c axes) in degrees",
        dataType="REAL",
    )
    register_property_type(oBis, pt_beta)

    pt_gamma = oBis.new_property_type(
        code="ANGLE_GAMMA",
        label="Gamma (deg)",
        description="Gamma angle (between a and b axes) in degrees",
        dataType="REAL",
    )
    register_property_type(oBis, pt_gamma)

    pt_atom_wavefunc_reuse = oBis.new_property_type(
        code="ATOM_WAVEFUNC_REUSE",
        label="Wavefunctions from a previous run?",
        description="Are the initial wavefunctions from a previous calculation? // Stammen die Anfangswellenfunktionen aus einer früheren Berechnung?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_wavefunc_reuse)

    pt_atom_chgdens_reuse = oBis.new_property_type(
        code="ATOM_CHGDENS_REUSE",
        label="Charge density from a previous run?",
        description="Are the initial charge densities from a previous calculation? // Stammen die Anfangsladungsdichten aus einer früheren Berechnung?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_chgdens_reuse)

    pt_atomistic_calc_type = oBis.new_property_type(
        code="ATOMISTIC_CALC_TYPE",
        label="Atomistic Calculation Type",
        description="Type of atomistic calculation // Art der atomistischen Berechnung",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_atomistic_calc_type_vocab.code,
    )
    register_property_type(oBis, pt_atomistic_calc_type)

    pt_atom_cell_vol_relax = oBis.new_property_type(
        code="ATOM_CELL_VOL_RELAX",
        label="Cell Volume Relaxation",
        description="Degrees of freedom - Cell volume relaxation // Freiheitsgrade - Zellvolumenrelaxation",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_cell_vol_relax)

    pt_atom_cell_shp_relax = oBis.new_property_type(
        code="ATOM_CELL_SHP_RELAX",
        label="Cell Shape Relaxation",
        description="Degrees of freedom - Cell shape relaxation // Freiheitsgrade - Zellformrelaxation",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_cell_shp_relax)

    pt_atom_pos_relax = oBis.new_property_type(
        code="ATOM_POS_RELAX",
        label="Atomic Position Relaxation",
        description="Degrees of freedom - Atomic position relaxation // Freiheitsgrade - Atomare Positionsrelaxation",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_pos_relax)

    pt_atom_xc_functional = oBis.new_property_type(
        code="ATOM_XC_FUNCTIONAL",
        label="XC functional",
        description="Exchange-correlation functional // Austausch-Korrelations-Funktional",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_atom_xc_functional_vocab.code,
    )
    register_property_type(oBis, pt_atom_xc_functional)

    pt_atom_xc_u_correction = oBis.new_property_type(
        code="ATOM_XC_U_CORRECTION",
        label="U Correction?",
        description="Are U corrections included? // Sind U-Korrekturen enthalten?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_xc_u_correction)

    pt_electronic_smearing = oBis.new_property_type(
        code="ELECTRONIC_SMEARING",
        label="Partial Occupancies",
        description="Partial occupancies // Teilbesetzungen",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_electronic_smearing_vocab.code,
    )
    register_property_type(oBis, pt_electronic_smearing)

    pt_atom_ionic_min_algo = oBis.new_property_type(
        code="ATOM_IONIC_MIN_ALGO",
        label="Minimization Algorithm for Ionic Steps",
        description="Minimization algorithm for ionic steps // Minimalisierungsalgorithmus zur ionischen Schritten",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_minimization_algo_vocab.code,
    )
    register_property_type(oBis, pt_atom_ionic_min_algo)

    pt_atom_elec_min_algo = oBis.new_property_type(
        code="ATOM_ELEC_MIN_ALGO",
        label="Minimization Algorithm for Electronic Steps",
        description="Minimization algorithm for electronic steps // Minimalisierungsalgorithmus zur elektronischen Schritten",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_minimization_algo_vocab.code,
    )
    register_property_type(oBis, pt_atom_elec_min_algo)

    pt_atom_spin_polarized = oBis.new_property_type(
        code="ATOM_SPIN_POLARIZED",
        label="Calculation Spin-polarized?",
        description="Is the calculation spin-polarized? // Ist die Berechnung spinpolarisiert?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_spin_polarized)

    pt_atom_non_coll_mag = oBis.new_property_type(
        code="ATOM_NON_COLL_MAG",
        label="Non-collinear Magnetism?",
        description="Are the magnetic moments non-collinear? // Sind die magnetischen Momente nicht kollinear?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_non_coll_mag)

    pt_atom_kpoint_type = oBis.new_property_type(
        code="ATOM_KPOINT_TYPE",
        label="K-points Specification Type",
        description="K-points specification type // K-Punkte-Spezifikation Typ",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_atom_kpoint_type_vocab.code,
    )
    register_property_type(oBis, pt_atom_kpoint_type)

    pt_atom_kpt_gamma_cent = oBis.new_property_type(
        code="ATOM_KPT_GAMMA_CENT",
        label="Gamma-centered?",
        description="Are the K-points centered around the gamma point? // Sind die k-Punkte um den Gamma-Punkt zentriert?",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_kpt_gamma_cent)

    pt_atomistic_n_kpt_x = oBis.new_property_type(
        code="ATOMISTIC_N_KPT_X",
        label="Number of K-points in x-direction",
        description="Number of K-points in x-direction // Anzahl der K-Punkte in x-Richtung",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_atomistic_n_kpt_x)

    pt_atomistic_n_kpt_y = oBis.new_property_type(
        code="ATOMISTIC_N_KPT_Y",
        label="Number of K-points in y-direction",
        description="Number of K-points in y-direction // Anzahl der K-Punkte in y-Richtung",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_atomistic_n_kpt_y)

    pt_atomistic_n_kpt_z = oBis.new_property_type(
        code="ATOMISTIC_N_KPT_Z",
        label="Number of K-points in z-direction",
        description="Number of K-points in z-direction // Anzahl der K-Punkte in z-Richtung",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_atomistic_n_kpt_z)

    pt_atomistic_kpt_spacin_in_1_a = oBis.new_property_type(
        code="ATOMISTIC_KPT_SPACIN_IN_1_A",
        label="K-spacing [1/Å]",
        description="K-spacing value [1/Å] // K-Abstandswert",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atomistic_kpt_spacin_in_1_a)

    pt_atomistic_kpt_full = oBis.new_property_type(
        code="ATOMISTIC_KPT_FULL",
        label="Full list of K-points",
        description="Full list of K-points // Vollständige Liste der K-Punkte",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_atomistic_kpt_full)

    pt_atom_e_tol_ion_in_ev = oBis.new_property_type(
        code="ATOM_E_TOL_ION_IN_EV",
        label="Ionic Energy Tolerance [eV]",
        description="Energy tolerance for ionic minimization [eV] // Energietoleranz zur ionische Minimierung [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_e_tol_ion_in_ev)

    pt_atom_el_e_tol_in_ev = oBis.new_property_type(
        code="ATOM_EL_E_TOL_IN_EV",
        label="Electronic Energy Tolerance [eV]",
        description="Energy tolerance for electronic minimization [eV] // Energietoleranz zur elektronische Minimierung [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_el_e_tol_in_ev)

    pt_atom_f_tol_in_ev_a = oBis.new_property_type(
        code="ATOM_F_TOL_IN_EV_A",
        label="Ionic Force Tolerance [eV/Å]",
        description="Force tolerance for minimization [eV/Å] // Krafttoleranz für Minimierung [eV/Å]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_f_tol_in_ev_a)

    pt_atom_e_cutoff_in_ev = oBis.new_property_type(
        code="ATOM_E_CUTOFF_IN_EV",
        label="Energy Cutoff [eV]",
        description="Energy cutoff for wavefunctions [eV] // Energiegrenzwert für Wellenfunktionen [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_e_cutoff_in_ev)

    pt_atom_sigma_in_ev = oBis.new_property_type(
        code="ATOM_SIGMA_IN_EV",
        label="Sigma Value [eV]",
        description="Sigma value [eV] // Sigma-Wert [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_sigma_in_ev)

    pt_atom_fin_tot_eng_in_ev = oBis.new_property_type(
        code="ATOM_FIN_TOT_ENG_IN_EV",
        label="Final Total Energy [eV]",
        description="Final Total Energy [eV] // Letzte Gesamtenergie [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_fin_tot_eng_in_ev)

    pt_atom_fin_pot_eng_in_ev = oBis.new_property_type(
        code="ATOM_FIN_POT_ENG_IN_EV",
        label="Final Potential Energy [eV]",
        description="Final Potential Energy [eV] // Letzte potenzielle Energie [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_fin_pot_eng_in_ev)

    pt_atom_fin_press_in_gpa = oBis.new_property_type(
        code="ATOM_FIN_PRESS_IN_GPA",
        label="Final Pressure [GPa]",
        description="Final pressure [GPa] // Letzter Druck [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_fin_press_in_gpa)

    pt_atom_fin_vol_in_a3 = oBis.new_property_type(
        code="ATOM_FIN_VOL_IN_A3",
        label="Final Volume [Å^3]",
        description="Final Volume [Å^3] // Letztes Volumen [Å^3]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_fin_vol_in_a3)

    pt_atom_force_max_in_ev_a = oBis.new_property_type(
        code="ATOM_FORCE_MAX_IN_EV_A",
        label="Final Maximum Force Component [eV/Å]",
        description="Final maximum force component [eV/Å] // Letzte maximale Kraftkomponente [eV/Å]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_force_max_in_ev_a)

    pt_atom_fin_totmgmo_in_mub = oBis.new_property_type(
        code="ATOM_FIN_TOTMGMO_IN_MUB",
        label="Final Total Magnetic Moment [μ_B]",
        description="Final total magnetic moment [μ_B] // Letztes magnetisches Gesamtmoment [μ_B]",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_atom_fin_totmgmo_in_mub)

    pt_notes = oBis.new_property_type(
        code="NOTES",
        label="Notes",
        description="Notes // Notizen",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_notes)

    pt_periodic_boundary_x = oBis.new_property_type(
        code="PERIODIC_BOUNDARY_X",
        label="Simulation Periodicity in X-Direction",
        description="Simulation periodicity in X-direction // Periodizität der Simulation in X-Richtung",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_periodic_boundary_x)

    pt_periodic_boundary_y = oBis.new_property_type(
        code="PERIODIC_BOUNDARY_Y",
        label="Simulation Periodicity in Y-Direction",
        description="Simulation periodicity in Y-direction // Periodizität der Simulation in Y-Richtung",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_periodic_boundary_y)

    pt_periodic_boundary_z = oBis.new_property_type(
        code="PERIODIC_BOUNDARY_Z",
        label="Simulation Periodicity in Z-Direction",
        description="Simulation periodicity in Z-direction // Periodizität der Simulation in Z-Richtung",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_periodic_boundary_z)

    pt_atom_md_ensemble = oBis.new_property_type(
        code="ATOM_MD_ENSEMBLE",
        label="Statistical Ensemble",
        description="Statistical ensemble set in the simulation // Statistisches Ensemble in der Simulation",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_thermodyn_ensemble_vocab.code,
    )
    register_property_type(oBis, pt_atom_md_ensemble)

    pt_atom_md_time_stp_in_ps = oBis.new_property_type(
        code="ATOM_MD_TIME_STP_IN_PS",
        label="Time Step Size [ps]",
        description="Time step size [ps] // Zeitschrittweite [ps]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_md_time_stp_in_ps)

    pt_atom_sim_time_ps_in_ps = oBis.new_property_type(
        code="ATOM_SIM_TIME_PS_IN_PS",
        label="Simulation Time [ps]",
        description="Simulated timespan [ps] //  Simulierte Zeitspanne [ps]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_sim_time_ps_in_ps)

    pt_atom_md_langevin = oBis.new_property_type(
        code="ATOM_MD_LANGEVIN",
        label="Langevin Dynamics",
        description="Use of Langevin dynamics // Verwendung der Langevin-Dynamik",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_atom_md_langevin)

    pt_max_iters = oBis.new_property_type(
        code="MAX_ITERS",
        label="Maximum Iterations",
        description="Maximum number of iterations // Maximale Anzahl von Iterationen",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_max_iters)

    pt_atom_md_init_temp_in_k = oBis.new_property_type(
        code="ATOM_MD_INIT_TEMP_IN_K",
        label="Initial Temperature [K]",
        description="Initial temperature [K] // Anfangstemperatur [K]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_md_init_temp_in_k)

    pt_atom_md_init_press_in_gpa = oBis.new_property_type(
        code="ATOM_MD_INIT_PRESS_IN_GPA",
        label="Initial Pressure [GPa]",
        description="Initial pressure [GPa] // Anfangsdruck [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_md_init_press_in_gpa)

    pt_atom_md_targ_temp_in_k = oBis.new_property_type(
        code="ATOM_MD_TARG_TEMP_IN_K",
        label="Target Temperature [K]",
        description="Target temperature [K] // Zieltemperatur [K]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_md_targ_temp_in_k)

    pt_atom_targ_press_in_gpa = oBis.new_property_type(
        code="ATOM_TARG_PRESS_IN_GPA",
        label="Target Pressure [GPa]",
        description="Target pressure [GPa] // Ziel-Druck [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_targ_press_in_gpa)

    pt_atom_ionic_steps = oBis.new_property_type(
        code="ATOM_IONIC_STEPS",
        label="N Ionic Steps",
        description="Number of ionic steps // Anzahl der ionischen Schritten",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_atom_ionic_steps)

    pt_atom_md_avg_temp_in_k = oBis.new_property_type(
        code="ATOM_MD_AVG_TEMP_IN_K",
        label="Average Temperature [K]",
        description="Average temperature over time steps [K] // Durchschnittstemperatur [K]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_md_avg_temp_in_k)

    pt_atom_avg_press_in_gpa = oBis.new_property_type(
        code="ATOM_AVG_PRESS_IN_GPA",
        label="Average Pressure [GPa]",
        description="Average pressure over time steps [GPa] // Durchschnittsdruck [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_avg_press_in_gpa)

    pt_atom_avg_tot_eng_in_ev = oBis.new_property_type(
        code="ATOM_AVG_TOT_ENG_IN_EV",
        label="Average Total Energy [eV]",
        description="Average Total Energy over time steps [eV] // Durchschnittsgesamtenergie [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_avg_tot_eng_in_ev)

    pt_atom_avg_vol_in_a3 = oBis.new_property_type(
        code="ATOM_AVG_VOL_IN_A3",
        label="Average Volume [Å^3]",
        description="Average Volume over time steps [Å^3] // Durchschnittliches Volumen [Å^3]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_avg_vol_in_a3)

    pt_atom_avg_pot_eng_in_ev = oBis.new_property_type(
        code="ATOM_AVG_POT_ENG_IN_EV",
        label="Average Potential Energy [eV]",
        description="Average Potential Energy over time steps (eV) // Durchschnittliche potenzielle Energie [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_avg_pot_eng_in_ev)

    pt_atom_fin_fnorm_in_ev_a = oBis.new_property_type(
        code="ATOM_FIN_FNORM_IN_EV_A",
        label="Final Force Norm [eV/Å]",
        description="Final Force norm [eV/Å] // Letzte Kraftnorm [eV/Å]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_fin_fnorm_in_ev_a)

    pt_atom_avg_fnorm_in_ev_a = oBis.new_property_type(
        code="ATOM_AVG_FNORM_IN_EV_A",
        label="Average Force Norm [eV/Å]",
        description="Average Force norm over time steps [eV/Å] // Durchschnittskraftnorm [eV/Å]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_avg_fnorm_in_ev_a)

    pt_murn_eqn_of_state = oBis.new_property_type(
        code="MURN_EQN_OF_STATE",
        label="Equation of State",
        description="Equation of state used for fit // Für das Fitting verwendete Zustandsgleichung",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_murn_eqn_of_state_vocab.code,
    )
    register_property_type(oBis, pt_murn_eqn_of_state)

    pt_murn_fit_eqn_order = oBis.new_property_type(
        code="MURN_FIT_EQN_ORDER",
        label="Fit Order (if Polynomial)",
        description="Fit order (if polynomial) // Grad des Polynoms",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_murn_fit_eqn_order)

    pt_murn_strain_axes = oBis.new_property_type(
        code="MURN_STRAIN_AXES",
        label="Strain Axes",
        description="Axes along which cell is strained // Achsen, entlang derer die Zelle belastet wird",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_murn_strain_axes)

    pt_murn_n_data_points = oBis.new_property_type(
        code="MURN_N_DATA_POINTS",
        label="Number of Data Points",
        description="Number of data points // Anzahl der Datenpunkte",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_murn_n_data_points)

    pt_murn_strainvol_range = oBis.new_property_type(
        code="MURN_STRAINVOL_RANGE",
        label="Volume Range (Fractional)",
        description="Volume range (fractional) // Volumenbereich (fraktioniert)",
        dataType="REAL",
    )
    register_property_type(oBis, pt_murn_strainvol_range)

    pt_atom_equil_k_mod_in_gpa = oBis.new_property_type(
        code="ATOM_EQUIL_K_MOD_IN_GPA",
        label="Equilibrium Bulk Modulus [GPa]",
        description="Equilibrium bulk modulus [GPa] // Kompressionsmodul im  Gleichgewicht [GPa]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_equil_k_mod_in_gpa)

    pt_atom_equil_toteng_in_ev = oBis.new_property_type(
        code="ATOM_EQUIL_TOTENG_IN_EV",
        label="Equilibrium Total Energy [eV]",
        description="Equilibrium total energy [eV] // Gesamtenergie im Gleichgewicht [eV]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_equil_toteng_in_ev)

    pt_atom_equil_vol_in_a3 = oBis.new_property_type(
        code="ATOM_EQUIL_VOL_IN_A3",
        label="Equilibrium Volume [Å^3]",
        description="Equilibrium volume [Å^3] // Volumen im Gleichgewicht [Å^3]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_atom_equil_vol_in_a3)

    pt_chem_species_addressed = oBis.new_property_type(
        code="CHEM_SPECIES_ADDRESSED",
        label="Chemical Species Addressed",
        description="Chemical species addressed // Angesprochene chemische Arten",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_chem_species_addressed)

    pt_atom_potential_style = oBis.new_property_type(
        code="ATOM_POTENTIAL_STYLE",
        label="Interatomic Potential Style",
        description="Interatomic Potential Style // Interatomarer Potential Stil",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_atom_potential_style_vocab.code,
    )
    register_property_type(oBis, pt_atom_potential_style)

    pt_source_link = oBis.new_property_type(
        code="SOURCE_LINK",
        label="Source for download",
        description="Source/Download // Quelle/Herunterladen",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_source_link)

    pt_software_compatibility = oBis.new_property_type(
        code="SOFTWARE_COMPATIBILITY",
        label="Software Compatibility",
        description="Software which can use this file // Software, die diese Datei verwenden kann",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_software_compatibility)

    pt_reference = oBis.new_property_type(
        code="REFERENCE",
        label="References",
        description="Useful references // Nützliche Referenzen",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_reference)

    pt_version = oBis.new_property_type(
        code="PSEUDOPOT_VERSION",
        label="Version",
        description="Pseudopotential Version // Version",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_version)

    pt_license = oBis.new_property_type(
        code="PSEUDOPOT_LICENSE",
        label="License",
        description="Pseudopotential License // Lizenz",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_license)

    pt_pseudopot_type = oBis.new_property_type(
        code="PSEUDOPOT_TYPE",
        label="Type of Pseudopotential",
        description="Type of pseudopotential // Art des Pseudopotentials",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_pseudopot_type_vocab.code,
    )
    register_property_type(oBis, pt_pseudopot_type)

    pt_pseudopot_func = oBis.new_property_type(
        code="PSEUDOPOT_FUNC",
        label="Functional Compatibility",
        description="Functional compatibility // Funktional-Kompatibilität",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_pseudopot_functional_vocab.code,
    )
    register_property_type(oBis, pt_pseudopot_func)

    pt_pseudopot_semicore = oBis.new_property_type(
        code="PSEUDOPOT_SEMICORE",
        label="Semicore Shells Considered as Valence",
        description="Semicore shells considered as valence // Halbkernschalen, die als Valenz betrachtet werden",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_pseudopot_semicore)

    pt_external_identifier = oBis.new_property_type(
        code="EXTERNAL_IDENTIFIER",
        label="Id",
        description="External Identifier",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_external_identifier)

    pt_sim_job_finished = oBis.new_property_type(
        code="SIM_JOB_FINISHED",
        label="Is the job finished?",
        description="Finished = True, Aborted or incomplete = False // Beendet = Wahr, Abgebrochen oder unvollständig = Falsch",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_sim_job_finished)

    pt_sim_walltime_in_hours = oBis.new_property_type(
        code="SIM_WALLTIME_IN_HOURS",
        label="Job Run Time (Walltime) [hr]",
        description="Total job run time [hr] // Gesamtlaufzeit des Jobs [Stunden]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sim_walltime_in_hours)

    pt_sim_coretime_in_hours = oBis.new_property_type(
        code="SIM_CORETIME_IN_HOURS",
        label="Total Job Core Time [hr]",
        description="Total core hours used [hr] // Gesamtkernstundenzeit des Jobs [Stunden]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sim_coretime_in_hours)

    pt_job_type = oBis.new_property_type(
        code="PYIRON_JOB_TYPE",
        label="Job Type",
        description="Pyiron Job Type",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_pyiron_job_type.code,
    )
    register_property_type(oBis, pt_job_type)

    pt_table_preview = oBis.new_property_type(
        code="PYIRON_TABLE_PREVIEW",
        label="Preview",
        description="Preview",
        dataType="XML",
    )
    register_property_type(oBis, pt_table_preview)

    pt_lammps_unit_style = oBis.new_property_type(
        code="LAMMPS_UNIT_STYLE",
        label="Unit Style",
        description="Lammps Unit Style",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_lammps_unit.code,
    )
    register_property_type(oBis, pt_lammps_unit_style)

    pt_dwf = oBis.new_property_type(
        code="DWF",
        label="Debye–Waller factor",
        description="Debye–Waller factor (comma-seperated)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_dwf)

    # #########################################################################
    # Objects
    # #########################################################################

    # Simulation run / experiment
    obj_sim_exp = oBis.new_object_type(
        code=code_obj_simulation_experiment,
        generatedCodePrefix="SIM",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_sim_exp)

    obj_sim_exp = oBis.get_object_type(code_obj_simulation_experiment)
    obj_sim_exp.description = "Simulation job / run / experiment"
    register_object_type(oBis, obj_sim_exp)

    obj_sim_exp = oBis.get_object_type(code_obj_simulation_experiment)

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props_sim_exp = {
        "General": [
            ("$NAME", 1, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("COMMENTS", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Simulation Details": [
            ("SIMULATION_TYPE", 1, None),
            ("EBSDSIM_TYPE", 0, None),
            ("ATOMSIM_TYPE", 0, None),
            ("CLUSTER_JOBID", 0, None),
            ("NCORES", 0, None),
            ("NGPUS", 0, None),
            ("NNODES", 0, None),
            ("JOB_RUN_TIME_SEC", 0, None),
            ("JOB_REQ_RUN_TIME", 0, None),
        ],
        "DFT-specific": [
            ("ATOMISTIC_N_KPT_X", 0, None),
            ("ATOMISTIC_N_KPT_Y", 0, None),
            ("ATOMISTIC_N_KPT_Z", 0, None),
            ("ATOM_KPOINT_TYPE", 0, None),
            ("ATOM_E_CUTOFF_IN_EV", 0, None),
            ("ATOM_SIGMA_IN_EV", 0, None),
            ("ATOM_FIN_TOTMGMO_IN_MUB", 0, None),
        ],
        "Output": [
            ("ATOM_FIN_TOT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_POT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_VOL_IN_A3", 0, None),
        ],
        "Experimental Details": [
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS", 0, None),
        ],
    }
    assign_property_types(obj_sim_exp, sec2props_sim_exp)

    obj_sim_artefact = oBis.new_object_type(
        code=code_obj_sim_artefact,
        generatedCodePrefix="SIM_ARTEFACT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_sim_artefact)

    obj_sim_artefact = oBis.get_object_type(code_obj_sim_artefact)
    obj_sim_artefact.description = "Simulation artefact (excluding structure)"
    register_object_type(oBis, obj_sim_artefact)

    obj_sim_artefact = oBis.get_object_type(code_obj_sim_artefact)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("COMMENTS", 0, None),
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(obj_sim_artefact, sec2props)

    # Define Interatomic Potential Object Type
    obj_interatomic_potential = oBis.new_object_type(
        code=code_obj_interatomic_potential,
        generatedCodePrefix=code_obj_interatomic_potential,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_interatomic_potential)

    obj_interatomic_potential = oBis.get_object_type(code_obj_interatomic_potential)
    obj_interatomic_potential.description = "Interatomic Potential"
    register_object_type(oBis, obj_interatomic_potential)

    obj_interatomic_potential = oBis.get_object_type(code_obj_interatomic_potential)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Material Information": [
            ("CHEM_SPECIES_ADDRESSED", 0, None),
            ("NUMBER_OF_SPECIES", 0, "MATERIAL_SPECIES"),
        ],
        "Method-specific": [
            ("ATOM_POTENTIAL_STYLE", 0, None),
        ],
        "Technical Information": [
            ("EXTERNAL_IDENTIFIER", 0, None),
            ("SOURCE_LINK", 0, None),
            ("SOFTWARE_COMPATIBILITY", 0, None),
        ],
        "External Documentation": [
            ("REFERENCE", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_interatomic_potential, sec2props)

    # Define Pseudo Potential Object Type
    obj_pseudopotential = oBis.new_object_type(
        code=code_obj_pseudopotential,
        generatedCodePrefix=code_obj_pseudopotential,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_pseudopotential)

    obj_pseudopotential = oBis.get_object_type(code_obj_pseudopotential)
    obj_pseudopotential.description = (
        "Pseudopotential for electronic structure simulations"
    )
    register_object_type(oBis, obj_pseudopotential)

    obj_pseudopotential = oBis.get_object_type(code_obj_pseudopotential)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("PSEUDOPOT_VERSION", 0, None),
            ("PSEUDOPOT_LICENSE", 0, None),
            ("NUM_DATASETS", 0, None),
        ],
        "Material Information": [
            ("CHEM_SPECIES_ADDRESSED", 0, None),
        ],
        "Method-specific": [
            ("PSEUDOPOT_TYPE", 0, None),
            ("PSEUDOPOT_FUNC", 0, None),
        ],
        "Technical Information": [
            ("SOFTWARE_COMPATIBILITY", 0, None),
            ("EXTERNAL_IDENTIFIER", 0, None),
            ("PSEUDOPOT_SEMICORE", 0, None),
        ],
        "External Documentation": [
            ("REFERENCE", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_pseudopotential, sec2props)

    # Define Pyiron Generic Job
    obj_pyiron_generic = oBis.new_object_type(
        code=code_obj_pyiron_generic,
        generatedCodePrefix=code_obj_pyiron_generic,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_pyiron_generic)

    obj_pyiron_generic = oBis.get_object_type(code_obj_pyiron_generic)
    obj_pyiron_generic.description = "Generic pyiron job"
    register_object_type(oBis, obj_pyiron_generic)

    obj_pyiron_generic = oBis.get_object_type(code_obj_pyiron_generic)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("USER_NAME", 0, None),
            ("NUM_DATASETS", 0, None),
        ],
        "Technical Information": [
            ("WORKFLOW_MANAGER", 0, None),
        ],
        "Pyiron Information": [
            ("SIM_JOB_FINISHED", 0, None),
            ("PYIRON_JOB_TYPE", 0, None),
            ("START_DATE", 0, None),
            ("SIM_WALLTIME_IN_HOURS", 0, None),
            ("SIM_CORETIME_IN_HOURS", 0, None),
            ("NCORES", 0, None),
            ("NGPUS", 0, None),
            ("NTHREADS", 0, None),
            ("HPC_JOB_QUEUE", 0, None),
            ("HPC_JOB_ID", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
            ("PYIRON_TABLE_PREVIEW", 0, None),
        ],
        "Annotations": [
            ("PYIRON_CONCEPTUAL_DICTIONARY", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_pyiron_generic, sec2props)

    # Define Pyiron VASP Job
    obj_pyiron_vasp = oBis.new_object_type(
        code=code_obj_pyiron_vasp,
        generatedCodePrefix=code_obj_pyiron_vasp,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_pyiron_vasp)

    obj_pyiron_vasp = oBis.get_object_type(code_obj_pyiron_vasp)
    obj_pyiron_vasp.description = "VASP pyiron job"
    register_object_type(oBis, obj_pyiron_vasp)

    obj_pyiron_vasp = oBis.get_object_type(code_obj_pyiron_vasp)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("USER_NAME", 0, None),
            ("NUM_DATASETS", 0, None),
        ],
        "Technical Information": [
            ("WORKFLOW_MANAGER", 0, None),
        ],
        "Pyiron Information": [
            ("SIM_JOB_FINISHED", 0, None),
            ("START_DATE", 0, None),
            ("SIM_WALLTIME_IN_HOURS", 0, None),
            ("SIM_CORETIME_IN_HOURS", 0, None),
            ("NCORES", 0, None),
            ("NGPUS", 0, None),
            ("NTHREADS", 0, None),
            ("HPC_JOB_QUEUE", 0, None),
            ("HPC_JOB_ID", 0, None),
        ],
        "Method-specific": [
            ("ATOM_WAVEFUNC_REUSE", 0, None),
            ("ATOM_CHGDENS_REUSE", 0, None),
            ("ATOMISTIC_CALC_TYPE", 0, None),
            ("ATOM_CELL_VOL_RELAX", 0, None),
            ("ATOM_CELL_SHP_RELAX", 0, None),
            ("ATOM_POS_RELAX", 0, None),
            ("ATOM_XC_FUNCTIONAL", 0, None),
            ("ATOM_XC_U_CORRECTION", 0, None),
            ("ELECTRONIC_SMEARING", 0, None),
            ("ATOM_IONIC_MIN_ALGO", 0, None),
            ("ATOM_ELEC_MIN_ALGO", 0, None),
            ("ATOM_SPIN_POLARIZED", 0, None),
            ("ATOM_NON_COLL_MAG", 0, None),
            ("ATOM_KPOINT_TYPE", 0, None),
            ("ATOM_KPT_GAMMA_CENT", 0, None),
        ],
        "Input": [
            ("ATOMISTIC_N_KPT_X", 0, None),
            ("ATOMISTIC_N_KPT_Y", 0, None),
            ("ATOMISTIC_N_KPT_Z", 0, None),
            ("ATOMISTIC_KPT_SPACIN_IN_1_A", 0, None),
            ("ATOMISTIC_KPT_FULL", 0, None),
            ("ATOM_E_TOL_ION_IN_EV", 0, None),
            ("ATOM_EL_E_TOL_IN_EV", 0, None),
            ("ATOM_F_TOL_IN_EV_A", 0, None),
            ("ATOM_E_CUTOFF_IN_EV", 0, None),
            ("ATOM_SIGMA_IN_EV", 0, None),
        ],
        "Output": [
            ("ATOM_IONIC_STEPS", 0, None),
            ("ATOM_FIN_TOT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_POT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_PRESS_IN_GPA", 0, None),
            ("ATOM_FIN_VOL_IN_A3", 0, None),
            ("ATOM_FORCE_MAX_IN_EV_A", 0, None),
            ("ATOM_FIN_TOTMGMO_IN_MUB", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Annotations": [
            ("PYIRON_CONCEPTUAL_DICTIONARY", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_pyiron_vasp, sec2props)

    # Define Pyiron LAMMPS Job
    obj_pyiron_lammps = oBis.new_object_type(
        code=code_obj_pyiron_lammps,
        generatedCodePrefix=code_obj_pyiron_lammps,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_pyiron_lammps)

    obj_pyiron_lammps = oBis.get_object_type(code_obj_pyiron_lammps)
    obj_pyiron_lammps.description = "LAMMPS pyiron job"
    register_object_type(oBis, obj_pyiron_lammps)

    obj_pyiron_lammps = oBis.get_object_type(code_obj_pyiron_lammps)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("USER_NAME", 0, None),
            ("NUM_DATASETS", 0, None),
        ],
        "Technical Information": [
            ("WORKFLOW_MANAGER", 0, None),
        ],
        "Pyiron Information": [
            ("SIM_JOB_FINISHED", 0, None),
            ("START_DATE", 0, None),
            ("SIM_WALLTIME_IN_HOURS", 0, None),
            ("SIM_CORETIME_IN_HOURS", 0, None),
            ("NCORES", 0, None),
            ("NGPUS", 0, None),
            ("NTHREADS", 0, None),
            ("HPC_JOB_QUEUE", 0, None),
            ("HPC_JOB_ID", 0, None),
        ],
        "Method-specific": [
            ("ATOMISTIC_CALC_TYPE", 0, None),
            ("PERIODIC_BOUNDARY_X", 0, None),
            ("PERIODIC_BOUNDARY_Y", 0, None),
            ("PERIODIC_BOUNDARY_Z", 0, None),
            ("ATOM_CELL_VOL_RELAX", 0, None),
            ("ATOM_CELL_SHP_RELAX", 0, None),
            ("ATOM_POS_RELAX", 0, None),
            ("ATOM_MD_ENSEMBLE", 0, None),
            ("ATOM_IONIC_MIN_ALGO", 0, None),
        ],
        "Input": [
            ("ATOM_MD_TIME_STP_IN_PS", 0, None),
            ("ATOM_SIM_TIME_PS_IN_PS", 0, None),
            ("ATOM_MD_LANGEVIN", 0, None),
            ("MAX_ITERS", 0, None),
            ("ATOM_E_TOL_ION_IN_EV", 0, None),
            ("ATOM_F_TOL_IN_EV_A", 0, None),
            ("ATOM_MD_INIT_TEMP_IN_K", 0, None),
            ("ATOM_MD_INIT_PRESS_IN_GPA", 0, None),
            ("ATOM_MD_TARG_TEMP_IN_K", 0, None),
            ("ATOM_TARG_PRESS_IN_GPA", 0, None),
        ],
        "Output": [
            ("ATOM_IONIC_STEPS", 0, None),
            ("ATOM_MD_AVG_TEMP_IN_K", 0, None),
            ("ATOM_AVG_PRESS_IN_GPA", 0, None),
            ("ATOM_FIN_TOT_ENG_IN_EV", 0, None),
            ("ATOM_AVG_TOT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_VOL_IN_A3", 0, None),
            ("ATOM_AVG_VOL_IN_A3", 0, None),
            ("ATOM_FIN_POT_ENG_IN_EV", 0, None),
            ("ATOM_AVG_POT_ENG_IN_EV", 0, None),
            ("ATOM_FIN_FNORM_IN_EV_A", 0, None),
            ("ATOM_AVG_FNORM_IN_EV_A", 0, None),
            ("ATOM_FORCE_MAX_IN_EV_A", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Annotations": [
            ("PYIRON_CONCEPTUAL_DICTIONARY", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_pyiron_lammps, sec2props)

    # Define Pyiron MURNAGHAN Job
    obj_pyiron_murnaghan = oBis.new_object_type(
        code=code_obj_pyiron_murnaghan,
        generatedCodePrefix=code_obj_pyiron_murnaghan,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_pyiron_murnaghan)

    obj_pyiron_murnaghan = oBis.get_object_type(code_obj_pyiron_murnaghan)
    obj_pyiron_murnaghan.description = "MURNAGHAN pyiron job"
    register_object_type(oBis, obj_pyiron_murnaghan)

    obj_pyiron_murnaghan = oBis.get_object_type(code_obj_pyiron_murnaghan)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("DESCRIPTION_MULTILINE", 0, None),
            ("USER_NAME", 0, None),
            ("NUM_DATASETS", 0, None),
        ],
        "Technical Information": [
            ("WORKFLOW_MANAGER", 0, None),
        ],
        "Pyiron Information": [
            ("SIM_JOB_FINISHED", 0, None),
            ("START_DATE", 0, None),
            ("SIM_WALLTIME_IN_HOURS", 0, None),
            ("SIM_CORETIME_IN_HOURS", 0, None),
            ("NCORES", 0, None),
            ("NGPUS", 0, None),
            ("NTHREADS", 0, None),
            ("HPC_JOB_QUEUE", 0, None),
            ("HPC_JOB_ID", 0, None),
        ],
        "Method-specific": [
            ("MURN_EQN_OF_STATE", 0, None),
            ("MURN_FIT_EQN_ORDER", 0, None),
            ("MURN_STRAIN_AXES", 0, None),
        ],
        "Input": [
            ("MURN_N_DATA_POINTS", 0, None),
            ("MURN_STRAINVOL_RANGE", 0, None),
        ],
        "Output": [
            ("ATOM_EQUIL_K_MOD_IN_GPA", 0, None),
            ("ATOM_EQUIL_TOTENG_IN_EV", 0, None),
            ("ATOM_EQUIL_VOL_IN_A3", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Annotations": [
            ("PYIRON_CONCEPTUAL_DICTIONARY", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(obj_pyiron_murnaghan, sec2props)

    # #########################################################################
    # Dataset Types
    # #########################################################################

    # Crystalline Structure

    ds_crys_data_data = oBis.new_dataset_type(
        code=code_ds_crys_struct_data,
        description="Data from Structure File",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_crys_data_data)

    ds_crys_data_data = oBis.get_dataset_type(code_ds_crys_struct_data)

    # Define display order

    composition_properties = []
    for eid in range(1, 1 + MAX_NUM_ELEMENTS):
        composition_properties += [
            ("ELEMENT_%s" % eid, 1 if eid == 1 else 0, None),
            ("ELEMENT_%s_AT_PERCENT" % eid, 0, None),
            ("ELEMENT_%s_NUMBER" % eid, 0, None),
        ]
    sec2props = {
        "General": [
            ("$NAME", 0, None),
            ("DATE", 0, None),  # Name, Comments -> manual entry
            ("COMMENTS", 0, None),
            ("FILE_SOURCE", 0, None),
            ("LAMMPS_UNIT_STYLE", 0, None),
            ("DWF", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        "Composition": composition_properties,
        "Content": [
            ("NUMBER_OF_ATOMS", 0, None),
            ("NUMBER_OF_ATOM_TYPES", 0, None),
            ("LIST_OF_ATOM_TYPES", 0, None),
            ("NUMBER_OF_SPECIES", 0, None),
            ("LIST_OF_SPECIES", 0, None),
        ],
        "Simulation Box": [
            ("BOX_LENGTH_A", 0, None),
            ("BOX_LENGTH_B", 0, None),
            ("BOX_LENGTH_C", 0, None),
            ("ANGLE_ALPHA", 0, None),
            ("ANGLE_BETA", 0, None),
            ("ANGLE_GAMMA", 0, None),
        ],
        "Symmetry": [("SPACE_GROUP", 0, None)],
    }
    assign_property_types(ds_crys_data_data, sec2props)

    # Archive

    ds_sim_archive_data = oBis.new_dataset_type(
        code=code_ds_sim_archive_data,
        description="Archive containing files from a simulation job",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_sim_archive_data)

    ds_sim_archive_data = oBis.get_dataset_type(code_ds_sim_archive_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
    }
    assign_property_types(ds_sim_archive_data, sec2props)

    # Generic artefact

    ds_sim_artefact_data = oBis.new_dataset_type(
        code=code_ds_sim_artefact_data,
        description="Output file from simulation (excluding structure files)",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_sim_artefact_data)

    ds_sim_artefact_data = oBis.get_dataset_type(code_ds_sim_artefact_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
    }
    assign_property_types(ds_sim_artefact_data, sec2props)

    # Define Computational Environment File Dataset Type
    ds_env_file = oBis.new_dataset_type(
        code=code_ds_env_file,
        description="Computational environment file",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_env_file)

    ds_env_file = oBis.get_dataset_type(code_ds_env_file)

    # Define display order
    sec2props = {
        "General Information": [
            ("$NAME", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        "Technical Information": [
            ("ENV_TOOL", 0, None),
        ],
        "Additional Information": [
            ("NOTES", 0, None),
        ],
        "Comments": [
            ("$XMLCOMMENTS", 0, None),
            ("$ANNOTATIONS_STATE", 0, None),
        ],
    }
    assign_property_types(ds_env_file, sec2props)

    # Interatomic potential data
    ds_interatomic_potential = oBis.new_dataset_type(
        code=code_ds_interatomic_potential_data,
        description="Interatomic Potential File",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_interatomic_potential)

    ds_interatomic_potential = oBis.get_dataset_type(code_ds_interatomic_potential_data)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
            ("MD5_HASH", 0, None),
            ("SOFTWARENAME", 0, None),
            ("SOFTWAREVERSION", 0, None),
            ("USER_NAME", 0, None),
        ],
    }
    assign_property_types(ds_interatomic_potential, sec2props)

    # Pyiron Conceptual Dictionary (as a file)

    ds_pyiron_concept_dict = oBis.new_dataset_type(
        code=code_ds_pyiron_concept_dict,
        description="Pyiron Conceptual Dictionary File",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_pyiron_concept_dict)

    ds_pyiron_concept_dict = oBis.get_dataset_type(code_ds_pyiron_concept_dict)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
            ("MD5_HASH", 0, None),
        ],
    }
    assign_property_types(ds_pyiron_concept_dict, sec2props)

    # Pyiron HDF5
    ds_pyiron_h5 = oBis.new_dataset_type(
        code=code_ds_pyiron_h5,
        description="Pyiron HDF5 file",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_pyiron_h5)

    ds_pyiron_h5 = oBis.get_dataset_type(code_ds_pyiron_h5)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
    }
    assign_property_types(ds_pyiron_h5, sec2props)
