#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""

    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for (micro/nano) mechanical testing

"""

from pybis import Openbis
from schema_helpers import *

code_obj_micro_mecha_exp = "MICRO_MECH_EXP"
code_ds_indent = "NANOINDENTATION_DATA"
code_ds_pillar = "PILLAR_COMPRESSION_DATA"


def create_micromechanics_schema(oBis: Openbis):
    """Creates (Micro/Nano) Mechanics Object Types and Dataset Types.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """
    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    # Test Type
    terms_micro_mecha_test_type = [
        {
            "code": "CANTILEVER_BENDING",
            "label": "Microcantilever Bending",
            "description": "Microcantilever Bending",
        },
        {
            "code": "PILLARCOMPRESSION",
            "label": "Micropillar Compression",
            "description": "Micropillar Compression",
        },
        {
            "code": "NANOINDENTATION",
            "label": "Nanoindentation",
            "description": "Nanoindentation",
        },
        {
            "code": "PILLARSPLITTING",
            "label": "Micropillar Splitting",
            "description": "Micropillar Splitting",
        },
    ]
    voc_micro_mecha_test_type = oBis.new_vocabulary(
        code="NANOMECH_VOCAB",
        description="Type of (Micro/Nano)mechanical test",
        terms=terms_micro_mecha_test_type,
    )
    register_controlled_vocabulary(
        oBis, voc_micro_mecha_test_type, terms_micro_mecha_test_type
    )

    # Tip Material
    terms_tip_material = [
        {
            "code": "DIAMOND_TIP",
            "label": "Diamond",
            "description": "Diamond Nano Indenter Tip",
        },
        {
            "code": "SAPPHIRE_TIP",
            "label": "Sapphire",
            "description": "Sapphire Nano Indenter Tip",
        },
        {
            "code": "TUNGSTEN_CARBIDE_TIP",
            "label": "Tungsten Carbide",
            "description": "Tungsten Carbide Nano Indenter Tip",
        },
        {
            "code": "QUARTZ_TIP",
            "label": "Quartz",
            "description": "Quartz Nano Indenter Tip",
        },
        {
            "code": "SILICON_TIP",
            "label": "Silicon",
            "description": "Silicon Nano Indenter Tip",
        },
        {
            "code": "STEEL_TIP",
            "label": "Steel",
            "description": "Steel Nano Indenter Tip",
        },
    ]
    voc_tip_material = oBis.new_vocabulary(
        code="TIP_MATERIAL_VOCAB",
        description="Material for nano indenter tip",
        terms=terms_tip_material,
    )
    register_controlled_vocabulary(oBis, voc_tip_material, terms_tip_material)

    # Tip Type
    terms_tip_type = [
        {"code": "BERKOVICH", "label": "Berkovich", "description": "Berkovich Tip"},
        {"code": "VICKERS", "label": "Vickers", "description": "Vickers Tip"},
        {"code": "KNOOP", "label": "Knoop", "description": "Knoop Tip"},
        {
            "code": "CUBE_CORNER",
            "label": "Cube Corner",
            "description": "Cube Corner Tip",
        },
        {"code": "CONE", "label": "Cone", "description": "Cone Tip"},
        {"code": "SPHERE", "label": "Sphere", "description": "Sphere Tip"},
        {"code": "FLAT_PUNCH", "label": "Flat Punch", "description": "Flat Punch Tip"},
    ]
    voc_tip_type = oBis.new_vocabulary(
        code="TIP_TYPE_VOCAB",
        description="Geometry of nano indenter tip",
        terms=terms_tip_type,
    )
    register_controlled_vocabulary(oBis, voc_tip_type, terms_tip_type)

    # Indent control
    terms_control = [
        {
            "code": "INDENT_LOAD",
            "label": "Load",
            "description": "Load controlled indentation",
        },
        {
            "code": "INDENT_DISP",
            "label": "Displacement",
            "description": "Displacement controlled indentation",
        },
    ]
    voc_control = oBis.new_vocabulary(
        code="INDENT_CONTROL_VOCAB",
        description="Control method for the indentation experiment",
        terms=terms_control,
    )
    register_controlled_vocabulary(oBis, voc_control, terms_control)

    # #########################################################################
    # Property Types
    # #########################################################################

    # Experiment Type (Generic)
    pt_test_type = oBis.new_property_type(
        code="TEST_TYPE",
        label="Test Type",
        description="Type of (Micro/Nano) Mechanical Test",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_micro_mecha_test_type.code,
    )
    register_property_type(oBis, pt_test_type)

    # Experiment Type (Specific)
    pt_experiment_type = oBis.new_property_type(
        code="EXPERIMENT_TYPE",
        label="Experiment Type",
        description="(Micro/Nano) Mechanical Experiment Type as given by manufacturer",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_experiment_type)

    # Number of tests
    pt_test_count = oBis.new_property_type(
        code="TEST_COUNT",
        label="Test Count",
        description="Number of tests performed",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_test_count)

    # Nano indent Array Spacing A
    pt_indent_spacing_a = oBis.new_property_type(
        code="ARRAY_SPACING_A_UM",
        label="Spacing A (µm)",
        description="Nano Indent Array Spacing (A-direction) in micrometers",
        dataType="REAL",
    )
    register_property_type(oBis, pt_indent_spacing_a)

    # Nano indent Array Spacing B
    pt_indent_spacing_b = oBis.new_property_type(
        code="ARRAY_SPACING_B_UM",
        label="Spacing B (µm)",
        description="Nano Indent Array Spacing (B-direction) in micrometers",
        dataType="REAL",
    )
    register_property_type(oBis, pt_indent_spacing_b)

    # Indent control
    pt_control = oBis.new_property_type(
        code="INDENT_CONTROL",
        label="Control Method",
        description="Control Method",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_control.code,
    )
    register_property_type(oBis, pt_control)

    # Tip material
    pt_tip_material = oBis.new_property_type(
        code="TIP_MATERIAL",
        label="Tip Material",
        description="Nano indenter tip material",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_tip_material.code,
    )
    register_property_type(oBis, pt_tip_material)

    # Tip type
    pt_tip_type = oBis.new_property_type(
        code="TIP_TYPE",
        label="Tip Type",
        description="Nano indenter tip type",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_tip_type.code,
    )
    register_property_type(oBis, pt_tip_type)

    # Tip Diameter
    pt_tip_diameter = oBis.new_property_type(
        code="TIP_DIAMETER_UM",
        label="Tip Diameter (µm)",
        description="Tip Diameter (µm)",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tip_diameter)

    # Frame stiffness
    pt_frame_stiffness = oBis.new_property_type(
        code="INDENT_FRAME_STIFFNESS",
        label="Frame Stiffness (N/m)",
        description="Frame Stiffness [N/m]",
        dataType="REAL",
    )
    register_property_type(oBis, pt_frame_stiffness)

    # Tip ID
    pt_tip_id = oBis.new_property_type(
        code="INDENT_TIP_ID",
        label="Tip ID",
        description="Tip ID",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_tip_id)

    # Tip Modulus ratio
    pt_tip_modulus = oBis.new_property_type(
        code="TIP_MODULUS_GPA",
        label="Tip Modulus (GPa)",
        description="Tip Modulus in GPa",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tip_modulus)

    # Tip Poisson ratio
    pt_tip_poisson = oBis.new_property_type(
        code="TIP_POISSON_RATIO",
        label="Tip Poisson Ratio",
        description="Tip Poisson Ratio",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tip_poisson)

    # Sample Poisson ratio
    pt_sample_poisson = oBis.new_property_type(
        code="SAMPLE_POISSON_RATIO",
        label="Sample Poisson Ratio",
        description="Sample Poisson Ratio",
        dataType="REAL",
    )
    register_property_type(oBis, pt_sample_poisson)

    # Oliver Pharr
    pt_beta_oliver_pharr = oBis.new_property_type(
        code="OLIVER_PHARR_BETA",
        label="β (Oliver-Pharr)",
        description="Oliver-Pharr Beta",
        dataType="REAL",
    )
    register_property_type(oBis, pt_beta_oliver_pharr)

    pt_epsilon_oliver_pharr = oBis.new_property_type(
        code="OLIVER_PHARR_EPSILON",
        label="ε (Oliver-Pharr)",
        description="Oliver-Pharr Epsilon",
        dataType="REAL",
    )
    register_property_type(oBis, pt_epsilon_oliver_pharr)

    pt_calibration_details = oBis.new_property_type(
        code="CALIBRATION_INFO",
        label="Calibration Details",
        description="Calibration Details",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_calibration_details)

    pt_damping_coefficients = oBis.new_property_type(
        code="DAMPING_COEFFICIENTS",
        label="Damping Coefficients",
        description="Damping Coefficients",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_damping_coefficients)

    pt_spring_coefficients = oBis.new_property_type(
        code="SPRING_COEFFICIENTS",
        label="Spring Coefficients",
        description="Spring Coefficients",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_spring_coefficients)

    pt_tip_area = oBis.new_property_type(
        code="TIP_AREA_CONSTANT",
        label="Tip Area Constant",
        description="Tip Area Constant",
        dataType="REAL",
    )
    register_property_type(oBis, pt_tip_area)

    pt_tip_area_function = oBis.new_property_type(
        code="TIP_AREA_FUNCTION_TYPE",
        label="Tip Area Function Type",
        description="Tip Area Function Type",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_tip_area_function)

    pt_max_load = oBis.new_property_type(
        code="MICROMECHA_MAXIMUM_LOAD",
        label="Maximum Load (mN)",
        description="Maximum / Target Load in mN",
        dataType="REAL",
    )
    register_property_type(oBis, pt_max_load)

    pt_max_depth = oBis.new_property_type(
        code="MICROMECHA_MAXIMUM_DEPTH",
        label="Maximum Depth (nm)",
        description="Maximum / Target Depth in nm",
        dataType="REAL",
    )
    register_property_type(oBis, pt_max_depth)

    pt_csm_freq = oBis.new_property_type(
        code="CSM_FREQUENCY_HZ",
        label="CSM Frequency (Hz)",
        description="CSM Frequency in Hertz",
        dataType="REAL",
    )
    register_property_type(oBis, pt_csm_freq)

    pt_drift = oBis.new_property_type(
        code="DRIFT_SETTLE_TIME",
        label="Drift Settle Time (s)",
        description="Drift Settle Time in seconds",
        dataType="REAL",
    )
    register_property_type(oBis, pt_drift)

    pt_hold_time = oBis.new_property_type(
        code="HOLD_TIME_NANOINDENT",
        label="Hold Time (s)",
        description="Hold Time in seconds at Maximum Load",
        dataType="REAL",
    )
    register_property_type(oBis, pt_hold_time)

    pt_strain_rate_min = oBis.new_property_type(
        code="MICROMECHA_MINIMUM_STRAIN_RATE",
        label="Min. Strain Rate (1/s)",
        description="Minimum Strain Rate in a micromechanical experiment",
        dataType="REAL",
    )
    register_property_type(oBis, pt_strain_rate_min)

    pt_strain_rate_max = oBis.new_property_type(
        code="MICROMECHA_MAXIMUM_STRAIN_RATE",
        label="Max. Strain Rate (1/s)",
        description="Maximum Strain Rate in a micromechanical experiment",
        dataType="REAL",
    )
    register_property_type(oBis, pt_strain_rate_max)

    # #########################################################################
    # Objects
    # #########################################################################

    # Experiment
    obj_indent_exp = oBis.new_object_type(
        code=code_obj_micro_mecha_exp,
        generatedCodePrefix=code_obj_micro_mecha_exp,
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_indent_exp)

    obj_indent_exp = oBis.get_object_type(code_obj_micro_mecha_exp)
    obj_indent_exp.description = "(Micro / Nano) mechanical test experiment"
    register_object_type(oBis, obj_indent_exp)

    obj_indent_exp = oBis.get_object_type(code_obj_micro_mecha_exp)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("$SHOW_IN_PROJECT_OVERVIEW", 0, None),
            ("FINISHED_FLAG", 0, None),
            ("START_DATE", 1, None),
            ("END_DATE", 0, None),
            ("COMMENTS", 0, None),
            ("MEASUREMENT_POSITION", 0, None),
            ("NUM_DATASETS", 0, "ENTITY_DATASET_COUNT"),
        ],
        "Experimental Details": [
            ("TIP_TYPE", 1, None),
            ("TIP_MATERIAL", 1, None),
            ("TEST_TYPE", 1, None),
            ("TIP_DIAMETER_UM", 0, None),
            ("TEMPERATURE", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_GOALS", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_DESCRIPTION", 0, None),
            ("EXPERIMENTAL_STEP.EXPERIMENTAL_RESULTS", 0, None),
            ("EXPERIMENTAL_STEP.SPREADSHEET", 0, None),
        ],
    }
    assign_property_types(obj_indent_exp, sec2props)

    # #########################################################################
    # Dataset types
    # #########################################################################

    # Indentation data
    ds_nano_indent = oBis.new_dataset_type(
        code=code_ds_indent,
        description="Nano indentation data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_nano_indent)

    ds_nano_indent = oBis.get_dataset_type(code_ds_indent)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("DEVICE_MANUFACTURER", 0, None),
            ("DEVICE_NAME", 0, None),
            ("SOFTWAREVERSION", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
            ("COMMENTS", 0, None),
        ],
        "Initial Analysis": [
            ("OLIVER_PHARR_BETA", 0, None),
            ("OLIVER_PHARR_EPSILON", 0, None),
            ("SAMPLE_POISSON_RATIO", 0, None),
        ],
        "Experiment": [
            ("MICROMECHA_MAXIMUM_LOAD", 0, None),
            ("MICROMECHA_MAXIMUM_DEPTH", 0, None),
            ("SAMPLING_RATE_HZ", 0, None),
            ("SAMPLING_RATE_MIN", 0, None),
            ("SAMPLING_RATE_MAX", 0, None),
            ("INDENT_CONTROL", 0, None),
            ("ARRAY_SPACING_A_UM", 0, None),
            ("ARRAY_SPACING_B_UM", 0, None),
            ("EXPERIMENT_TYPE", 0, None),
            ("TEST_COUNT", 0, None),
            ("CSM_FREQUENCY_HZ", 0, None),
            ("DRIFT_SETTLE_TIME", 0, None),
            ("HOLD_TIME_NANOINDENT", 0, None),
            ("MICROMECHA_MINIMUM_STRAIN_RATE", 0, None),
            ("MICROMECHA_MAXIMUM_STRAIN_RATE", 0, None),
        ],
        "Tip": [
            ("INDENT_TIP_ID", 0, None),
            ("TIP_MODULUS_GPA", 0, None),
            ("TIP_POISSON_RATIO", 0, None),
            ("TIP_AREA_CONSTANT", 0, None),
            ("TIP_AREA_FUNCTION_TYPE", 0, None),
        ],
        "Calibration": [
            ("CALIBRATION_INFO", 0, None),
            ("CALIBRATION_DATE", 0, None),
            ("INDENT_FRAME_STIFFNESS", 0, None),
            ("DAMPING_COEFFICIENTS", 0, None),
            ("SPRING_COEFFICIENTS", 0, None),
        ],
    }
    assign_property_types(ds_nano_indent, sec2props)

    # Micro pillar compression
    ds_pillar_comp = oBis.new_dataset_type(
        code=code_ds_pillar,
        description="Micro pillar compression data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_pillar_comp)

    ds_pillar_comp = oBis.get_dataset_type(code_ds_pillar)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("DEVICE_MANUFACTURER", 0, None),
            ("DEVICE_NAME", 0, None),
            ("SOFTWAREVERSION", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
            ("COMMENTS", 0, None),
        ],
        "Calibration": [
            ("INDENT_FRAME_STIFFNESS", 0, None),
        ],
        "Tip": [
            ("INDENT_TIP_ID", 0, None),
            ("TIP_MODULUS_GPA", 0, None),
            ("TIP_POISSON_RATIO", 0, None),
        ],
        "Experiment": [
            ("SAMPLING_RATE_HZ", 0, None),
            ("SAMPLING_RATE_MIN", 0, None),
            ("SAMPLING_RATE_MAX", 0, None),
            ("INDENT_CONTROL", 0, None),
            ("EXPERIMENT_TYPE", 0, None),
            ("TEST_COUNT", 0, None),
        ],
    }
    assign_property_types(ds_pillar_comp, sec2props)
