#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for Electrochemistry and Corrosion studies

"""


from pybis import Openbis
from schema_helpers import *


# openBIS codes for the objects and datasets (also used in other modules)

code_obj_electrochem_protocol = "ELECTROCHEM_PROTOCOL"
code_ds_electrochem_data = "ELECTROCHEM_DATA"  # Generic
code_ds_svet_data = "SVET_DATA"
code_ds_siet_data = "SIET_DATA"
code_ds_pdp_data = "PDP_DATA"
code_ds_ocp_data = "OCP_DATA"
code_ds_xps_data = "XPS_DATA"
code_ds_xrr_data = "XRR_DATA"
code_ds_eis_data = "EIS_DATA"


def create_electrochem_schema(oBis: Openbis):
    """Creates Electrochemistry Entity Types.

        We create "empty" types so that data can be uploaded to the correct place.
        Once parsers are developed, we update the properties.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """
    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    # terms_CHANGEME =[
    #     {'code': '', 'label': '', 'description': ''},
    # ]
    # voc_CHANGEME = oBis.new_vocabulary(
    #     code = '',
    #     description = '',
    #     terms = terms_CHANGEME,
    # )
    # register_controlled_vocabulary(oBis, voc_CHANGEME, terms_CHANGEME)

    # #########################################################################
    # Property Types
    # #########################################################################

    pt_scan_rate = oBis.new_property_type(
        code="SCAN_RATE",
        label="Scan Rate (mV/s)",
        description="Scan Rate (mV/s)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_scan_rate)

    pt_ref_electrode = oBis.new_property_type(
        code="REFERENCE_ELECTRODE",
        label="Reference Electrode",
        description="Reference Electrode",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary="ELECTRODE_VOCAB",  # Defined in Instruments
    )
    register_property_type(oBis, pt_ref_electrode)

    pt_counter_electrode = oBis.new_property_type(
        code="COUNTER_ELECTRODE",
        label="Counter Electrode",
        description="Counter Electrode",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_counter_electrode)

    # #########################################################################
    # Objects
    # #########################################################################

    obj_electrochem_protocol = oBis.new_object_type(
        code=code_obj_electrochem_protocol,
        generatedCodePrefix="ECHEMPROT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_electrochem_protocol)

    obj_electrochem_protocol = oBis.get_object_type(code_obj_electrochem_protocol)
    obj_electrochem_protocol.description = "Electrochemistry Protocol"
    register_object_type(oBis, obj_electrochem_protocol)

    obj_electrochem_protocol = oBis.get_object_type(code_obj_electrochem_protocol)

    # Define display order
    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DESCRIPTION", 0, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
        ],
        "Technical Setup Name": [
            ("ELECTROLYTE", 0, None),
            ("REFERENCE_ELECTRODE", 0, None),
            ("COUNTER_ELECTRODE", 0, None),
            ("GAS_EXP", 0, None),
        ],
    }
    assign_property_types(obj_electrochem_protocol, sec2props)

    # #########################################################################
    # Dataset Types
    # #########################################################################

    # Generic

    ds_electrochem_data = oBis.new_dataset_type(
        code=code_ds_electrochem_data,
        description="Generic Electrochemical Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_electrochem_data)

    ds_electrochem_data = oBis.get_dataset_type(code_ds_electrochem_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_electrochem_data, sec2props)

    # SVET

    ds_svet_data = oBis.new_dataset_type(
        code=code_ds_svet_data,
        description="Scanning Vibrating Electrode Technique (SVET) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_svet_data)

    ds_svet_data = oBis.get_dataset_type(code_ds_svet_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_svet_data, sec2props)

    # SIET

    ds_siet_data = oBis.new_dataset_type(
        code=code_ds_siet_data,
        description="Scanning Ion-selective Electrode Technique (SIET) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_siet_data)

    ds_siet_data = oBis.get_dataset_type(code_ds_siet_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_siet_data, sec2props)

    # PDP

    ds_pdp_data = oBis.new_dataset_type(
        code=code_ds_pdp_data,
        description="Potentiodynamic Polarization (PDP) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_pdp_data)

    ds_pdp_data = oBis.get_dataset_type(code_ds_pdp_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("COMMENTS", 0, None),
            ("DATE", 0, None),
            ("TIME", 0, None),
            ("DATETIME", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        "Experimental Details": [
            ("ELECTROLYTE", 0, None),
            ("SCAN_RATE", 0, None),
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_pdp_data, sec2props)

    # OCP

    ds_ocp_data = oBis.new_dataset_type(
        code=code_ds_ocp_data,
        description="Open Circuit Potential (OCP) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_ocp_data)

    ds_ocp_data = oBis.get_dataset_type(code_ds_ocp_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("COMMENTS", 0, None),
            ("DATE", 0, None),
            ("TIME", 0, None),
            ("DATETIME", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        "Experimental Details": [
            ("ELECTROLYTE", 0, None),
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_ocp_data, sec2props)

    # XPS

    ds_xps_data = oBis.new_dataset_type(
        code=code_ds_xps_data,
        description="X-ray Photoelectron Spectroscopy (XPS) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_xps_data)

    ds_xps_data = oBis.get_dataset_type(code_ds_xps_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_xps_data, sec2props)

    # XRR

    ds_xrr_data = oBis.new_dataset_type(
        code=code_ds_xrr_data,
        description="X-ray reflectometry (XRR) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_xrr_data)

    ds_xrr_data = oBis.get_dataset_type(code_ds_xrr_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_xrr_data, sec2props)

    # EIS

    ds_eis_data = oBis.new_dataset_type(
        code=code_ds_eis_data,
        description="Electrochemical Impedance Spectroscopy (EIS) Data",
        mainDataSetPattern=None,
        mainDataSetPath=None,
        disallowDeletion=False,
        validationPlugin=None,
    )
    register_dataset_type(oBis, ds_eis_data)

    ds_eis_data = oBis.get_dataset_type(code_ds_eis_data)

    # Define display order

    sec2props = {
        "General": [
            ("$NAME", 1, None),
            ("DATE", 0, None),
            ("COMMENTS", 0, None),
            ("S3_DOWNLOAD_LINK", 0, None),  # Download Link -> automatic script
        ],
        # 'Section Name': [
        #     ('PROPERTY_CODE', is_mandatory, plugin_code),
        # ],
    }
    assign_property_types(ds_eis_data, sec2props)
