#!/usr/bin/env python
# -*- coding: utf-8 -*-


"""
    Copyright 2025 Ulrich Kerzel, Khalil Rejiba

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

    _summary_ : Create metadata schema for consumables

    Consumables such as chemicals, etc.
"""


from pybis import Openbis
from schema_helpers import *


# openBIS codes for the objects (also used in other modules)

code_material = "MATERIAL"
code_amorphous_material = "AMORPHOUS_MATERIAL"
code_crystalline_material = "CRYSTALLINE_MATERIAL"
code_engineered_material = "ENGINEERED_MATERIAL"


def create_crystallography_schema(oBis: Openbis):
    """Creates Crystallography object types.

    Args:
        oBis (pybis.Openbis): OpenBIS python object.
    """

    # #########################################################################
    # Controlled Vocabularies
    # #########################################################################

    # Crystal Systems

    terms_crystal_system = [
        {
            "code": "CRYSTAL_SYSTEM_CUBIC",
            "label": "cubic",
            "description": "cubic (23, m-3, 432, -43m, m-3m)",
        },
        {
            "code": "CRYSTAL_SYSTEM_HEXAGONAL",
            "label": "hexagonal",
            "description": "hexagonal (6, -6, 6/m, 622, 6mm, -6m2, 6/mmm)",
        },
        {
            "code": "CRYSTAL_SYSTEM_MONOCLINIC",
            "label": "monoclinic",
            "description": "monoclinic (2, m, 2/m)",
        },
        {
            "code": "CRYSTAL_SYSTEM_ORTHORHOMBIC",
            "label": "orthorhombic",
            "description": "orthorhombic (222, mm2, mmm)",
        },
        {
            "code": "CRYSTAL_SYSTEM_TETRAGONAL",
            "label": "tetragonal",
            "description": "tetragonal (4, -4, 4/m, 422, 4mm, -42m, 4/mmm)",
        },
        {
            "code": "CRYSTAL_SYSTEM_TRICLINIC",
            "label": "triclinic",
            "description": "triclinic (1, -1)",
        },
        {
            "code": "CRYSTAL_SYSTEM_TRIGONAL",
            "label": "trigonal",
            "description": "trigonal (3, -3, 32, 3m, -3m)",
        },
    ]
    assert len(terms_crystal_system) == 7, "7 Crystal Systems"

    voc_crystal_system = oBis.new_vocabulary(
        code="CRYSTAL_SYSTEM",
        description="Crystal System",
        terms=terms_crystal_system,
    )
    register_controlled_vocabulary(oBis, voc_crystal_system, terms_crystal_system)

    # Bravais Lattices

    terms_bravais_lattice_vocab = [
        {
            "code": "BRAVAIS_LATTICE_SIMPLE_CUBIC",
            "label": "Simple Cubic (SC)",
            "description": "Simple cubic (SC) // Kubisch-primitiv",
        },
        {
            "code": "BRAVAIS_LATTICE_BODY_CENTER_CUBIC",
            "label": "Body-Centered Cubic (BCC)",
            "description": "Body-centered cubic (BCC) // Kubisch-raumzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_FACE_CENTER_CUBIC",
            "label": "Face-Centered Cubic (FCC)",
            "description": "Face-centered cubic (FCC) // Kubisch-flächenzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_HEX_CLOSE_PACK",
            "label": "Hexagonal",
            "description": "Hexagonal // Hexagonal",
        },
        {
            "code": "BRAVAIS_LATTICE_RHOMBOHEDRAL",
            "label": "Rhombohedral",
            "description": "Rhombohedral // Rhomboedrisch",
        },
        {
            "code": "BRAVAIS_LATTICE_SIMPLE_TETRA",
            "label": "Simple Tetragonal",
            "description": "Simple tetragonal // Tetragonal-primitiv",
        },
        {
            "code": "BRAVAIS_LATTICE_BODY_CENTER_TETRA",
            "label": "Body-Centered Tetragonal (BCT)",
            "description": "Body-centered tetragonal (BCT) // Tetragonal-raumzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_SIMPLE_ORTHORMB",
            "label": "Simple Orthorhombic",
            "description": "Simple orthorhombic // Orthorhombisch-primitiv",
        },
        {
            "code": "BRAVAIS_LATTICE_BASE_ORTHORMB",
            "label": "Base-Centered Orthorhombic",
            "description": "Base-centered orthorhombic // Orthorhombisch-basisflächenzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_BODY_ORTHORMB",
            "label": "Body-centered Orthorhombic",
            "description": "Body-centered orthorhombic // Orthorhombisch-raumzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_FACE_ORTHORMB",
            "label": "Face-Centered Orthorhombic",
            "description": "Face-centered orthorhombic // Orthorhombisch-flächenzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_SIMPLE_MONOCLINIC",
            "label": "Simple Monoclinic",
            "description": "Simple monoclinic // Monoklin-primitiv",
        },
        {
            "code": "BRAVAIS_LATTICE_BASE_MONOCLINIC",
            "label": "Base-Centered Monoclinic",
            "description": "Base-centered monoclinic // Monoklin-basisflächenzentriert",
        },
        {
            "code": "BRAVAIS_LATTICE_TRICLINIC",
            "label": "Triclinic",
            "description": "Triclinic // Triklin",
        },
    ]

    assert len(terms_bravais_lattice_vocab) == 14, "14 Bravais Lattices"

    voc_bravais_lattice_vocab = oBis.new_vocabulary(
        code="BRAVAIS_LATTICE_VOCAB",
        description="Bravais lattice // Bravais-Gitter",
        terms=terms_bravais_lattice_vocab,
    )
    register_controlled_vocabulary(
        oBis, voc_bravais_lattice_vocab, terms_bravais_lattice_vocab
    )

    # Point Groups

    point_groups = {
        "PG1": "1",
        "PG-1": "-1",
        "PG2": "2",
        "PG2_M": "2/m",
        "PG3": "3",
        "PG-3": "-3",
        "PG-3M": "-3m",
        "PG3M": "3m",
        "PG4": "4",
        "PG4_M": "4/m",
        "PG4_MMM": "4/mmm",
        "PG-4": "-4",
        "PG-42M": "-42m",
        "PG-43M": "-43m",
        "PG4MM": "4mm",
        "PG6": "6",
        "PG6_M": "6/m",
        "PG6_MMM": "6/mmm",
        "PG-6": "-6",
        "PG-6M2": "-6m2",
        "PG6MM": "6mm",
        "PG23": "23",
        "PG32": "32",
        "PG222": "222",
        "PG422": "422",
        "PG432": "432",
        "PG622": "622",
        "PGM": "m",
        "PGM-3": "m-3",
        "PGM-3M": "m-3m",
        "PGMM2": "mm2",
        "PGMMM": "mmm",
    }
    terms_point_group = []
    for k, v in point_groups.items():
        minus_pos = v.find("-")
        if minus_pos > -1:
            v = v[:minus_pos] + v[minus_pos + 1] + "\u0305" + v[minus_pos + 2 :]
        terms_point_group.append(
            {
                "code": k,
                "label": v,
                "description": "Crystallographic Point Group " + v,
            }
        )

    assert len(terms_point_group) == 32, "32 Point Groups"

    voc_point_group = oBis.new_vocabulary(
        code="POINT_GROUP_VOCAB",
        description="Crystallographic Point Groups (International Symbol)",
        terms=terms_point_group,
    )
    register_controlled_vocabulary(oBis, voc_point_group, terms_point_group)

    # Crystallographic Space group (3D)
    terms_space_group_vocab = [
        {
            "code": "SPACE_GROUP_P1",
            "label": "P1",
            "description": "Triclinic, 1 // Triklin, 1",
        },
        {
            "code": "SPACE_GROUP_P-1",
            "label": "P-1",
            "description": "Triclinic, -1 // Triklin, -1",
        },
        {
            "code": "SPACE_GROUP_P2",
            "label": "P2",
            "description": "Monoclinic, 2 // Monoklin, 2",
        },
        {
            "code": "SPACE_GROUP_P21",
            "label": "P2_1",
            "description": "Monoclinic, 2 // Monoklin, 2",
        },
        {
            "code": "SPACE_GROUP_C2",
            "label": "C2",
            "description": "Monoclinic, 2 // Monoklin, 2",
        },
        {
            "code": "SPACE_GROUP_Pm",
            "label": "Pm",
            "description": "Monoclinic, m // Monoklin, m",
        },
        {
            "code": "SPACE_GROUP_Pc",
            "label": "Pc",
            "description": "Monoclinic, m // Monoklin, m",
        },
        {
            "code": "SPACE_GROUP_Cm",
            "label": "Cm",
            "description": "Monoclinic, m // Monoklin, m",
        },
        {
            "code": "SPACE_GROUP_Cc",
            "label": "Cc",
            "description": "Monoclinic, m // Monoklin, m",
        },
        {
            "code": "SPACE_GROUP_P2_m",
            "label": "P2/m",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_P21_m",
            "label": "P2_1/m",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_P2_c",
            "label": "P2/c",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_P21_c",
            "label": "P2_1/c",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_C2_c",
            "label": "C2/c",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_C2_m",
            "label": "C2/m",
            "description": "Monoclinic, 2/m // Monoklin, 2/m",
        },
        {
            "code": "SPACE_GROUP_P222",
            "label": "P222",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_P2221",
            "label": "P222_1",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_P21212",
            "label": "P2_12_12",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_P212121",
            "label": "P2_12_12_1",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_C2221",
            "label": "C222_1",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_C222",
            "label": "C222",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_F222",
            "label": "F222",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_I222",
            "label": "I222",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_I212121",
            "label": "I2_12_12_1",
            "description": "Orthorhombic, 222 // Orthorhombisch, 222",
        },
        {
            "code": "SPACE_GROUP_Pmm2",
            "label": "Pmm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pmc21",
            "label": "Pmc2_1",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pcc2",
            "label": "Pcc2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pma2",
            "label": "Pma2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pca21",
            "label": "Pca2_1",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pnc2",
            "label": "Pnc2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pmn21",
            "label": "Pmn2_1",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pba2",
            "label": "Pba2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pna21",
            "label": "Pna2_1",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pnn2",
            "label": "Pnn2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Cmm2",
            "label": "Cmm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Cmc21",
            "label": "Cmc2_1",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Ccc2",
            "label": "Ccc2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Amm2",
            "label": "Amm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Abm2",
            "label": "Abm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Ama2",
            "label": "Ama2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Aba2",
            "label": "Aba2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Fmm2",
            "label": "Fmm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Fdd2",
            "label": "Fdd2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Imm2",
            "label": "Imm2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Iba2",
            "label": "Iba2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Ima2",
            "label": "Ima2",
            "description": "Orthorhombic, mm2 // Orthorhombisch, mm2",
        },
        {
            "code": "SPACE_GROUP_Pmmm",
            "label": "Pmmm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pnnn",
            "label": "Pnnn",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pccm",
            "label": "Pccm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pban",
            "label": "Pban",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pmma",
            "label": "Pmma",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pnna",
            "label": "Pnna",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pmna",
            "label": "Pmna",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pcca",
            "label": "Pcca",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pbam",
            "label": "Pbam",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pccn",
            "label": "Pccn",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pbcm",
            "label": "Pbcm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pnnm",
            "label": "Pnnm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pmmn",
            "label": "Pmmn",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pbcn",
            "label": "Pbcn",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pbca",
            "label": "Pbca",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Pnma",
            "label": "Pnma",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Cmcm",
            "label": "Cmcm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Cmca",
            "label": "Cmca",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Cmmm",
            "label": "Cmmm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Cccm",
            "label": "Cccm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Cmma",
            "label": "Cmma",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Ccca",
            "label": "Ccca",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Fmmm",
            "label": "Fmmm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Fddd",
            "label": "Fddd",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Immm",
            "label": "Immm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Ibam",
            "label": "Ibam",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Ibcm",
            "label": "Ibcm",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_Imma",
            "label": "Imma",
            "description": "Orthorhombic, mmm // Orthorhombisch, mmm",
        },
        {
            "code": "SPACE_GROUP_P4",
            "label": "P4",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_P41",
            "label": "P4_1",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_P42",
            "label": "P4_2",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_P43",
            "label": "P4_3",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_I4",
            "label": "I4",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_I41",
            "label": "I4_1",
            "description": "Tetragonal, 4 // Tetragonal, 4",
        },
        {
            "code": "SPACE_GROUP_P-4",
            "label": "P-4",
            "description": "Tetragonal, -4 // Tetragonal, -4",
        },
        {
            "code": "SPACE_GROUP_I-4",
            "label": "I-4",
            "description": "Tetragonal, -4 // Tetragonal, -4",
        },
        {
            "code": "SPACE_GROUP_P4_m",
            "label": "P4/m",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_P42_m",
            "label": "P4_2/m",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_P4_n",
            "label": "P4/n",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_P42_n",
            "label": "P4_2/n",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_I4_m",
            "label": "I4/m",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_I41_a",
            "label": "I4_1/a",
            "description": "Tetragonal, 4/m // Tetragonal, 4/m",
        },
        {
            "code": "SPACE_GROUP_P422",
            "label": "P422",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P4212",
            "label": "P42_12",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P4122",
            "label": "P4_122",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P41212",
            "label": "P4_12_12",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P4222",
            "label": "P4222",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P42212",
            "label": "P422_12",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P4322",
            "label": "P4_322",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P43212",
            "label": "P4_32_12",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_I422",
            "label": "I422",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_I4122",
            "label": "I4_122",
            "description": "Tetragonal, 422 // Tetragonal, 422",
        },
        {
            "code": "SPACE_GROUP_P4mm",
            "label": "P4mm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P4bm",
            "label": "P4bm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P42cm",
            "label": "P4_2cm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P42nm",
            "label": "P4_2nm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P4cc",
            "label": "P4cc",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P4nc",
            "label": "P4nc",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P42mc",
            "label": "P4_2mc",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P42bc",
            "label": "P4_2bc",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_I4mm",
            "label": "I4mm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_I4cm",
            "label": "I4cm",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_I41md",
            "label": "I4_1md",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_I41cd",
            "label": "I4_1cd",
            "description": "Tetragonal, 4mm // Tetragonal, 4mm",
        },
        {
            "code": "SPACE_GROUP_P-42m",
            "label": "P-42m",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-42c",
            "label": "P-42c",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-421m",
            "label": "P-42_1m",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-421c",
            "label": "P-42_1c",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_I-42m",
            "label": "I-42m",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_I-42d",
            "label": "I-42d",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-4m2",
            "label": "P-4m2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-4c2",
            "label": "P-4c2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-4b2",
            "label": "P-4b2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P-4n2",
            "label": "P-4n2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_I-4m2",
            "label": "I-4m2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_I-4c2",
            "label": "I-4c2",
            "description": "Tetragonal, -42m // Tetragonal, -42m",
        },
        {
            "code": "SPACE_GROUP_P4_mmm",
            "label": "P4/mmm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_mcc",
            "label": "P4/mcc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_nbm",
            "label": "P4/nbm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_nnc",
            "label": "P4/nnc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_mbm",
            "label": "P4/mbm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_mnc",
            "label": "P4/mnc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_nmm",
            "label": "P4/nmm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P4_ncc",
            "label": "P4/ncc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_mmc",
            "label": "P4_2/mmc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_mcm",
            "label": "P4_2/mcm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_nbc",
            "label": "P4_2/nbc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_nnm",
            "label": "P4_2/nnm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_mbc",
            "label": "P4_2/mbc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_mnm",
            "label": "P4_2/mnm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_nmc",
            "label": "P4_2/nmc",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P42_ncm",
            "label": "P4_2/ncm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_I4_mmm",
            "label": "I4/mmm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_I4_mcm",
            "label": "I4/mcm",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_I41_amd",
            "label": "I4_1/amd",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_I41_acd",
            "label": "I4_1/acd",
            "description": "Tetragonal, 4/mmm // Tetragonal, 4/mmm",
        },
        {
            "code": "SPACE_GROUP_P3",
            "label": "P3",
            "description": "Trigonal, 3 // Trigonal, 3",
        },
        {
            "code": "SPACE_GROUP_P31",
            "label": "P3_1",
            "description": "Trigonal, 3 // Trigonal, 3",
        },
        {
            "code": "SPACE_GROUP_P32",
            "label": "P3_2",
            "description": "Trigonal, 3 // Trigonal, 3",
        },
        {
            "code": "SPACE_GROUP_R3",
            "label": "R3",
            "description": "Trigonal, 3 // Trigonal, 3",
        },
        {
            "code": "SPACE_GROUP_P-3",
            "label": "P-3",
            "description": "Trigonal, -3 // Trigonal, -3",
        },
        {
            "code": "SPACE_GROUP_R-3",
            "label": "R-3",
            "description": "Trigonal, -3 // Trigonal, -3",
        },
        {
            "code": "SPACE_GROUP_P312",
            "label": "P312",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P321",
            "label": "P321",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P3112",
            "label": "P3_112",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P3121",
            "label": "P3_121",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P3212",
            "label": "P3_212",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P3221",
            "label": "P3_221",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_R32",
            "label": "R32",
            "description": "Trigonal, 32 // Trigonal, 32",
        },
        {
            "code": "SPACE_GROUP_P3m1",
            "label": "P3m1",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_P31m",
            "label": "P31m",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_P3c1",
            "label": "P3c1",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_P31c",
            "label": "P31c",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_R3m",
            "label": "R3m",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_R3c",
            "label": "R3c",
            "description": "Trigonal, 3m // Trigonal, 3m",
        },
        {
            "code": "SPACE_GROUP_P-31m",
            "label": "P-31m",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_P-31c",
            "label": "P-31c",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_P-3m1",
            "label": "P-3m1",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_P-3c1",
            "label": "P-3c1",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_R-3m",
            "label": "R-3m",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_R-3c",
            "label": "R-3c",
            "description": "Trigonal, -3m // Trigonal, -3m",
        },
        {
            "code": "SPACE_GROUP_P6",
            "label": "P6",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P61",
            "label": "P6_1",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P65",
            "label": "P6_5",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P62",
            "label": "P6_2",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P64",
            "label": "P6_4",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P63",
            "label": "P6_3",
            "description": "Hexagonal, 6 // Hexagonal, 6",
        },
        {
            "code": "SPACE_GROUP_P-6",
            "label": "P-6",
            "description": "Hexagonal, -6 // Hexagonal, -6",
        },
        {
            "code": "SPACE_GROUP_P6_m",
            "label": "P6/m",
            "description": "Hexagonal, 6/m // Hexagonal, 6/m",
        },
        {
            "code": "SPACE_GROUP_P63_m",
            "label": "P6_3/m",
            "description": "Hexagonal, 6/m // Hexagonal, 6/m",
        },
        {
            "code": "SPACE_GROUP_P622",
            "label": "P622",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6122",
            "label": "P6_122",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6522",
            "label": "P6_522",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6222",
            "label": "P6_222",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6422",
            "label": "P6_422",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6322",
            "label": "P6_322",
            "description": "Hexagonal, 622 // Hexagonal, 622",
        },
        {
            "code": "SPACE_GROUP_P6mm",
            "label": "P6mm",
            "description": "Hexagonal, 6mm // Hexagonal, 6mm",
        },
        {
            "code": "SPACE_GROUP_P6cc",
            "label": "P6cc",
            "description": "Hexagonal, 6mm // Hexagonal, 6mm",
        },
        {
            "code": "SPACE_GROUP_P6_3cm",
            "label": "P6_3cm",
            "description": "Hexagonal, 6mm // Hexagonal, 6mm",
        },
        {
            "code": "SPACE_GROUP_P6_3mc",
            "label": "P6_3mc",
            "description": "Hexagonal, 6mm // Hexagonal, 6mm",
        },
        {
            "code": "SPACE_GROUP_P-6m2",
            "label": "P-6m2",
            "description": "Hexagonal, -6m2 // Hexagonal, -6m2",
        },
        {
            "code": "SPACE_GROUP_P-6c2",
            "label": "P-6c2",
            "description": "Hexagonal, -6m2 // Hexagonal, -6m2",
        },
        {
            "code": "SPACE_GROUP_P-62m",
            "label": "P-62m",
            "description": "Hexagonal, -6m2 // Hexagonal, -6m2",
        },
        {
            "code": "SPACE_GROUP_P-62c",
            "label": "P-62c",
            "description": "Hexagonal, -6m2 // Hexagonal, -6m2",
        },
        {
            "code": "SPACE_GROUP_P6_mmm",
            "label": "P6/mmm",
            "description": "Hexagonal, 6/mmm // Hexagonal, 6/mmm",
        },
        {
            "code": "SPACE_GROUP_P6_mcc",
            "label": "P6/mcc",
            "description": "Hexagonal, 6/mmm // Hexagonal, 6/mmm",
        },
        {
            "code": "SPACE_GROUP_P63_mcm",
            "label": "P6_3/mcm",
            "description": "Hexagonal, 6/mmm // Hexagonal, 6/mmm",
        },
        {
            "code": "SPACE_GROUP_P63_mmc",
            "label": "P6_3/mmc",
            "description": "Hexagonal, 6/mmm // Hexagonal, 6/mmm",
        },
        {
            "code": "SPACE_GROUP_P23",
            "label": "P23",
            "description": "Cubic, 23 // Kubisch, 23",
        },
        {
            "code": "SPACE_GROUP_F23",
            "label": "F23",
            "description": "Cubic, 23 // Kubisch, 23",
        },
        {
            "code": "SPACE_GROUP_I23",
            "label": "I23",
            "description": "Cubic, 23 // Kubisch, 23",
        },
        {
            "code": "SPACE_GROUP_P213",
            "label": "P2_13",
            "description": "Cubic, 23 // Kubisch, 23",
        },
        {
            "code": "SPACE_GROUP_I213",
            "label": "I2_13",
            "description": "Cubic, 23 // Kubisch, 23",
        },
        {
            "code": "SPACE_GROUP_Pm-3",
            "label": "Pm-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Pn-3",
            "label": "Pn-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Fm-3",
            "label": "Fm-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Fd-3",
            "label": "Fd-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Im-3",
            "label": "Im-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Pa-3",
            "label": "Pa-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_Ia-3",
            "label": "Ia-3",
            "description": "Cubic, m-3 // Kubisch, m-3",
        },
        {
            "code": "SPACE_GROUP_P432",
            "label": "P432",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_P4232",
            "label": "P4_232",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_F432",
            "label": "F432",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_F4132",
            "label": "F4_132",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_I432",
            "label": "I432",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_P4332",
            "label": "P4_332",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_P4132",
            "label": "P4_132",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_I4132",
            "label": "I4_132",
            "description": "Cubic, 432 // Kubisch, 432",
        },
        {
            "code": "SPACE_GROUP_P-43m",
            "label": "P-43m",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_F-43m",
            "label": "F-43m",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_I-43m",
            "label": "I-43m",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_P-43n",
            "label": "P-43n",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_F-43c",
            "label": "F-43c",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_I-43d",
            "label": "I-43d",
            "description": "Cubic, -43m // Kubisch, -43m",
        },
        {
            "code": "SPACE_GROUP_Pm-3m",
            "label": "Pm-3m",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Pn-3n",
            "label": "Pn-3n",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Pm-3n",
            "label": "Pm-3n",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Pn-3m",
            "label": "Pn-3m",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Fm-3m",
            "label": "Fm-3m",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Fm-3c",
            "label": "Fm-3c",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Fd-3m",
            "label": "Fd-3m",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Fd-3c",
            "label": "Fd-3c",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Im-3m",
            "label": "Im-3m",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
        {
            "code": "SPACE_GROUP_Ia-3d",
            "label": "Ia-3d",
            "description": "Cubic, m-3m // Kubisch, m-3m",
        },
    ]
    for i, term in enumerate(terms_space_group_vocab, 1):
        label = term["label"]
        for k in ["label", "description"]:
            v = term[k]
            minus_pos = v.find("-")
            while minus_pos > -1:
                v = v[:minus_pos] + v[minus_pos + 1] + "\u0305" + v[minus_pos + 2 :]
                term[k] = v
                minus_pos = v.find("-")
            underscore_pos = v.find("_")
            while underscore_pos > -1:
                subscript_map = {
                    "0": "₀",
                    "1": "₁",
                    "2": "₂",
                    "3": "₃",
                    "4": "₄",
                    "5": "₅",
                    "6": "₆",
                    "7": "₇",
                    "8": "₈",
                    "9": "₉",
                }
                subscript = subscript_map[v[underscore_pos + 1]]
                v = v[:underscore_pos] + subscript + v[underscore_pos + 2 :]
                term[k] = v
                underscore_pos = v.find("_")

        term["code"] = term["code"].upper()
        term["label"] = term["label"] + f" ({i})"
        term["description"] = term["description"] + f" ({i}, {label})"

    assert len(terms_space_group_vocab) == 230, "230 Space Groups"

    voc_space_group_vocab = oBis.new_vocabulary(
        code="SPACE_GROUP_VOCAB",
        description="Crystallographic space groups // Raumgruppen",
        terms=terms_space_group_vocab,
    )
    register_controlled_vocabulary(oBis, voc_space_group_vocab, terms_space_group_vocab)

    # #########################################################################
    # Plugins
    # #########################################################################

    dynamic_property_plugin_1 = oBis.new_plugin(
        name="MATERIAL_SPECIES",
        pluginType="DYNAMIC_PROPERTY",
        entityKind="SAMPLE",  # Sample meaning Object
        script=open("jython_scripts/ChemicalSpecies.py", "r").read(),
    )
    dynamic_property_plugin_1.description = (
        "Returns the number of chemical species in a material"
    )
    register_plugin(oBis, dynamic_property_plugin_1)

    # #########################################################################
    # Property Types
    # #########################################################################

    # Structure Animation
    # Rich Text has to be enabled

    pt_animation = oBis.new_property_type(
        code="STRUCTURE_ANIMATION",
        label="Preview",
        description="Animation of structure",
        dataType="MULTILINE_VARCHAR",
    )
    register_property_type(oBis, pt_animation)

    # Bonding Type (free-text)

    pt_bonding_type = oBis.new_property_type(
        code="MAT_BONDING_TYPE",
        label="Bonding Type",
        description="Material Bonding Type",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_bonding_type)

    # Flag for crystallographic prototypes

    pt_is_prototype = oBis.new_property_type(
        code="IS_PROTOTYPE",
        label="Prototype ?",
        description="Flag for crystallographic prototypes",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_is_prototype)

    # Strukturbericht

    pt_strukturbericht = oBis.new_property_type(
        code="STRUKTURBERICHT_DESIGNATION",
        label="Strukturbericht",
        description="Strukturbericht designation",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_strukturbericht)

    # Phase code

    pt_phase_code = oBis.new_property_type(
        code="PHASE_CODE",
        label="Phase",
        description="Phase code",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_phase_code)

    # Pearson Symbol

    pt_pearson_symbol = oBis.new_property_type(
        code="PEARSON_SYMBOL",
        label="Pearson Symbol",
        description="Pearson Symbol",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_pearson_symbol)

    # Space group short name

    pt_space_group = oBis.new_property_type(
        code="SPACE_GROUP_SHORT",
        label="Space Group",
        description="Space Group",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_space_group_vocab.code,
    )
    register_property_type(oBis, pt_space_group)

    # Point Group

    pt_point_group = oBis.new_property_type(
        code="POINT_GROUP",
        label="Point Group",
        description="Point Group (short international notation)",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_point_group.code,
    )
    register_property_type(oBis, pt_point_group)

    # Space Group International Number

    pt_space_group_number = oBis.new_property_type(
        code="SPACE_GROUP",
        label="Space Group",
        description="Space Group (International Number)",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_space_group_number)

    # Bravais Lattice

    pt_bravais_lattice = oBis.new_property_type(
        code="BRAVAIS_LATTICE",
        label="Bravais Lattice",
        description="Bravais lattice",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_bravais_lattice_vocab.code,
    )
    register_property_type(oBis, pt_bravais_lattice)

    # Crystal System

    pt_crystal_system = oBis.new_property_type(
        code="CRYSTAL_SYSTEM",
        label="Crystal System",
        description="Crystal System",
        dataType="CONTROLLEDVOCABULARY",
        vocabulary=voc_crystal_system.code,
    )
    register_property_type(oBis, pt_crystal_system)

    # Chemical Composition
    # Python Dict exported as str ({'Cu': 66.67, 'Mg': 33.33}, {'Fe': 53.85, 'W': 46.15}, etc.)

    pt_compo_at = oBis.new_property_type(
        code="COMPO_ATOMIC_PERCENT",
        label="Composition (at.)",
        description="Composition in atomic percent",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_compo_at)

    # Stoichiometry, Anonymous Formula (AB2, A6B7, etc.)

    pt_stoichiometry = oBis.new_property_type(
        code="STOICHIOMETRY",
        label="Stoichiometry",
        description="Stoichiometry (increasing content)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_stoichiometry)

    # Reduced Formula (Mg1 Cu2, Fe7 W6, etc.)

    pt_reduced_chemical_formula = oBis.new_property_type(
        code="REDUCED_CHEMICAL_FORMULA",
        label="Reduced Formula",
        description="Reduced chemical formula (space-separated)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_reduced_chemical_formula)

    # Full Formula (Mg2 Cu4, Fe7 W6, etc.)

    pt_full_chemical_formula = oBis.new_property_type(
        code="FULL_CHEMICAL_FORMULA",
        label="Formula",
        description="Full chemical formula (space-separated)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_full_chemical_formula)

    # Chemical System (Cu-Mg, Fe-W, etc.)

    pt_chemical_system = oBis.new_property_type(
        code="CHEMICAL_SYSTEM",
        label="Chem. Sys.",
        description="Chemical System (alphabetically sorted)",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_chemical_system)

    # Number of chemical species

    pt_number_of_species = oBis.new_property_type(
        code="NUMBER_OF_SPECIES",
        label="Number of Chemical Species",
        description="Number of chemical species / elements contained in simulation box",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_number_of_species)

    # Coordination number

    pt_coordination_number = oBis.new_property_type(
        code="COORDINATION_NUMBER",
        label="CN",
        description="Coordination number",
        dataType="INTEGER",
    )
    register_property_type(oBis, pt_coordination_number)

    # Experimentally confirmed ?

    pt_exp_confirm = oBis.new_property_type(
        code="EXP_CONFIRM",
        label="Exp. confirmed ?",
        description="Experimentally confirmed",
        dataType="BOOLEAN",
    )
    register_property_type(oBis, pt_exp_confirm)

    # Link to external database (Materials Project, PubChem, ...)

    pt_external_link = oBis.new_property_type(
        code="EXTERNAL_LINK",
        label="Link",
        description="Link",
        dataType="HYPERLINK",
    )
    register_property_type(oBis, pt_external_link)

    # Werkstoffnummer

    pt_material_number = oBis.new_property_type(
        code="MATERIAL_NUMBER",
        label="Werkstoff-Nr.",
        description="Werkstoffnummer // Material Number",
        dataType="VARCHAR",
    )
    register_property_type(oBis, pt_material_number)

    # Composition Range

    pt_elements_range = oBis.new_property_type(
        code="ELEMENTS_RANGE",
        label="Composition Range",
        description="Composition Range",
        dataType="XML",
    )
    register_property_type(oBis, pt_elements_range)

    # #########################################################################
    # Objects
    # #########################################################################

    # Material
    obj_mat = oBis.new_object_type(
        code=code_material,
        generatedCodePrefix="MAT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_mat)

    obj_mat = oBis.get_object_type(code_material)
    obj_mat.description = "Material"
    register_object_type(oBis, obj_mat)

    obj_mat = oBis.get_object_type(code_material)

    # Define display order
    # (PROPERTY_CODE, IS_MANADATORY, PLUGIN_CODE)
    sec2props_mat = {
        "General": [
            ("$NAME", 1, None),
            ("DESCRIPTION", 0, None),
            ("MAT_BONDING_TYPE", 0, None),
            ("COMPO_ATOMIC_PERCENT", 0, None),
            ("STOICHIOMETRY", 0, None),
            ("CHEMICAL_SYSTEM", 0, None),
            ("NUMBER_OF_SPECIES", 0, "MATERIAL_SPECIES"),
            ("EXTERNAL_LINK", 0, None),  # PubChem
            ("COMMENTS", 0, None),
        ],
    }
    assign_property_types(obj_mat, sec2props_mat)

    # Amorphous Material
    obj_amorphous_mat = oBis.new_object_type(
        code=code_amorphous_material,
        generatedCodePrefix="AMORMAT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_amorphous_mat)

    obj_amorphous_mat = oBis.get_object_type(code_amorphous_material)
    obj_amorphous_mat.description = "Amorphous Material, http://purls.helmholtz-metadaten.de/cdos/cdco/AmorphousMaterial"
    register_object_type(oBis, obj_amorphous_mat)

    obj_amorphous_mat = oBis.get_object_type(code_amorphous_material)

    # Define display order
    # (PROPERTY_CODE, IS_MANADATORY, PLUGIN_CODE)
    sec2props_amorphous_mat = {
        "General": [
            ("$NAME", 1, None),
            ("ALT_NAME", 0, None),
            ("COMPO_ATOMIC_PERCENT", 0, None),
            ("STOICHIOMETRY", 0, None),
            ("REDUCED_CHEMICAL_FORMULA", 0, None),
            ("FULL_CHEMICAL_FORMULA", 0, None),
            ("CHEMICAL_SYSTEM", 0, None),
            ("NUMBER_OF_SPECIES", 0, "MATERIAL_SPECIES"),
            ("COMMENTS", 0, None),
        ],
    }
    assign_property_types(obj_amorphous_mat, sec2props_amorphous_mat)

    # Crystalline Material
    obj_crys_mat = oBis.new_object_type(
        code=code_crystalline_material,
        generatedCodePrefix="CRYSMAT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_crys_mat)

    obj_crys_mat = oBis.get_object_type(code_crystalline_material)
    obj_crys_mat.description = "Crystalline Material, http://purls.helmholtz-metadaten.de/cdos/cdco/CrystallineMaterial"
    register_object_type(oBis, obj_crys_mat)

    obj_crys_mat = oBis.get_object_type(code_crystalline_material)

    # Define display order
    # (PROPERTY_CODE, IS_MANDATORY, PLUGIN_CODE)
    sec2props_crys_mat = {
        "General": [
            ("$NAME", 1, None),
            ("ALT_NAME", 0, None),
            ("COMPO_ATOMIC_PERCENT", 0, None),
            ("STOICHIOMETRY", 0, None),
            ("REDUCED_CHEMICAL_FORMULA", 0, None),
            ("FULL_CHEMICAL_FORMULA", 0, None),
            ("CHEMICAL_SYSTEM", 0, None),
            ("NUMBER_OF_SPECIES", 0, "MATERIAL_SPECIES"),
            ("DESCRIPTION", 0, None),
            ("MAT_BONDING_TYPE", 0, None),
            ("STRUCTURE_ANIMATION", 0, None),
            ("COORDINATION_NUMBER", 0, None),
            ("EXP_CONFIRM", 0, None),
            ("COMMENTS", 0, None),
        ],
        "Crystallographic Information": [
            ("CRYSTAL_SYSTEM", 0, None),
            ("SPACE_GROUP_SHORT", 0, None),
            ("STRUKTURBERICHT_DESIGNATION", 0, None),
            ("BRAVAIS_LATTICE", 0, None),
            ("PEARSON_SYMBOL", 0, None),
            ("IS_PROTOTYPE", 0, None),
            ("EXTERNAL_LINK", 0, None),  # Materials Project
            ("PHASE_CODE", 0, None),
        ],
    }
    assign_property_types(obj_crys_mat, sec2props_crys_mat)

    # Engineered Material
    obj_eng_mat = oBis.new_object_type(
        code=code_engineered_material,
        generatedCodePrefix="ENGMAT",
        autoGeneratedCode=True,
    )
    register_object_type(oBis, obj_eng_mat)

    obj_eng_mat = oBis.get_object_type(code_engineered_material)
    obj_eng_mat.description = "Material that is output of a Manufacturing Process, https://w3id.org/pmd/co/PMD_0000002"
    register_object_type(oBis, obj_eng_mat)

    obj_eng_mat = oBis.get_object_type(code_engineered_material)

    # Define display order
    # (PROPERTY_CODE, IS_MANADATORY, PLUGIN_CODE)
    sec2props_eng_mat = {
        "General": [
            ("$NAME", 1, None),
            ("ALT_NAME", 0, None),
            ("MATERIAL_NUMBER", 0, None),
            ("DESCRIPTION", 0, None),
            ("COMMENTS", 0, None),
            ("ELEMENTS_RANGE", 0, None),
        ],
    }
    assign_property_types(obj_eng_mat, sec2props_eng_mat)
