"""
Systematic Error Budget Analysis
=================================

Objective: Quantify EVERY source of systematic error in the torsion balance experiment
and calculate the total systematic uncertainty.

This preempts the "you forgot about X" criticism by showing you've thought through
all possible error sources and have mitigation strategies.

Error Sources:
1. Fiber torsion constant uncertainty
2. Test mass position/geometry uncertainty
3. Magnetic field gradient inhomogeneity
4. Temperature drift and fluctuations
5. Seismic coupling
6. Electrostatic charging
7. Residual gas pressure
8. Gravitational gradient from building
9. Casimir effect
10. Fiber anisotropy
"""

import numpy as np
import json
from dataclasses import dataclass
from typing import List, Dict

# Physical Constants
K_B = 1.380649e-23  # J/K
EPSILON_0 = 8.854187817e-12  # F/m
HBAR = 1.054571817e-34  # J·s
C_LIGHT = 299792458  # m/s

# Experimental Parameters
T_CRYO = 4.0  # K
M_TEST = 1e-3  # kg
R_TEST = 0.01  # m
FIBER_KAPPA = 1.2e-11  # N·m/rad
FIBER_LENGTH = 0.1  # m
FIBER_DIAMETER = 50e-6  # m
A_TARGET = 1e-10  # m/s²
A_SIGNAL = 9e-11  # m/s²


@dataclass
class SystematicError:
    """A single systematic error source."""
    name: str
    description: str
    magnitude: float  # Fractional error or absolute error
    units: str
    contribution_to_signal: float  # Error in measured acceleration (m/s²)
    mitigation_strategy: str
    residual_after_mitigation: float  # Remaining error after mitigation


class SystematicErrorBudget:
    """Calculate and track all systematic errors."""
    
    def __init__(self):
        self.errors: List[SystematicError] = []
        
    def add_error(self, error: SystematicError):
        """Add an error source to the budget."""
        self.errors.append(error)
    
    def calculate_fiber_kappa_error(self) -> SystematicError:
        """
        Fiber torsion constant uncertainty.
        
        Sources:
        - Manufacturing tolerance on diameter (±1 μm)
        - Shear modulus uncertainty (±2%)
        - Length measurement (±0.1 mm)
        """
        # κ = (π G d⁴) / (32 L)
        # Fractional error: Δκ/κ = 4(Δd/d) + (ΔG/G) + (ΔL/L)
        
        diameter_tolerance = 1e-6  # m
        diameter_error = 4 * (diameter_tolerance / FIBER_DIAMETER)
        
        shear_modulus_error = 0.02  # 2%
        
        length_tolerance = 1e-4  # m
        length_error = length_tolerance / FIBER_LENGTH
        
        total_fractional_error = diameter_error + shear_modulus_error + length_error
        
        # Impact on measured acceleration
        # a = (κ θ) / (m r), so Δa/a = Δκ/κ
        acceleration_error = total_fractional_error * A_SIGNAL
        
        # Mitigation: Calibrate κ using known torque
        residual_error = acceleration_error * 0.1  # 10x reduction via calibration
        
        return SystematicError(
            name="Fiber Torsion Constant",
            description="Uncertainty in κ from manufacturing tolerances",
            magnitude=total_fractional_error,
            units="fractional",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="Calibrate κ using precision torque standard (e.g., electrostatic comb drive)",
            residual_after_mitigation=residual_error
        )
    
    def calculate_mass_geometry_error(self) -> SystematicError:
        """
        Test mass position and geometry uncertainty.
        
        Sources:
        - Position uncertainty (±0.1 mm)
        - Mass uncertainty (±0.01 mg)
        - Radius uncertainty (±0.05 mm)
        """
        # Torque: τ = m r a
        # Fractional error: Δτ/τ = (Δm/m) + (Δr/r)
        
        mass_tolerance = 1e-8  # kg (0.01 mg)
        mass_error = mass_tolerance / M_TEST
        
        radius_tolerance = 5e-5  # m (0.05 mm)
        radius_error = radius_tolerance / R_TEST
        
        total_fractional_error = mass_error + radius_error
        
        acceleration_error = total_fractional_error * A_SIGNAL
        
        # Mitigation: Precision mass measurement + optical position tracking
        residual_error = acceleration_error * 0.05  # 20x reduction
        
        return SystematicError(
            name="Test Mass Geometry",
            description="Uncertainty in mass and position",
            magnitude=total_fractional_error,
            units="fractional",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="Precision balance (±1 μg) + optical interferometry for position",
            residual_after_mitigation=residual_error
        )
    
    def calculate_magnetic_field_error(self) -> SystematicError:
        """
        Magnetic field gradient inhomogeneity.
        
        Sources:
        - Coil geometry imperfections (±2%)
        - Current stability (±0.1%)
        - Spatial inhomogeneity (±3%)
        """
        # Force: F = χ V (∇B) B / μ₀
        # Fractional error in gradient
        
        coil_geometry_error = 0.02
        current_stability_error = 0.001
        spatial_inhomogeneity = 0.03
        
        total_fractional_error = np.sqrt(coil_geometry_error**2 + 
                                         current_stability_error**2 + 
                                         spatial_inhomogeneity**2)
        
        acceleration_error = total_fractional_error * A_SIGNAL
        
        # Mitigation: Map field with Hall probe + active current stabilization
        residual_error = acceleration_error * 0.2  # 5x reduction
        
        return SystematicError(
            name="Magnetic Field Gradient",
            description="Inhomogeneity and drift in applied field",
            magnitude=total_fractional_error,
            units="fractional",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="3D field mapping + PID-controlled current source (±0.01%)",
            residual_after_mitigation=residual_error
        )
    
    def calculate_temperature_drift_error(self) -> SystematicError:
        """
        Temperature drift and fluctuations.
        
        Effects:
        - Thermal expansion of fiber (changes κ)
        - Thermal noise increase
        - Outgassing from warm surfaces
        """
        # Temperature coefficient of κ for quartz: ~1×10⁻⁴ K⁻¹
        temp_coeff_kappa = 1e-4  # K⁻¹
        
        # Expected drift: ±0.001 K/hour
        temp_drift = 0.001  # K
        
        fractional_kappa_change = temp_coeff_kappa * temp_drift
        
        acceleration_error = fractional_kappa_change * A_SIGNAL
        
        # Mitigation: Active temperature control (±0.0001 K)
        residual_drift = 0.0001  # K
        residual_error = temp_coeff_kappa * residual_drift * A_SIGNAL
        
        return SystematicError(
            name="Temperature Drift",
            description="Thermal effects on fiber and test mass",
            magnitude=temp_drift,
            units="K",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="Active PID temperature control with ±0.0001 K stability",
            residual_after_mitigation=residual_error
        )
    
    def calculate_seismic_coupling_error(self) -> SystematicError:
        """
        Seismic vibrations coupling to torsion balance.
        
        Mechanism: Building vibrations create apparent torque via:
        - Direct coupling to fiber
        - Magnetic field modulation (if coils vibrate)
        """
        # Typical lab seismic noise: 10⁻⁷ m/s²/√Hz at 1 Hz
        seismic_amplitude = 1e-7  # m/s²/√Hz
        
        # Measurement bandwidth: ~0.1 Hz
        bandwidth = 0.1  # Hz
        
        # Integrated seismic acceleration
        a_seismic_rms = seismic_amplitude * np.sqrt(bandwidth)
        
        # Coupling coefficient (depends on geometry, typically ~0.01)
        coupling_coeff = 0.01
        
        acceleration_error = coupling_coeff * a_seismic_rms
        
        # Mitigation: Passive vibration isolation (factor of 100 at 1 Hz)
        isolation_factor = 100
        residual_error = acceleration_error / isolation_factor
        
        return SystematicError(
            name="Seismic Coupling",
            description="Building vibrations coupling to measurement",
            magnitude=a_seismic_rms,
            units="m/s² RMS",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="Multi-stage passive isolation (spring-mass stack, factor of 100)",
            residual_after_mitigation=residual_error
        )
    
    def calculate_electrostatic_error(self) -> SystematicError:
        """
        Electrostatic charging and forces.
        
        Sources:
        - Triboelectric charging during assembly
        - Cosmic ray ionization
        - Patch potentials on surfaces
        """
        # Assume test mass acquires ~100 elementary charges
        q_test = 100 * 1.602e-19  # C
        
        # Stray electric field in cryostat: ~100 V/m
        E_stray = 100  # V/m
        
        # Electrostatic force
        F_electrostatic = q_test * E_stray
        
        # Acceleration
        a_electrostatic = F_electrostatic / M_TEST
        
        # Mitigation: UV discharge + conductive coating
        discharge_factor = 100  # Reduce charge by 100x
        residual_error = a_electrostatic / discharge_factor
        
        return SystematicError(
            name="Electrostatic Charging",
            description="Stray charges creating spurious forces",
            magnitude=q_test,
            units="C",
            contribution_to_signal=a_electrostatic,
            mitigation_strategy="UV discharge before cooldown + conductive (gold) coating on surfaces",
            residual_after_mitigation=residual_error
        )
    
    def calculate_residual_gas_error(self) -> SystematicError:
        """
        Residual gas pressure effects.
        
        Effects:
        - Gas damping (changes Q-factor)
        - Molecular bombardment (noise)
        - Outgassing (pressure drift)
        """
        # Target vacuum: 10⁻⁶ mbar
        pressure = 1e-6  # mbar = 0.1 Pa
        
        # Mean free path at this pressure: ~10 m (>> apparatus size)
        # Damping is negligible
        
        # Molecular bombardment creates force noise
        # F_rms = sqrt(2 k_B T P A / c_thermal)
        # where A is test mass area, c_thermal is mean thermal velocity
        
        area = 4 * np.pi * R_TEST**2  # m²
        c_thermal = np.sqrt(8 * K_B * T_CRYO / (np.pi * 4 * 1.66e-27))  # m/s (assuming He)
        
        F_rms = np.sqrt(2 * K_B * T_CRYO * (pressure * 100) * area / c_thermal)  # N
        a_gas = F_rms / M_TEST
        
        # This is white noise, not systematic error, but include for completeness
        # Mitigation: Better vacuum (10⁻⁸ mbar)
        residual_pressure = 1e-8  # mbar
        residual_error = a_gas * np.sqrt(residual_pressure / pressure)
        
        return SystematicError(
            name="Residual Gas Pressure",
            description="Molecular bombardment and damping",
            magnitude=pressure,
            units="mbar",
            contribution_to_signal=a_gas,
            mitigation_strategy="Cryopumping + ion pump to achieve <10⁻⁸ mbar",
            residual_after_mitigation=residual_error
        )
    
    def calculate_gravitational_gradient_error(self) -> SystematicError:
        """
        Gravitational gradient from building mass distribution.
        
        Nearby masses (walls, equipment, people) create tidal forces.
        """
        # Gravitational gradient from a mass M at distance r:
        # ∇g ≈ 2 G M / r³
        
        G = 6.674e-11  # m³/(kg·s²)
        
        # Assume 1000 kg mass (person + equipment) at 1 m distance
        M_nearby = 1000  # kg
        r_nearby = 1.0  # m
        
        grad_g = 2 * G * M_nearby / r_nearby**3  # s⁻²
        
        # Tidal acceleration across test mass (size ~2 cm)
        a_tidal = grad_g * (2 * R_TEST)
        
        # Mitigation: Symmetry + shielding (dense walls)
        # Also: modulate the signal (tidal force is DC, signal is AC)
        residual_error = a_tidal * 0.01  # 100x reduction via AC modulation
        
        return SystematicError(
            name="Gravitational Gradient",
            description="Tidal forces from nearby masses",
            magnitude=grad_g,
            units="s⁻²",
            contribution_to_signal=a_tidal,
            mitigation_strategy="AC modulation of signal (tidal force is DC) + symmetric mass distribution",
            residual_after_mitigation=residual_error
        )
    
    def calculate_casimir_effect_error(self) -> SystematicError:
        """
        Casimir force between test mass and nearby surfaces.
        
        Only relevant if surfaces are very close (<1 μm).
        """
        # Casimir force between parallel plates:
        # F = (π² ℏ c A) / (240 d⁴)
        
        # Assume nearest surface is 1 mm away (fiber)
        d_surface = 1e-3  # m
        
        # Area of interaction (rough estimate)
        A_casimir = np.pi * (FIBER_DIAMETER / 2)**2
        
        F_casimir = (np.pi**2 * HBAR * C_LIGHT * A_casimir) / (240 * d_surface**4)
        a_casimir = F_casimir / M_TEST
        
        # This is negligible at mm distances
        # No mitigation needed
        residual_error = a_casimir
        
        return SystematicError(
            name="Casimir Effect",
            description="Quantum vacuum forces",
            magnitude=d_surface,
            units="m (separation)",
            contribution_to_signal=a_casimir,
            mitigation_strategy="Not needed (effect is negligible at >1 mm separation)",
            residual_after_mitigation=residual_error
        )
    
    def calculate_fiber_anisotropy_error(self) -> SystematicError:
        """
        Fiber torsion constant anisotropy.
        
        Real fibers are not perfectly isotropic; κ varies with angle.
        """
        # Typical anisotropy for drawn quartz fiber: ~1%
        anisotropy = 0.01
        
        # This creates a periodic signal at 2× rotation frequency
        # Can be distinguished from DC signal
        
        acceleration_error = anisotropy * A_SIGNAL
        
        # Mitigation: Average over multiple rotation angles
        residual_error = acceleration_error / np.sqrt(10)  # 10 angle measurements
        
        return SystematicError(
            name="Fiber Anisotropy",
            description="Angular dependence of torsion constant",
            magnitude=anisotropy,
            units="fractional",
            contribution_to_signal=acceleration_error,
            mitigation_strategy="Measure at 10 different rotation angles and average",
            residual_after_mitigation=residual_error
        )
    
    def calculate_total_systematic_uncertainty(self) -> Dict:
        """Calculate total systematic uncertainty from all sources."""
        # Add all error sources
        self.add_error(self.calculate_fiber_kappa_error())
        self.add_error(self.calculate_mass_geometry_error())
        self.add_error(self.calculate_magnetic_field_error())
        self.add_error(self.calculate_temperature_drift_error())
        self.add_error(self.calculate_seismic_coupling_error())
        self.add_error(self.calculate_electrostatic_error())
        self.add_error(self.calculate_residual_gas_error())
        self.add_error(self.calculate_gravitational_gradient_error())
        self.add_error(self.calculate_casimir_effect_error())
        self.add_error(self.calculate_fiber_anisotropy_error())
        
        # Calculate total (add in quadrature)
        total_before_mitigation = np.sqrt(sum(e.contribution_to_signal**2 for e in self.errors))
        total_after_mitigation = np.sqrt(sum(e.residual_after_mitigation**2 for e in self.errors))
        
        # Signal-to-systematic ratio
        signal_to_systematic_before = A_SIGNAL / total_before_mitigation
        signal_to_systematic_after = A_SIGNAL / total_after_mitigation
        
        return {
            'total_systematic_before_mitigation': total_before_mitigation,
            'total_systematic_after_mitigation': total_after_mitigation,
            'signal_to_systematic_before': signal_to_systematic_before,
            'signal_to_systematic_after': signal_to_systematic_after,
            'individual_errors': [
                {
                    'name': e.name,
                    'description': e.description,
                    'magnitude': float(e.magnitude),
                    'units': e.units,
                    'contribution': float(e.contribution_to_signal),
                    'mitigation': e.mitigation_strategy,
                    'residual': float(e.residual_after_mitigation)
                }
                for e in self.errors
            ]
        }


def run_systematic_error_analysis():
    """Run complete systematic error budget analysis."""
    
    print("=" * 80)
    print("SYSTEMATIC ERROR BUDGET ANALYSIS")
    print("=" * 80)
    print(f"\nTarget Signal: {A_SIGNAL:.2e} m/s²")
    print(f"Measurement Duration: 1 hour")
    print(f"Temperature: {T_CRYO} K\n")
    
    budget = SystematicErrorBudget()
    results = budget.calculate_total_systematic_uncertainty()
    
    print("-" * 80)
    print("INDIVIDUAL ERROR SOURCES")
    print("-" * 80)
    
    # Sort by contribution (before mitigation)
    sorted_errors = sorted(results['individual_errors'], 
                          key=lambda x: x['contribution'], 
                          reverse=True)
    
    for i, error in enumerate(sorted_errors, 1):
        print(f"\n{i}. {error['name']}")
        print(f"   Description: {error['description']}")
        print(f"   Magnitude: {error['magnitude']:.2e} {error['units']}")
        print(f"   Contribution to Signal: {error['contribution']:.2e} m/s²")
        print(f"   Mitigation: {error['mitigation']}")
        print(f"   Residual After Mitigation: {error['residual']:.2e} m/s²")
    
    print("\n" + "=" * 80)
    print("TOTAL SYSTEMATIC UNCERTAINTY")
    print("=" * 80)
    
    print(f"\nBefore Mitigation:")
    print(f"  Total Systematic Error: {results['total_systematic_before_mitigation']:.2e} m/s²")
    print(f"  Signal/Systematic Ratio: {results['signal_to_systematic_before']:.2f}")
    print(f"  Fractional Uncertainty: {results['total_systematic_before_mitigation'] / A_SIGNAL * 100:.1f}%")
    
    print(f"\nAfter Mitigation:")
    print(f"  Total Systematic Error: {results['total_systematic_after_mitigation']:.2e} m/s²")
    print(f"  Signal/Systematic Ratio: {results['signal_to_systematic_after']:.2f}")
    print(f"  Fractional Uncertainty: {results['total_systematic_after_mitigation'] / A_SIGNAL * 100:.1f}%")
    
    print("\n" + "=" * 80)
    print("ASSESSMENT")
    print("=" * 80)
    
    if results['signal_to_systematic_after'] >= 10:
        print(f"\n✅ EXCELLENT: Signal/Systematic = {results['signal_to_systematic_after']:.1f} (>10)")
        print("   Systematic errors are well-controlled and negligible compared to signal.")
    elif results['signal_to_systematic_after'] >= 3:
        print(f"\n✅ GOOD: Signal/Systematic = {results['signal_to_systematic_after']:.1f} (>3)")
        print("   Systematic errors are manageable.")
    else:
        print(f"\n⚠️  MARGINAL: Signal/Systematic = {results['signal_to_systematic_after']:.1f} (<3)")
        print("   Systematic errors may dominate. Additional mitigation needed.")
    
    # Identify dominant error sources
    print("\n📊 DOMINANT ERROR SOURCES (After Mitigation):")
    sorted_residual = sorted(results['individual_errors'], 
                            key=lambda x: x['residual'], 
                            reverse=True)
    
    for i, error in enumerate(sorted_residual[:3], 1):
        contribution_pct = (error['residual'] / results['total_systematic_after_mitigation']) * 100
        print(f"  {i}. {error['name']}: {error['residual']:.2e} m/s² ({contribution_pct:.1f}% of total)")
    
    # Save results
    output_path = '/home/shri/Desktop/Tortion Balance/simulations/systematic_error_budget.json'
    with open(output_path, 'w') as f:
        json.dump(results, f, indent=2)
    
    print(f"\n✓ Results saved to: {output_path}")
    
    return results


if __name__ == "__main__":
    results = run_systematic_error_analysis()
