import numpy as np
import pandas as pd
import ProFAST


def create_and_populate_profast(pf_config):
    """Create ProFAST object and populate it with inputs.

    Args:
        pf_config (dict): populated dictionary of ProFAST inputs.

    Returns:
        ProFAST object: profast object initialized with data from pf_config
    """
    pf = ProFAST.ProFAST()
    config_keys = list(pf_config.keys())
    if "params" in config_keys:
        params = pf_config["params"]
        for i in params:
            pf.set_params(i, params[i])

    if "feedstocks" in config_keys:
        variables = pf_config["feedstocks"]
        for i in variables:
            pf.add_feedstock(
                i,
                variables[i]["usage"],
                variables[i]["unit"],
                variables[i]["cost"],
                variables[i]["escalation"],
            )

    if "capital_items" in config_keys:
        variables = pf_config["capital_items"]
        for i in variables:
            pf.add_capital_item(
                i,
                variables[i]["cost"],
                variables[i]["depr_type"],
                variables[i]["depr_period"],
                variables[i]["refurb"],
            )

    if "fixed_costs" in config_keys:
        variables = pf_config["fixed_costs"]
        for i in variables:
            pf.add_fixed_cost(
                i,
                variables[i]["usage"],
                variables[i]["unit"],
                variables[i]["cost"],
                variables[i]["escalation"],
            )

    if "coproducts" in config_keys:
        variables = pf_config["coproducts"]
        for i in variables:
            pf.add_coproduct(
                i,
                variables[i]["usage"],
                variables[i]["unit"],
                variables[i]["cost"],
                variables[i]["escalation"],
            )

    if "incentives" in config_keys:
        variables = pf_config["incentives"]
        for i in variables:
            pf.add_incentive(
                i,
                variables[i]["value"],
                variables[i]["decay"],
                variables[i]["sunset_years"],
                variables[i]["tax_credit"],
            )
    return pf


def run_profast(pf):
    """Simulate financials with ProFAST

    Args:
        pf (ProFAST object): populated ProFAST object

    Returns:
        3-element tuple containing

        - **sol** (dict): solved price generated by ProFAST
        - **summary** (dict): summary vals generated by ProFAST
        - **price_breakdown** (pd.DataFrame): price breakdown generated by ProFAST
    """
    sol = pf.solve_price()
    summary = pf.get_summary_vals()
    price_breakdown = pf.get_cost_breakdown()
    return sol, summary, price_breakdown


def make_price_breakdown(price_breakdown, pf_config):
    """Make a custom price breakdown of primary cost items in $/unit of commodity.
        Distributes non-component financial contributions (remaining financials)
        across primary cost items as weighted by cost item capex.

    Args:
        price_breakdown (pd.DataFrame): price breakdown generated by ProFAST
        pf_config (dict): dictionary of ProFAST inputs.
            can be used to recreate replica of ProFAST simulation.

    Returns:
        2-element tuple containing

        - **full_price_breakdown** (dict): dictionary of each item's contribution to overall LCO.
        - **lco_check** (float): if ``lcoh_check==sol['price']``
            then the results of ``full_price_breakdown`` are accurate
    """
    price_breakdown_feedstocks = {}
    price_breakdown_capex = {}
    price_breakdown_fixed_cost = {}
    full_price_breakdown = {}
    lco_str = "LCO{}".format(pf_config["params"]["commodity"]["name"][0].upper())
    lco_units = "$/{}".format(pf_config["params"]["commodity"]["unit"])
    config_keys = list(pf_config.keys())
    if "capital_items" in config_keys:
        capital_items = pf_config["capital_items"]
        total_price_capex = 0
        capex_fraction = {}
        for item in capital_items:
            total_price_capex += price_breakdown.loc[
                price_breakdown["Name"] == item, "NPV"
            ].tolist()[0]
        for item in capital_items:
            capex_fraction[item] = (
                price_breakdown.loc[price_breakdown["Name"] == item, "NPV"].tolist()[0]
                / total_price_capex
            )
    cap_expense = (
        price_breakdown.loc[price_breakdown["Name"] == "Repayment of debt", "NPV"].tolist()[0]
        + price_breakdown.loc[price_breakdown["Name"] == "Interest expense", "NPV"].tolist()[0]
        + price_breakdown.loc[price_breakdown["Name"] == "Dividends paid", "NPV"].tolist()[0]
        - price_breakdown.loc[price_breakdown["Name"] == "Inflow of debt", "NPV"].tolist()[0]
        - price_breakdown.loc[price_breakdown["Name"] == "Inflow of equity", "NPV"].tolist()[0]
        - price_breakdown.loc[
            price_breakdown["Name"] == "One time capital incentive", "NPV"
        ].tolist()[0]
    )
    remaining_financial = (
        price_breakdown.loc[price_breakdown["Name"] == "Non-depreciable assets", "NPV"].tolist()[0]
        + price_breakdown.loc[price_breakdown["Name"] == "Cash on hand reserve", "NPV"].tolist()[0]
        + price_breakdown.loc[price_breakdown["Name"] == "Property insurance", "NPV"].tolist()[0]
        - price_breakdown.loc[
            price_breakdown["Name"] == "Sale of non-depreciable assets", "NPV"
        ].tolist()[0]
        - price_breakdown.loc[price_breakdown["Name"] == "Cash on hand recovery", "NPV"].tolist()[0]
    )

    if "capital_items" in config_keys:
        capital_items = pf_config["capital_items"]
        for item in capital_items:
            key_name = f"{lco_str}: {item} ({lco_units})"
            price_breakdown_capex[key_name] = (
                price_breakdown.loc[price_breakdown["Name"] == item, "NPV"].tolist()[0]
                + cap_expense * capex_fraction[item]
            )
        full_price_breakdown.update(price_breakdown_capex)

    if "fixed_costs" in config_keys:
        fixed_items = pf_config["fixed_costs"]
        for item in fixed_items:
            key_name = f"{lco_str}: {item} ({lco_units})"
            price_breakdown_fixed_cost[key_name] = price_breakdown.loc[
                price_breakdown["Name"] == item, "NPV"
            ].tolist()[0]
        full_price_breakdown.update(price_breakdown_fixed_cost)

    if "feedstocks" in config_keys:
        feedstock_items = pf_config["feedstocks"]
        for item in feedstock_items:
            key_name = f"{lco_str}: {item} ({lco_units})"
            price_breakdown_feedstocks[key_name] = price_breakdown.loc[
                price_breakdown["Name"] == item, "NPV"
            ].tolist()[0]
        full_price_breakdown.update(price_breakdown_feedstocks)

    price_breakdown_taxes = (
        price_breakdown.loc[price_breakdown["Name"] == "Income taxes payable", "NPV"].tolist()[0]
        - price_breakdown.loc[price_breakdown["Name"] == "Monetized tax losses", "NPV"].tolist()[0]
    )

    if pf_config["params"]["general inflation rate"] > 0:
        price_breakdown_taxes = (
            price_breakdown_taxes
            + price_breakdown.loc[
                price_breakdown["Name"] == "Capital gains taxes payable", "NPV"
            ].tolist()[0]
        )

    full_price_breakdown[f"{lco_str}: Taxes ({lco_units})"] = price_breakdown_taxes
    full_price_breakdown[f"{lco_str}: Finances ({lco_units})"] = remaining_financial
    lco_check = sum(list(full_price_breakdown.values()))
    full_price_breakdown[f"{lco_str}: Total ({lco_units})"] = lco_check

    return full_price_breakdown, lco_check


def create_years_of_operation(
    plant_life_years, financial_analysis_start_year, installation_period_months
):
    """Create list of years of operation.

    Args:
        plant_life_years (int): plant life duration in years
        financial_analysis_start_year (int): year to start analysis.
        installation_period_months (float | int): installation period in months

    Returns:
        list[str]: list of years of operation.
    """
    operation_start_year = financial_analysis_start_year + (installation_period_months / 12)
    years_of_operation = np.arange(
        int(operation_start_year), int(operation_start_year + plant_life_years), 1
    )
    year_keys = [f"{y}" for y in years_of_operation]
    return year_keys


def format_profast_price_breakdown_per_year(price_breakdown):
    """
    Formats a price breakdown DataFrame to expand yearly amounts into separate columns.

    Args:
        price_breakdown (pd.DataFrame): A DataFrame containing "Type", "Name", "Amount", and "NPV"
            - "Amount" should be an array-like object per row, representing values for each year.
    Returns:
        pd.DataFrame: A formatted DataFrame with columns:
            - "Type"
            - "Name"
            - "Year {i} Amount" for each year (e.g., "Year 0 Amount", "Year 1 Amount", ...)
            - "NPV"
        Each row corresponds to an entry in the input DataFrame,
        with yearly amounts expanded into separate columns.
    """
    n_years = len(price_breakdown.iloc[0]["Amount"])
    year_cols = [f"Year {i} Amount" for i in range(n_years)]

    amount_df = pd.DataFrame(np.array(price_breakdown["Amount"].to_list()), columns=year_cols)
    formatted_df = pd.concat(
        [price_breakdown[["Type", "Name"]], amount_df, price_breakdown["NPV"]], axis=1
    )
    return formatted_df
