import numpy as np
from utils.errors.errors import ObservableNotComputableError


# TODO: Use is_monotonic to check for monotonicity
def split_forward_reverse(independent: list, dependent: list) -> tuple[list, list, list, list]:
    """
    Assuming a behaviour for the independent list in two parts, a monotonic increase followed by a monotonic decrease.
    Under this assumption the function will split both independent and dependent lists by finding when the first
    reverses.
    This function checks:
        1) Whether the lists are of equal length
        2) If there is more than one change in direction
    """

    if len(independent) != len(dependent):
        raise ObservableNotComputableError("IV CALC: Input lists are of unequal length, cannot split forward and reverse")

    # Squash down all detail to detect the reversing index later
    change = np.diff(independent)
    direction = [np.sign(v) for v in change]

    # Check for constant array
    if set(direction) == {0}:
        raise ObservableNotComputableError("IV CALC: Independent list is constant")

    # Filter direction to treat no change as incoming change
    filtered_direction = []
    for i, d in enumerate(direction):
        if d != 0: # If change, copy
            filtered_direction.append(d)
        elif i < len(direction) - 1: # If no change, copy next
            filtered_direction.append(direction[i + 1])
        else: # plateau could lead to out of bounds access
            filtered_direction.append(filtered_direction[-1])

    direction = filtered_direction

    # Check the number of reversions in the independent list
    num_reversions = sum(1 for i in range(1, len(direction)) if direction[i] != direction[i - 1])

    # Only the case where at most one reversion is detected is allowed, others are not valid
    if num_reversions > 1:
        raise ObservableNotComputableError("IV CALC: Independent list is not monotonic, cannot find a unique reversing point")

    # The independent list reverses exactly once -> split where the array starts going back
    elif num_reversions == 1:
        reversing_index = direction.index(-1) + 1
        return independent[:reversing_index], independent[reversing_index:], dependent[:reversing_index], dependent[reversing_index:]

    # The independent list does not reverse, constant lists have already been selected out
    else:
        first_element = direction[0]
        # Independent is monotonically increasing, thus fully forward
        if first_element == 1:
            return independent, [], dependent, []

        # Independent is monotonically decreasing, thus fully reverse
        else:
            return [], independent, [], dependent


def trim_iv(voltages: list, currents: list, to_trim: list, isc: float, voc: float) -> list:
    """
        Will return a list that is trimmed based on the IV curve, Voc and Isc.

        This function assumes a monotonic increase of voltage (independent var -> ok) and current (noise -> not sure).
        The last current value below Isc and the first voltage value above Voc are searched, their indexes found and
        a the to_trim list is sublisted between the two indices.
    """
    # Find the index of the last negative voltage, this will have been the one used for Isc
    low = voltages.index([v for v in voltages if v < 0][-1])
    # Find all voltage values above voc and return the index of the first
    high = voltages.index([v for v in voltages if v > voc][0]) + 1

    # Return all powers inbetween, note slicing goes up to but not including 'high'
    return to_trim[low:high]


def find_crossing(x: list, y:list) -> float:
    """
        Determine interpolated y-crossing for a given numerical plot. Could be confused by excessively noisy x-data.
    """
    # To find the y-crossing we seek a point where the data goes from negative to positive
    flag_found = False
    for index in range(len(x)-1):
        if x[index] < 0 < x[index + 1]:
            flag_found = True
            break

    if not flag_found:
        raise ObservableNotComputableError("IV Calc: No negative-to-positive y-crossing could be found")

    # Once the numerical y-crossing has been found: determine the interpolated y-crossing (x = 0)
    crossing = np.interp(0, x[index:index+2], y[index:index+2])
    return crossing


def find_local_slope(x: list, y: list, value: float) -> float:
    for i, v in enumerate(y):
        if y[i] < value < y[i + 1]:
            break
    return (y[i] - y[i+1])/(x[i] - x[i+1])