function [xScaled, shift1, shift2, scaleFactor] = scaleDataInRange(xIn, desiredMinMax)
% [xScaled, shift1, shift2, scaleFactor] = scaleDataInRange(xIn, desiredMinMax)
% -------------------------------------------------------------------------
%
% This function takes in the data matrix xIn plus an optional vector of
% min and max values, and scales its values so that its values fall 
% between the specified output min and max. If the output min-max vector 
% is not specified or is empty, the function will scale the data to the 
% range [0, 1]. Scaling is based upon the min and max values of the entire
% input data matrix.
% 
% INPUTS
%   xIn: Input numeric data matrix.
%   desiredMinMax (optional): Two-element vector specifying the desired 
%       min and max values (in order) of the scaled data. If not specified 
%       or empty, this will default to [0, 1].
%
% OUTPUTS
%   xScaled: The scaled data. This variable will be a data matrix the same 
%       size as the input variable xIn.
%   shift1: Shift factor applied to the input data to make the smallest 
%       value in the input data equal to 0.
%   shift2: Shift factor applied to the data after shift1 and scaleFactor,
%       to ensure that the smallest value of the output data equals the 
%       specified desired minimum value.
% scaleFactor: The value by which the data were scaled, in between applying
%       shift1 and shift2.

% This software is released under the MIT License, as follows:
%
% Copyright (c) 2025 Bernard C. Wang, Raymond Gifford, Nathan C. L. Kong, 
% Feng Ruan, Anthony M. Norcia, and Blair Kaneshiro.
% 
% Permission is hereby granted, free of charge, to any person obtaining 
% a copy of this software and associated documentation files (the 
% "Software"), to deal in the Software without restriction, including 
% without limitation the rights to use, copy, modify, merge, publish, 
% distribute, sublicense, and/or sell copies of the Software, and to 
% permit persons to whom the Software is furnished to do so, subject to 
% the following conditions:
% 
% The above copyright notice and this permission notice shall be included 
% in all copies or substantial portions of the Software.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
% OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
% MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
% IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY 
% CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
% TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
% SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

% If desiredMinMax was specified AND is not empty, 
if exist('desiredMinMax', 'var') && ~isempty(desiredMinMax)
    % verify that it is a 2-element vector
    if ~isvector(desiredMinMax) || length(desiredMinMax) ~= 2
        error('Input ''desiredMinMax'' should be a vector of length 2.'); end
    % verify that its second element is strictly greater than the first.
    if desiredMinMax(2) <= desiredMinMax(1)
        error('Second element (max) of input ''desiredMinMax'' must be strictly greater than first element (min).'); end
% Otherwise, if empty or not specified, set to [0, 1] and print warning message.
else
    warning('Output data scaling range not specified. Setting to [0, 1].');
    desiredMinMax = [0 1];
end
% disp(['desiredMinMax = ' mat2str(desiredMinMax)]) % For debugging

desiredMin = desiredMinMax(1);
desiredMax = desiredMinMax(2);

% Shift data by subtracting smallest value
xVec = xIn(:);
shift1 = min(xVec);
xVec = xVec - shift1;

%%% OUTPUTS %%%
% Scaled data
scaleFactor = (desiredMinMax(2) - desiredMinMax(1)) / ...
            (max(xIn(:)) - min(xIn(:)));

xVec = xVec .* scaleFactor;
shift2 = desiredMinMax(1);

xVec = xVec + shift2;

xScaled = reshape(xVec, size(xIn) );