import sys
import os
import click
import subprocess
sys.path.append(os.getenv("AUTOWARE_FUZZER"))
import dataparser as _Parser
from dataparser.scenario import WayPoint, NPCBehaviour
from run_vse import VSERunner
from manager import AutowareManager
import json
import tempfile
import math
import logger
import time
def distance(pos1,pos2):
    return math.sqrt( pow(pos1['x']-pos2['x'],2) + pow(pos1['y']-pos2['y'],2) + pow(pos1['z']-pos2['z'],2) )

class Runner:

    def __init__(self, seed, gt_output, pred_out=None):
        self.seed = seed
        self.scenario = _Parser.scenario_parser(seed)
        self.VSE_dict = json.load(open(seed,'r'))
        self.gt_json = gt_output
        self.autoware_manager = AutowareManager()
        self.pred_json = pred_out
        self.normal_is_collision = False
        self.gt_is_collision = False
        
        self.collision_time = 0
        self.duration = 0
        self.collision_object = []
        

    def restart_simulator(self):
        if self.simprocess != None:
            self.simprocess.kill()
            self.simprocess.wait()
        os.system("killall -9 simulator ~/Build_Local/simulator")
        time.sleep(3)
        # self.simprocess = subprocess.Popen([self.simdir], env=self.env)
        os.system("cd ~/Build_Local && ~/Build_Local/run.sh &")
        self.simulator_time = time.time()
        time.sleep(5)

    def save_data(self,output):
        data={
            "is_collision": self.gt_is_collision,
            "collision_time": self.collision_time,
            "collision_object": list(self.collision_object),
            "duration": self.duration
        }
        with open(output,"w") as o:
            json.dump(data,o)

    def run_once(self,run_time=30,record_gt_traj=False):
        with tempfile.TemporaryDirectory() as tmpdirname:
            os.mkdir(tmpdirname + "/collision")
            vse_runner = VSERunner(self.scenario, tmpdirname, 'none', autoware_manager=self.autoware_manager,end_after_collision=not record_gt_traj)
            if record_gt_traj:
                self.wp_recorder = subprocess.Popen(["python",os.path.join(os.getenv("AUTOWARE_FUZZER"),"node/wp_recorder.py"),self.gt_json],
                                                    stdin=subprocess.PIPE,
                                                    stderr=subprocess.DEVNULL,
                                                    stdout=subprocess.DEVNULL)
            try:
                vse_runner.run(run_time)
            except Exception as e:
                print(str(e))
                if record_gt_traj:
                    self.wp_recorder.kill()
                    self.wp_recorder.wait()
                vse_runner.sim.close()
                return False
            if record_gt_traj:
                self.wp_recorder.communicate(input='Shut down'.encode())
                self.wp_recorder.wait()
        return vse_runner.is_collision, vse_runner.collision_time, vse_runner.collision_object

    def run(self):
        start = time.time()
        # firstly, run once to get gt trajectory
        print("[+] Running normal mode to get ground truth trajectory")
        self.autoware_manager.default()
        self.autoware_manager.set_args('map',map=self.scenario.map["name"])
        self.autoware_manager.start_all()
        self.normal_is_collision,_,_ = self.run_once(record_gt_traj = True)
        self.autoware_manager.stop_all(mask = ["bridge"])
        # then, run with prediction
        print("[+] Running prediction ground truth mode")
        self.autoware_manager.prediction_ground_truth()
        self.autoware_manager.set_args('prediction_GT',gt_traj=self.gt_json)
        self.autoware_manager.set_args('map',map=self.scenario.map["name"])
        self.autoware_manager.start_all(mask = ["bridge"])
        self.gt_is_collision,self.collision_time,self.collision_object = self.run_once(record_gt_traj = False)
        self.duration = time.time() - start

    
    def clean(self):
        self.autoware_manager.clean()

@click.command()
@click.option(
    "-si",
    "--seed_input",
    type=click.Path(dir_okay=False, exists=True),
    required=True,
    help="input seed",
)
@click.option(
    "-go",
    "--gt_output",
    type=click.Path(dir_okay=False, exists=False),
    required=True,
    help="intermediate output of prediction ground truth, i.e., the trajectories of NPCs",
)
@click.option(
    "-ro",
    "--result_output",
    type=click.Path(dir_okay=False, exists=False),
    required=True,
    help="final result to be saved as file",
)
def main(seed_input, gt_output,result_output):
    runner = Runner(seed_input, gt_output)
    new_seed = runner.run()
    runner.save_data(result_output)
    runner.clean()

if __name__=='__main__':
    main()
        

