from commonroad.planning.planning_problem import PlanningProblemSet,PlanningProblem
from commonroad.common.file_writer import CommonRoadFileWriter,OverwriteExistingFile
from commonroad.common.file_reader import CommonRoadFileReader
from commonroad.scenario.scenario import Scenario, Tag, ScenarioID
from commonroad.scenario.obstacle import DynamicObstacle, ObstacleType
from commonroad.geometry.shape import Rectangle
from commonroad.scenario.trajectory import State, Trajectory,TraceState
from commonroad.prediction.prediction import TrajectoryPrediction
from commonroad.planning.goal import GoalRegion
from pyproj import Proj
import json
import math
import sys,os
import numpy as np
from commonroad.common.util import Interval

import argparse

# sys.path.append("/home/vm2/CommonRoadToLgsvl")
from crdesigner.map_conversion.map_conversion_interface import opendrive_to_commonroad



LGSVL_Obstacle_To_CR_Obstacle = {
    'Sedan':'car',
    'BoxTruck':'truck',
    'SchoolBus':'bus',
    'Bicyclist':'bicycle',
    'Robin':'pedestrian',
    'taxi':'Hatchback',
    'Hatchback':'taxi',
    'SUV':'car',
    'Jeep':'car'
}


obstacle_class_dict = {
    'truck': ObstacleType.TRUCK,
    'car': ObstacleType.CAR,
    'bus':ObstacleType.BUS,
    'bicycle':ObstacleType.BICYCLE,
    'pedestrian':ObstacleType.PEDESTRIAN,
    'taxi':ObstacleType.TAXI
}


def get_trajectories_autoware(data_path):
    f = open(data_path,"r")
    npc_state_list = []
    ego_states = []
    states = json.load(f)
    for obj_id in states:
        # print(obj_id)
        if obj_id not in  ["ego","start_time","collision_time","is_collision"]:
            npc_state_list.append(states[obj_id][-30:])
    ego_states = states["ego"][-30:]
    return ego_states,npc_state_list

# input
def get_trajectories(data_path):
    f = open(data_path,"r")
    npc_state_list = []
    ego_states = []
    tmp_buf = f.readline()
    npc_dict = {}
    while tmp_buf:
        tmp_obj = json.loads(tmp_buf)
        if type(tmp_obj) is list:
            if len(tmp_obj) != 0:
                for npc in tmp_obj:
                    wp = dict(Id=npc["Id"],Label=npc["Label"],Position=npc["Position"], Rotation=npc["Rotation"], Speed=npc["LinearVelocity"]["x"],Scale=npc["Scale"])
                    if wp["Id"] not in npc_dict:
                        npc_dict[wp["Id"]] = [wp]
                    else:
                        npc_dict[wp["Id"]].append(wp)
        else:
            wp = dict(Id=tmp_obj["Id"],Label="Ego",Position=tmp_obj["Position"], Rotation=tmp_obj["Rotation"], Speed=tmp_obj["LinearVelocity"]["x"],Scale=tmp_obj["Scale"])
            ego_states.append(wp)
        tmp_buf = f.readline()
    f.close()
    delta_t = 0.1
    ego_states = ego_states[-30:]
    npc_state_list =[]
    for npc_id in npc_dict:
        npc_state_list.append(npc_dict[npc_id][-30:])
    return ego_states,npc_state_list

def get_destination(seed_path):
    with open(seed_path,"r") as f:
        json_object = json.load(f)
        lgsvl_dst = json_object["agents"][0]["destinationPoint"]
        return lgsvl_dst

# output
def output_xml(scenario,planning_problem_set,output_path):
    writer = CommonRoadFileWriter(
        scenario=scenario,
        planning_problem_set=planning_problem_set,
        author="ADSDx",
        affiliation="",
        source="",
        tags={Tag.URBAN}
    )
    writer.write_to_file(output_path,
                        OverwriteExistingFile.ALWAYS)

# map
def generate_init_scanerio_with_map(xodr_path):
    scenario = opendrive_to_commonroad(xodr_path)
    return scenario



# obstacle
def get_georeference(xodr_path):
    f=open(xodr_path,"r")
    map_content = f.read()
    origin_geo_reference = map_content[map_content.find("<geoReference> ")+len("<geoReference> "):map_content.find(" </geoReference>")]
    arrays = origin_geo_reference.split(" ")
    arrays[4] = "+x_0=0"
    arrays[5] = "+y_0=0"
    return " ".join(arrays)

def cr2lgsvl_position(_cr_position_x,_cr_position_y,geo_reference):
    a = Proj('+proj=utm +zone=32 +ellps=WGS84')
    lon, lat = a(_cr_position_x,_cr_position_y, inverse=True)
    b = Proj(geo_reference)
    p_x, p_y = b(lon,lat)
    p_z = 0
    lgsvl_position = {}
    lgsvl_position['x'] = p_x
    lgsvl_position['y'] = p_z
    lgsvl_position['z'] = p_y
    return lgsvl_position

def lgsvl2cr_position(x,y,geo_reference):
    return x,y
    a = Proj(geo_reference)
    lon, lat = a(x,y, inverse=True)
    b = Proj('+proj=utm +zone=32 +ellps=WGS84')
    p_x, p_y = b(lon,lat)
    # return p_x,p_y

def cr2lgsvl_rotation(_cr_orientation):
    lgsvl_rotation = {}
    lgsvl_rotation['x'] = 0
    lgsvl_rotation['y'] = 90 - math.degrees(_cr_orientation)
    lgsvl_rotation['z'] = 0
    return lgsvl_rotation

def lgsvl2cr_rotation(lgsvl_rotation):
    _cr_orientation = math.radians(90 - lgsvl_rotation)
    return _cr_orientation

def generate_obstacles(scenario,npc_state_list,geo_reference):
    dynamic_obstacles = []
    for npc_state in npc_state_list:
        if len(npc_state)>0:
            dynamic_obstacle = generate_single_dynamic_obstacle(scenario,npc_state,geo_reference)
            dynamic_obstacles.append(dynamic_obstacle)

    return dynamic_obstacles

def generate_single_dynamic_obstacle(scenario,npc_state,geo_reference):
    dynamic_obstacle_id = scenario.generate_object_id()
    npc_scale = npc_state[0]["size"]
    width = npc_scale["y"]
    length = npc_scale["x"]
    if width < 1:
        obj_type = "bicycle"
    elif width < 3:
        obj_type = "car"
    else:
        obj_type = "truck"

    dynamic_obstacle_type = obstacle_class_dict[obj_type]
    dynamic_obstacle_shape = Rectangle(width=width, length=length)
    dynamic_obstacle_initial_state = None
    state_list = []
    # print(len(npc_state))
    for i in range(30):
        if i < len(npc_state):
            state = npc_state[i]
        else:
            state = npc_state[-1]
        x,y = lgsvl2cr_position(state["position"]["x"],state["position"]["z"],geo_reference)
        v = state["speed"]
        theta = lgsvl2cr_rotation(state["rotation"]["y"])
        if i < len(npc_state)-1:
            a = (npc_state[i+1]["speed"] - v) / 0.1
        else:
            a = 0
        cr_state = State()
        cr_state.position = np.array([x, y])
        cr_state.velocity = v
        cr_state.orientation = theta
        cr_state.acceleration = 0
        cr_state.time_step = i
        state_list.append(cr_state)
    dynamic_obstacle_initial_state = state_list[0]

    dynamic_obstacle_trajectory = Trajectory(1, state_list[1:])
    dynamic_obstacle_prediction = TrajectoryPrediction(dynamic_obstacle_trajectory, dynamic_obstacle_shape)

    return DynamicObstacle(dynamic_obstacle_id, dynamic_obstacle_type, dynamic_obstacle_shape,
                           dynamic_obstacle_initial_state, dynamic_obstacle_prediction)


# planningProblem
def generate_planning_problem(scenario,ego_state,cr_dst_pos,cr_dst_orientation,geo_reference):
    planning_problem_id = scenario.generate_object_id()
    state = ego_state[0]
    scale = ego_state[0]["size"]
    x,y = lgsvl2cr_position(state["position"]["x"],state["position"]["z"],geo_reference)
    v = state["speed"]
    theta = lgsvl2cr_rotation(state["rotation"]["y"])
    dynamic_obstacle_initial_state = State()
    dynamic_obstacle_initial_state.position=np.array([x, y])
    dynamic_obstacle_initial_state.velocity=v
    dynamic_obstacle_initial_state.orientation=theta
    dynamic_obstacle_initial_state.time_step=0
    dynamic_obstacle_initial_state.yaw_rate = 0.0
    dynamic_obstacle_initial_state.slip_angle = 0.0


    goal_position = Rectangle(width=scale["y"],length=scale["x"],
                              center=np.array(cr_dst_pos),
                              orientation=cr_dst_orientation)
    dynamic_obstacle_final_state = State()
    time_step_half_range = 25
    time_step_interval = Interval(0, 30+time_step_half_range)
    dynamic_obstacle_final_state.position=goal_position
    dynamic_obstacle_final_state.time_step=time_step_interval
    goal_region = GoalRegion([dynamic_obstacle_final_state])
    return PlanningProblem(planning_problem_id, dynamic_obstacle_initial_state,goal_region)

def handle(scenario_name,data_path,xodr_path,seed_path,output_path):
    lgsvl_dst = get_destination(seed_path)
    scenario = generate_init_scanerio_with_map(xodr_path)
    scenario.scenario_id = ScenarioID.from_benchmark_id(scenario_name+"0924", '2020a')
    ego_state,npc_state_list = get_trajectories_autoware(data_path)
    geo_reference = get_georeference(xodr_path)

    cr_dst_pos = lgsvl2cr_position(lgsvl_dst["position"]["x"],lgsvl_dst["position"]["z"],geo_reference)
    cr_dst_orientation = lgsvl2cr_rotation(lgsvl_dst["rotation"]["y"])

    dynamic_obstacles = generate_obstacles(scenario,npc_state_list,geo_reference)
    for do in dynamic_obstacles:
        scenario.add_objects(do)
        # generate planning problems
    planning_problem_set = PlanningProblemSet()
    planning_problem = generate_planning_problem(scenario,ego_state,cr_dst_pos,cr_dst_orientation,geo_reference)
    planning_problem_set.add_planning_problem(planning_problem)
    output_xml(scenario,planning_problem_set,output_path)

def handle2(scenario_name,data_path,xodr_path,seed_path,origin_scenario_path,output_path):
    # scenario_name = "Test"
    # data_path = sys.argv[1]
    # xodr_path = sys.argv[2]
    # seed_path = sys.argv[3]
    # origin_scenario_path = sys.argv[4]
    lgsvl_dst = get_destination(seed_path)
    scenario, _ = CommonRoadFileReader(origin_scenario_path).open()
    scenario.scenario_id = ScenarioID.from_benchmark_id(scenario_name+"0924", '2020a')
    ego_state,npc_state_list = get_trajectories_autoware(data_path)
    geo_reference = get_georeference(xodr_path)

    cr_dst_pos = lgsvl2cr_position(lgsvl_dst["position"]["x"],lgsvl_dst["position"]["z"],geo_reference)
    cr_dst_orientation = lgsvl2cr_rotation(lgsvl_dst["rotation"]["y"])

    dynamic_obstacles = generate_obstacles(scenario,npc_state_list,geo_reference)
    scenario.remove_obstacle(scenario.obstacles)
    for do in dynamic_obstacles:
        scenario.add_objects(do)

    planning_problem_set = PlanningProblemSet()
    planning_problem = generate_planning_problem(scenario,ego_state,cr_dst_pos,cr_dst_orientation,geo_reference)
    planning_problem_set.add_planning_problem(planning_problem)
    output_xml(scenario,planning_problem_set,output_path)

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument('-n','--scenarioname')
    parser.add_argument('-d','--data')
    parser.add_argument('-x','--xodr')
    parser.add_argument('-s','--seed')
    parser.add_argument('-o','--output')
    args = parser.parse_args()
    handle(args.scenarioname, args.data, args.xodr, args.seed, args.output)

