# add root folder to Python interpreter's path
import os
import sys
import time
import pandas as pd
import shutil
import yaml
import argparse
import click
import json
import tempfile

import commonroad_reach.utility.logger as util_logger
from commonroad_reach.data_structure.configuration_builder import ConfigurationBuilder
from commonroad_reach.data_structure.reach.reach_interface import ReachableSetInterface
from commonroad_reach.utility import visualization as util_visual

def judge_avoidable(rootdir, name_scenario):
    builder = ConfigurationBuilder(path_root=rootdir)
    config = builder.build_configuration(name_scenario=name_scenario)
    config.update()
    # logger = util_logger.initialize_logger(config)
    # config.print_configuration_summary()
    reach_interface = ReachableSetInterface(config)
    reach_interface.compute_reachable_sets()
    longitudinal_driving_corridors = reach_interface.extract_driving_corridors(to_goal_region=False)
    print("Number of longitudinal driving corridors %s:" % len(longitudinal_driving_corridors))

    return len(longitudinal_driving_corridors) > 0

@click.command()
@click.option(
    "-i",
    "--input",
    type=click.Path(dir_okay=False, exists=True),
    required=True,
    help="input xml for avoidable judging",
)
@click.option(
    "-V",
    "--vehicle_config",
    type=click.Path(dir_okay=False, exists=False),
    required=True,
    default=os.path.join(os.path.dirname(__file__),"default_vehicle.yaml"),
    help="vehicle config path",
)
@click.option(
    "-c",
    "--config_dir",
    type=click.Path(dir_okay=True, exists=False),
    required=False,
    default=None,
    help="vehicle config path",
)
@click.option(
    "-o",
    "--output",
    type=click.Path(dir_okay=False, exists=False),
    required=True,
    help="result data output path",
)
def main(input,vehicle_config,config_dir,output):
    """
    config_dir:
    |-- configurations
        |-- scenario.yaml (copied from default_vehicle.yaml, should be prepared externally)
    |-- scenarios
        |-- scenario.xml  (input)
    """
    scenario = os.path.basename(input).split(".")[0]
    result = {}
    if config_dir == None:
        config_dir = tempfile.TemporaryDirectory().name
        os.mkdir(config_dir)
    os.mkdir(os.path.join(config_dir, "configurations"))
    os.mkdir(os.path.join(config_dir, "scenarios"))
    shutil.copy(vehicle_config, os.path.join(config_dir, "configurations", f"{scenario}.yaml"))
    shutil.copy(input, os.path.join(config_dir, "scenarios", f"{scenario}.xml"))

    try:
        print("[+] scenario: {}".format(scenario))

        start_time = time.time()
        try:
            avoidable = judge_avoidable(config_dir,scenario)
            end_time = time.time()
        except Exception as e:
            print(f"[!!!] exception: {e}")
            end_time = time.time()
            result = {
                "avoidable": -1,
                "duration": end_time - start_time
            }
            with open(output,"w") as o:
                json.dump(result, o)
            raise
        else:
            result = {
                "avoidable": 1 if avoidable else 0,
                "duration": end_time - start_time
            }
            with open(output,"w") as o:
                json.dump(result, o)
    except Exception as e:
        print(e)
        raise

if __name__ == "__main__":
    main()