/**
 * Copyright (c) 2019-2021 LG Electronics, Inc.
 *
 * This software contains code licensed as described in LICENSE.
 *
 */

using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using Simulator.Bridge;
using Simulator.Bridge.Data;
using Simulator.Map;
using Simulator.Utilities;
using Simulator.Sensors.UI;

using Newtonsoft.Json;
using System.IO;
using System.IO.Compression;
using System.Text;
using System.Threading.Tasks;

namespace Simulator.Sensors
{
    [SensorType("3D Ground Truth", new[] { typeof(Detected3DObjectData) })]
    public class PerceptionSensor3D : SensorBase
    {
        [SensorParameter]
        [Range(1f, 100f)]
        public float Frequency = 10.0f;

        [SensorParameter]
        [Range(1f, 1000f)]
        public float MaxDistance = 100.0f;

        public RangeTrigger RangeTrigger;
        WireframeBoxes WireframeBoxes;

        private BridgeInstance Bridge;
        private Publisher<Detected3DObjectData> Publish;
        private Publisher<DetectedPredictionObjectData> Publish_pred;

        private Dictionary<uint, Tuple<Detected3DObject, Collider>> Detected;
        private HashSet<uint> CurrentIDs;

        [AnalysisMeasurement(MeasurementType.Count)]
        public int MaxTracked = -1;

        public override SensorDistributionType DistributionType => SensorDistributionType.MainOrClient;
        public override float PerformanceLoad { get; } = 0.2f;
        MapOrigin MapOrigin;

        public override void OnBridgeSetup(BridgeInstance bridge)
        {
            Bridge = bridge;
            Publish = Bridge.AddPublisher<Detected3DObjectData>(Topic);
            Publish_pred = Bridge.AddPublisher<DetectedPredictionObjectData>("/apollo/prediction");
        }

        private static int bufferSize = 1024 * 1024 * 30;
        private byte[] decompressedBytes = new byte[bufferSize];
        private List<DetectedPredictionObjectData> dpodList;
        Stream File; // lol

        protected override void Initialize()
        {
            // lol
            var path = Path.Combine(Simulator.Web.Config.PersistentDataPath, "pred.txt.gz");
            File = new GZipStream(new FileStream(path, FileMode.Open), CompressionMode.Decompress, false);
            File.Read(decompressedBytes, 0, bufferSize);
            dpodList = JsonConvert.DeserializeObject<List<DetectedPredictionObjectData>>(Encoding.Default.GetString(decompressedBytes));

            WireframeBoxes = SimulatorManager.Instance.WireframeBoxes;

            if (RangeTrigger == null)
            {
                RangeTrigger = GetComponentInChildren<RangeTrigger>();
            }

            RangeTrigger.SetCallbacks(WhileInRange);
            RangeTrigger.transform.localScale = MaxDistance * Vector3.one;

            MapOrigin = MapOrigin.Find();

            Detected = new Dictionary<uint, Tuple<Detected3DObject, Collider>>();
            CurrentIDs = new HashSet<uint>();

            StartCoroutine(OnPublish());
        }

        protected override void Deinitialize()
        {
            StopAllCoroutines();

            Detected.Clear();
            CurrentIDs.Clear();

            // lol
            File.Close();
            File = null;
        }

        private void FixedUpdate()
        {
            MaxTracked = Math.Max(MaxTracked, CurrentIDs.Count);
            CurrentIDs.Clear();
        }

        void WhileInRange(Collider other)
        {
            GameObject egoGO = transform.parent.gameObject;
            GameObject parent = other.transform.parent.gameObject;
            if (parent == egoGO)
            {
                return;
            }

            if (!(other.gameObject.layer == LayerMask.NameToLayer("GroundTruth")) || !parent.activeInHierarchy)
            {
                return;
            }

            uint id;
            string label;
            Vector3 velocity;
            float angular_speed;  // Angular speed around up axis of objects, in radians/sec
            if (parent.layer == LayerMask.NameToLayer("Agent"))
            {
                var controller = parent.GetComponent<IAgentController>();
                var rb = parent.GetComponent<Rigidbody>();
                id = controller.GTID;
                label = "Sedan";
                velocity = rb.velocity;
                angular_speed = rb.angularVelocity.y;
            }
            else if (parent.layer == LayerMask.NameToLayer("NPC"))
            {
                var npcC = parent.GetComponent<NPCController>();
                id = npcC.GTID;
                label = npcC.NPCLabel;
                velocity = npcC.GetVelocity();
                angular_speed = npcC.GetAngularVelocity().y;
            }
            else if (parent.layer == LayerMask.NameToLayer("Pedestrian"))
            {
                var pedC = parent.GetComponent<PedestrianController>();
                id = pedC.GTID;
                label = "Pedestrian";
                velocity = pedC.CurrentVelocity;
                angular_speed = pedC.CurrentAngularVelocity.y;
            }
            else
            {
                return;
            }

            Vector3 size = ((BoxCollider)other).size;
            if (size.magnitude == 0)
            {
                return;
            }

            // Linear speed in forward direction of objects, in meters/sec
            float speed = Vector3.Dot(velocity, parent.transform.forward);
            // Local position of object in ego local space
            Vector3 relPos = transform.InverseTransformPoint(parent.transform.position);
            // Relative rotation of objects wrt ego frame
            Quaternion relRot = Quaternion.Inverse(transform.rotation) * parent.transform.rotation;

            var mapRotation = MapOrigin.transform.localRotation;
            velocity = Quaternion.Inverse(mapRotation) * velocity;
            var heading = parent.transform.localEulerAngles.y - mapRotation.eulerAngles.y;

            // Center of bounding box
            GpsLocation location = MapOrigin.GetGpsLocation(((BoxCollider)other).bounds.center);
            GpsData gps = new GpsData()
            {
                Easting = location.Easting,
                Northing = location.Northing,
                Altitude = location.Altitude,
            };

            if (!Detected.ContainsKey(id))
            {
                var det = new Detected3DObject()
                {
                    Id = id,
                    Label = label,
                    Score = 1.0f,
                    Position = relPos,
                    Rotation = relRot,
                    Scale = size,
                    LinearVelocity = new Vector3(speed, 0, 0),
                    AngularVelocity = new Vector3(0, 0, angular_speed),
                    Velocity = velocity,
                    Gps = gps,
                    Heading = heading,
                    TrackingTime = 0f,
                };

                Detected.Add(id, new Tuple<Detected3DObject, Collider>(det, other));
            }
            else
            {
                var det = Detected[id].Item1;
                det.Position = relPos;
                det.Rotation = relRot;
                det.LinearVelocity = new Vector3(speed, 0, 0);
                det.AngularVelocity = new Vector3(0, 0, angular_speed);
                det.Acceleration = (velocity - det.Velocity) / Time.fixedDeltaTime;
                det.Velocity = velocity;
                det.Gps = gps;
                det.Heading = heading;
                det.TrackingTime += Time.fixedDeltaTime;
            }

            CurrentIDs.Add(id);
        }

        private IEnumerator OnPublish()
        {
            var flag = true;
            var dpodIdx = 0;
            DetectedPredictionObjectData dpod;

            uint seqId = 0;
            double nextSend = SimulatorManager.Instance.CurrentTime + 1.0f / Frequency;

            while (true)
            {
                yield return new WaitForFixedUpdate();

                var IDs = new HashSet<uint>(Detected.Keys);
                IDs.ExceptWith(CurrentIDs);
                foreach(uint id in IDs)
                {
                    Detected.Remove(id);
                }

                if (Bridge != null && Bridge.Status == Status.Connected)
                {
                    if (SimulatorManager.Instance.CurrentTime < nextSend)
                    {
                        continue;
                    }
                    nextSend = SimulatorManager.Instance.CurrentTime + 1.0f / Frequency;

                    var currentObjects = new List<Detected3DObject>();
                    foreach (uint id in CurrentIDs)
                    {
                        currentObjects.Add(Detected[id].Item1);
                    }

                    if (currentObjects.Count > 0 && flag) {
                        var npc_gt = currentObjects[0];
                        foreach (var npc_pred in dpodList[0].Data) {
                            if (npc_gt.Gps.Easting - npc_pred.Data.Gps.Easting < 0.0000001) {
                                System.Console.WriteLine("Right!");
                                flag = false;
                                dpodIdx = 0;
                                break;
                            }
                            else {
                                System.Console.WriteLine("Dismatch!");
                            }
                        }
                    }

                    var data = new Detected3DObjectData()
                    {
                        Name = Name,
                        Frame = Frame,
                        Time = SimulatorManager.Instance.CurrentTime,
                        Sequence = seqId++,
                        Data = currentObjects.ToArray(),
                    };

                    Publish(data);

                    if (flag == false && dpodIdx < dpodList.Count) {
                        dpod = dpodList[dpodIdx];
                        dpod.Time = SimulatorManager.Instance.CurrentTime;
                        dpodIdx++;
                        Publish_pred(dpod);
                    }
                }
            }
        }

        public override void OnVisualize(Visualizer visualizer)
        {
            foreach (uint id in CurrentIDs)
            {
                var col = Detected[id].Item2;
                if (col.gameObject.activeInHierarchy)
                {
                    GameObject parent = col.gameObject.transform.parent.gameObject;
                    Color color = Color.green;
                    if (parent.layer == LayerMask.NameToLayer("Pedestrian"))
                    {
                        color = Color.yellow;
                    }

                    BoxCollider box = col as BoxCollider;
                    WireframeBoxes.Draw
                    (
                        box.transform.localToWorldMatrix,
                        new Vector3(0f, box.bounds.extents.y, 0f),
                        box.size,
                        color
                    );
                }
            }
        }

        public override void OnVisualizeToggle(bool state) {}

        public bool CheckVisible(Bounds bounds)
        {
            return Vector3.Distance(transform.position, bounds.center) < 50f;
            //var activeCameraPlanes = Utility.CalculateFrustum(transform.position, (bounds.center - transform.position).normalized);
            //return GeometryUtility.TestPlanesAABB(activeCameraPlanes, bounds);
        }
    }
}
