
using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using Simulator.Bridge;
using Simulator.Bridge.Data;
using Simulator.Map;
using Simulator.Utilities;
using Simulator.Sensors.UI;

using Newtonsoft.Json;
using System.IO;
using System.IO.Compression;
using System.Text;
using System.Threading.Tasks;

namespace Simulator.Sensors
{
    [SensorType("3D Ground Truth", new[] { typeof(Detected3DObjectData) })]
    public class PerceptionSensor3D : SensorBase
    {
        [SensorParameter]
        [Range(1f, 100f)]
        public float Frequency = 10.0f;

        [SensorParameter]
        [Range(1f, 1000f)]
        public float MaxDistance = 100.0f;

        public RangeTrigger RangeTrigger;
        WireframeBoxes WireframeBoxes;

        private BridgeInstance Bridge;
        private Publisher<Detected3DObjectData> Publish;

        static JsonSerializerSettings jsonSettings = new JsonSerializerSettings(); // lol
        Stream File; // lol

        private Dictionary<uint, Tuple<Detected3DObject, Collider>> Detected;
        private HashSet<uint> CurrentIDs;

        [AnalysisMeasurement(MeasurementType.Count)]
        public int MaxTracked = -1;

        public override SensorDistributionType DistributionType => SensorDistributionType.MainOrClient;
        public override float PerformanceLoad { get; } = 0.2f;
        MapOrigin MapOrigin;

        private IAgentController Controller;
        private IVehicleDynamics Dynamics;
        public override void OnBridgeSetup(BridgeInstance bridge)
        {
            Bridge = bridge;
            Publish = Bridge.AddPublisher<Detected3DObjectData>(Topic);
        }

        protected override void Initialize()
        {
            // lol
            jsonSettings.Formatting = Formatting.None;
            jsonSettings.Converters.Add(new UnityConverters());
            var path = Path.Combine(Simulator.Web.Config.PersistentDataPath, "in.txt.gz");
            File = new GZipStream(new FileStream(path, FileMode.Create), CompressionMode.Compress, false);

            Controller = GetComponentInParent<IAgentController>();
            Dynamics = GetComponentInParent<IVehicleDynamics>();
            WireframeBoxes = SimulatorManager.Instance.WireframeBoxes;

            if (RangeTrigger == null)
            {
                RangeTrigger = GetComponentInChildren<RangeTrigger>();
            }

            RangeTrigger.SetCallbacks(WhileInRange);
            RangeTrigger.transform.localScale = MaxDistance * Vector3.one;

            MapOrigin = MapOrigin.Find();

            Detected = new Dictionary<uint, Tuple<Detected3DObject, Collider>>();
            CurrentIDs = new HashSet<uint>();

            StartCoroutine(OnPublish());
        }

        protected override void Deinitialize()
        {
            StopAllCoroutines();

            Detected.Clear();
            CurrentIDs.Clear();

            // lol
            File.Close();
            File = null;
        }

        public void Write(string type, string topic, string data)
        {
            Task.Run(() =>
            {
                // this Write method can be called from multiple threads at the same time
                // we want to make sure that type/topic line and data line are next to each other
                lock (this)
                {
                    if (File != null)
                    {
                        var bytes = Encoding.UTF8.GetBytes($"{type} {topic}\n");
                        // File.Write(bytes, 0, bytes.Length);

                        bytes = Encoding.UTF8.GetBytes(data);
                        File.Write(bytes, 0, bytes.Length);
                        File.WriteByte((byte)'\n');
                        File.Flush();
                    }
                }
            });
        }

        private void FixedUpdate()
        {
            MaxTracked = Math.Max(MaxTracked, CurrentIDs.Count);
            CurrentIDs.Clear();
        }

        void WhileInRange(Collider other)
        {
            GameObject egoGO = transform.parent.gameObject;
            GameObject parent = other.transform.parent.gameObject;
            if (parent == egoGO)
            {
                return;
            }

            if (!(other.gameObject.layer == LayerMask.NameToLayer("GroundTruth")) || !parent.activeInHierarchy)
            {
                return;
            }

            uint id;
            string label;
            Vector3 velocity;
            float angular_speed;  // Angular speed around up axis of objects, in radians/sec
            if (parent.layer == LayerMask.NameToLayer("Agent"))
            {
                id = Controller.GTID;
                label = "Sedan";
                velocity = Dynamics.Velocity;
                angular_speed = Dynamics.AngularVelocity.y;
            }
            else if (parent.layer == LayerMask.NameToLayer("NPC"))
            {
                var npcC = parent.GetComponent<NPCController>();
                id = npcC.GTID;
                label = npcC.NPCLabel;
                velocity = npcC.GetVelocity();
                angular_speed = npcC.GetAngularVelocity().y;
            }
            else if (parent.layer == LayerMask.NameToLayer("Pedestrian"))
            {
                var pedC = parent.GetComponent<PedestrianController>();
                id = pedC.GTID;
                label = "Pedestrian";
                velocity = pedC.CurrentVelocity;
                angular_speed = pedC.CurrentAngularVelocity.y;
            }
            else
            {
                return;
            }

            Vector3 size = ((BoxCollider)other).size;
            if (size.magnitude == 0)
            {
                return;
            }

            // Linear speed in forward direction of objects, in meters/sec
            float speed = Vector3.Dot(velocity, parent.transform.forward);
            // Local position of object in ego local space
            // Vector3 relPos = transform.InverseTransformPoint(parent.transform.position);
            Vector3 relPos = parent.transform.position;
            // Relative rotation of objects wrt ego frame
            // Quaternion relRot = Quaternion.Inverse(transform.rotation) * parent.transform.rotation;
            Quaternion relRot = parent.transform.rotation;
            relRot.x = parent.transform.localEulerAngles.x;
            relRot.y = parent.transform.localEulerAngles.y;
            relRot.z = parent.transform.localEulerAngles.z;

            var mapRotation = MapOrigin.transform.localRotation;
            velocity = Quaternion.Inverse(mapRotation) * velocity;
            var heading = parent.transform.localEulerAngles.y - mapRotation.eulerAngles.y;

            // Center of bounding box
            GpsLocation location = MapOrigin.GetGpsLocation(((BoxCollider)other).bounds.center);
            GpsData gps = new GpsData()
            {
                Easting = location.Easting,
                Northing = location.Northing,
                Altitude = location.Altitude,
            };

            if (!Detected.ContainsKey(id))
            {
                var det = new Detected3DObject()
                {
                    Id = id,
                    Label = label,
                    Score = 1.0f,
                    Position = relPos,
                    Rotation = relRot,
                    Scale = size,
                    LinearVelocity = new Vector3(speed, 0, 0),
                    AngularVelocity = new Vector3(0, 0, angular_speed),
                    Velocity = velocity,
                    Gps = gps,
                    Heading = heading,
                    TrackingTime = 0f,
                };

                Detected.Add(id, new Tuple<Detected3DObject, Collider>(det, other));
            }
            else
            {
                var det = Detected[id].Item1;
                det.Position = relPos;
                det.Rotation = relRot;
                det.LinearVelocity = new Vector3(speed, 0, 0);
                det.AngularVelocity = new Vector3(0, 0, angular_speed);
                det.Acceleration = (velocity - det.Velocity) / Time.fixedDeltaTime;
                det.Velocity = velocity;
                det.Gps = gps;
                det.Heading = heading;
                det.TrackingTime += Time.fixedDeltaTime;
            }

            CurrentIDs.Add(id);
        }

        private IEnumerator OnPublish()
        {
            uint seqId = 0;
            double nextSend = SimulatorManager.Instance.CurrentTime + 1.0f / Frequency;

            while (true)
            {
                yield return new WaitForFixedUpdate();

                var IDs = new HashSet<uint>(Detected.Keys);
                IDs.ExceptWith(CurrentIDs);
                foreach(uint id in IDs)
                {
                    Detected.Remove(id);
                }

                if (Bridge != null && Bridge.Status == Status.Connected)
                {
                    if (SimulatorManager.Instance.CurrentTime < nextSend)
                    {
                        continue;
                    }
                    nextSend = SimulatorManager.Instance.CurrentTime + 1.0f / Frequency;

                    var currentObjects = new List<Detected3DObject>();
                    foreach (uint id in CurrentIDs)
                    {
                        currentObjects.Add(Detected[id].Item1);
                    }

                    var data = new Detected3DObjectData()
                    {
                        Name = Name,
                        Frame = Frame,
                        Time = SimulatorManager.Instance.CurrentTime,
                        Sequence = seqId++,
                        Data = currentObjects.ToArray(),
                    };

                    Publish(data);

                    var ego = new Detected3DObject()
                    {
                        Id = 777,
                        Position = transform.position
                    };

                    if ( currentObjects.Count != 0 )
                        Write("PerceptionSensor3D", SimulatorManager.Instance.CurrentTime.ToString(), JsonConvert.SerializeObject(ego, jsonSettings));
                        Write("PerceptionSensor3D", SimulatorManager.Instance.CurrentTime.ToString(), JsonConvert.SerializeObject(currentObjects, jsonSettings));
                }
            }
        }

        public override void OnVisualize(Visualizer visualizer)
        {
            foreach (uint id in CurrentIDs)
            {
                var col = Detected[id].Item2;
                if (col.gameObject.activeInHierarchy)
                {
                    GameObject parent = col.gameObject.transform.parent.gameObject;
                    Color color = Color.green;
                    if (parent.layer == LayerMask.NameToLayer("Pedestrian"))
                    {
                        color = Color.yellow;
                    }

                    BoxCollider box = col as BoxCollider;
                    WireframeBoxes.Draw
                    (
                        box.transform.localToWorldMatrix,
                        new Vector3(0f, box.bounds.extents.y, 0f),
                        box.size,
                        color
                    );
                }
            }
        }

        public override void OnVisualizeToggle(bool state) {}

        public bool CheckVisible(Bounds bounds)
        {
            return Vector3.Distance(transform.position, bounds.center) < 50f;
            //var activeCameraPlanes = Utility.CalculateFrustum(transform.position, (bounds.center - transform.position).normalized);
            //return GeometryUtility.TestPlanesAABB(activeCameraPlanes, bounds);
        }
    }

    class UnityConverters : JsonConverter
    {
        public override bool CanConvert(Type objectType)
        {
            return objectType == typeof(Quaternion)
                || objectType == typeof(Vector3)
                || objectType == typeof(Matrix4x4);
        }

        public override object ReadJson(JsonReader reader, Type objectType, object existingValue, JsonSerializer serializer)
        {
            throw new NotImplementedException();
        }

        public override void WriteJson(JsonWriter writer, object value, JsonSerializer serializer)
        {
            if (value.GetType() == typeof(Quaternion))
            {
                var q = (Quaternion)value;
                writer.WriteStartObject();
                writer.WritePropertyName("x");
                writer.WriteValue(q.x);
                writer.WritePropertyName("y");
                writer.WriteValue(q.y);
                writer.WritePropertyName("z");
                writer.WriteValue(q.z);
                writer.WritePropertyName("w");
                writer.WriteValue(q.w);
                writer.WriteEndObject();
            }
            else if (value.GetType() == typeof(Vector3))
            {
                var v = (Vector3)value;
                writer.WriteStartObject();
                writer.WritePropertyName("x");
                writer.WriteValue(v.x);
                writer.WritePropertyName("y");
                writer.WriteValue(v.y);
                writer.WritePropertyName("z");
                writer.WriteValue(v.z);
                writer.WriteEndObject();
            }
            else if (value.GetType() == typeof(Matrix4x4))
            {
                var m = (Matrix4x4)value;
                writer.WriteStartArray();
                for (int i = 0; i < 16; i++)
                {
                    writer.WriteValue(m[i]);
                }
                writer.WriteEnd();
            }
        }
    }
}
