
#pragma once

#include <memory>
#include <string.h>
#include "cyber/cyber.h"

#include "modules/localization/proto/localization.pb.h"
#include "modules/map/relative_map/proto/navigation.pb.h"
#include "modules/planning/proto/planning.pb.h"
#include "modules/prediction/proto/prediction_obstacle.pb.h"
#include "modules/control/proto/control_cmd.pb.h"

namespace apollo {
namespace controlGT {

/**
 * @class ControlGTCore
 * @brief A module that simulates a 'perfect control' algorithm, which assumes
 * an ideal world where the car can be perfectly placed wherever the planning
 * asks it to be, with the expected speed, acceleration, etc.
 */
class ControlGTCore {
 public:

  explicit ControlGTCore(char* log_file,bool enable_prediction_gt);

  bool IsEnabled() const { return enabled_; }

  /**
   * @brief Starts the timer to publish simulated localization and chassis
   * messages.
   */
  void Start(double v);

  /**
   * @brief Stops the timer.
   */
  void Stop();

  /**
   * @brief Resets the internal state.
   */
  void Reset();

  void RunOnce();

 private:
  void OnPlanning(
      const std::shared_ptr<apollo::planning::ADCTrajectory> &trajectory);
  void OnRoutingResponse(
      const std::shared_ptr<apollo::routing::RoutingResponse> &routing);
  void OnReceiveNavigationInfo(
      const std::shared_ptr<apollo::relative_map::NavigationInfo>
          &navigation_info);
  void OnPredictionObstacles(
      const std::shared_ptr<apollo::prediction::PredictionObstacles>
          &obstacles);
  void OnPerceptionObstacles(
      const std::shared_ptr<apollo::perception::PerceptionObstacles>
          &obstacles);

  void LogTrajectoryPoint(apollo::common::TrajectoryPoint *point);
  /**
   * @brief Predict the next trajectory point using perfect control model
   */
  bool PerfectControlModel(
      apollo::common::TrajectoryPoint *point,
      apollo::canbus::Chassis::GearPosition *gear_position);

  void PublishChassis(double cur_speed,
                      apollo::canbus::Chassis::GearPosition gear_position);

  void PublishLocalization(const apollo::common::TrajectoryPoint &point);

  void PublishDummyPrediction();

  void InitTimerAndIO();

  void InitStartPoint(double start_velocity, double start_acceleration);

  // Reset the start point, which can be a dummy point on the map, a current
  // localization pose, or a start position received from the routing module.
  void SetStartPoint(const apollo::common::TrajectoryPoint &point);

  void Freeze();

  void ClearPlanning();

  void InternalReset();

  std::unique_ptr<cyber::Node> node_;

  std::shared_ptr<cyber::Reader<apollo::localization::LocalizationEstimate>>
      localization_reader_;
  std::shared_ptr<cyber::Reader<apollo::planning::ADCTrajectory>>
      planning_reader_;
  std::shared_ptr<cyber::Reader<apollo::prediction::PredictionObstacles>>
      prediction_reader_;
  std::shared_ptr<cyber::Reader<apollo::perception::PerceptionObstacles>>
      perception_reader_;


  std::shared_ptr<cyber::Writer<apollo::localization::LocalizationEstimate>>
      localization_writer_;
  std::shared_ptr<cyber::Writer<apollo::control::ControlCommand>>
      control_cmd_writer_;
  std::shared_ptr<cyber::Writer<apollo::canbus::Chassis>> chassis_writer_;
  std::shared_ptr<cyber::Writer<apollo::prediction::PredictionObstacles>>
      prediction_writer_;
  // The timer to publish simulated localization and chassis messages.
  std::unique_ptr<cyber::Timer> sim_control_timer_;
  // The timer to publish dummy prediction
  std::unique_ptr<cyber::Timer> sim_prediction_timer_;

  // Time interval of the timer, in milliseconds.
  static constexpr double kSimControlIntervalMs = 10;
  static constexpr double kSimPredictionIntervalMs = 100;

  // The latest received planning trajectory.
  std::shared_ptr<apollo::planning::ADCTrajectory> current_trajectory_;
  // The index of the previous and next point with regard to the
  // current_trajectory.
  int prev_point_index_ = 0;
  int next_point_index_ = 0;

  // Whether there's a planning received after the most recent routing.
  bool received_planning_ = false;
  bool received_simulator_signal_ = true;
  // Whether planning has requested a re-routing.
  bool re_routing_triggered_ = false;

  // Whether to send dummy predictions
  bool send_dummy_prediction_ = true;

    // whether in prediction GT mode
  bool _is_enable_prediction_gt = false;
  // Whether the sim control is enabled.
  bool enabled_ = false;
  char* _log_file_name = NULL; 
  int _waypoint_index = 0;
  // Whether start point is initialized from actual localization data
  bool start_point_from_localization_ = false;
  double initial_speed_ = 0.0;

  apollo::common::TrajectoryPoint prev_point_;
  apollo::common::TrajectoryPoint next_point_;

  common::PathPoint adc_position_;

  // Linearize reader/timer callbacks and external operations.
  std::mutex mutex_;
};

}  // namespace dreamview
}  // namespace apollo
