
#include "modules/controlGT/control_gt_core.h"

#include "cyber/common/file.h"
#include "cyber/time/clock.h"
#include "modules/common/adapters/adapter_gflags.h"
#include "modules/common/math/linear_interpolation.h"
#include "modules/common/math/math_utils.h"
#include "modules/common/math/quaternion.h"
#include "modules/common/util/message_util.h"
#include "modules/common/util/util.h"

namespace apollo {
namespace controlGT {

using apollo::canbus::Chassis;
using apollo::common::Header;
using apollo::common::Point3D;
using apollo::common::Quaternion;
using apollo::common::TrajectoryPoint;
using apollo::common::math::HeadingToQuaternion;
using apollo::common::math::InterpolateUsingLinearApproximation;
using apollo::common::math::InverseQuaternionRotate;
using apollo::common::util::FillHeader;
using apollo::common::util::DistanceXY;
using apollo::cyber::Clock;
using apollo::localization::LocalizationEstimate;
using apollo::planning::ADCTrajectory;
using apollo::prediction::PredictionObstacles;
using apollo::relative_map::NavigationInfo;
using apollo::routing::RoutingResponse;
using apollo::control::ControlCommand;
using apollo::perception::PerceptionObstacles;

namespace {

void TransformToVRF(const Point3D& point_mrf, const Quaternion& orientation,
                    Point3D* point_vrf) {
  Eigen::Vector3d v_mrf(point_mrf.x(), point_mrf.y(), point_mrf.z());
  auto v_vrf = InverseQuaternionRotate(orientation, v_mrf);
  point_vrf->set_x(v_vrf.x());
  point_vrf->set_y(v_vrf.y());
  point_vrf->set_z(v_vrf.z());
}

bool IsSameHeader(const Header& lhs, const Header& rhs) {
  return lhs.sequence_num() == rhs.sequence_num() &&
         lhs.timestamp_sec() == rhs.timestamp_sec();
}

}  // namespace

ControlGTCore::ControlGTCore(char* log_file,bool enable_prediction_gt)
    : node_(cyber::CreateNode("control_gt")),
      current_trajectory_(std::make_shared<ADCTrajectory>()) {
    _log_file_name = log_file;
    _is_enable_prediction_gt = enable_prediction_gt;
    AERROR << "ControlGTCore create!";
  InitTimerAndIO();
}

void ControlGTCore::InitTimerAndIO() {
  localization_reader_ =
      node_->CreateReader<LocalizationEstimate>("/apollo/gt_localization/pose");
  planning_reader_ = node_->CreateReader<ADCTrajectory>(
      "/apollo/planning",
      [this](const std::shared_ptr<ADCTrajectory>& trajectory) {
        this->OnPlanning(trajectory);
      });

  // prediction_reader_ = node_->CreateReader<PredictionObstacles>(
  //     "/apollo/prediction",
  //     [this](const std::shared_ptr<PredictionObstacles>& predictionObstacles) {
  //       this->OnPredictionObstacles(predictionObstacles);
  //     });

  perception_reader_ = node_->CreateReader<PerceptionObstacles>(
      "/apollo/perception/obstacles",
      [this](const std::shared_ptr<PerceptionObstacles>& obstacles) {
        this->OnPerceptionObstacles(obstacles);
      });


  prediction_writer_ =
      node_->CreateWriter<PredictionObstacles>("/apollo/prediction");

  localization_writer_ =
      node_->CreateWriter<LocalizationEstimate>("/apollo/localization/pose");
  chassis_writer_ = node_->CreateWriter<Chassis>("/apollo/canbus/chassis");

  control_cmd_writer_ = node_->CreateWriter<ControlCommand>("/apollo/control");
  // Start timer to publish localization and chassis messages.
  sim_control_timer_.reset(new cyber::Timer(
      kSimControlIntervalMs, [this]() { this->RunOnce(); }, false));
  
  sim_prediction_timer_.reset(new cyber::Timer(
      kSimPredictionIntervalMs, [this]() { this->PublishDummyPrediction(); },
      false));
}
void ControlGTCore::InitStartPoint(double start_velocity,
                                double start_acceleration) {
  TrajectoryPoint point;
  // Use the latest localization position as start point,
  // fall back to a dummy point from map
  localization_reader_->Observe();
  if (localization_reader_->Empty()) {
    AERROR << "localization is Empty";
    return;
  } else {
    start_point_from_localization_ = true;
    const auto& localization = localization_reader_->GetLatestObserved();
    localization_writer_->Write(localization);
    const auto& pose = localization->pose();
    point.mutable_path_point()->set_x(pose.position().x());
    point.mutable_path_point()->set_y(pose.position().y());
    point.mutable_path_point()->set_z(pose.position().z());
    point.mutable_path_point()->set_theta(pose.heading());
    point.set_v(initial_speed_);
    // point.set_v(
    //     std::hypot(pose.linear_velocity().x(), pose.linear_velocity().y()));
    // AERROR << "InitStartPoint pose.linear_velocity().x():"<<pose.linear_velocity().x()<<", pose.linear_velocity().y():"<<pose.linear_velocity().y()<<", pose.linear_velocity().z():"<<pose.linear_velocity().z();
    // Calculates the dot product of acceleration and velocity. The sign
    // of this projection indicates whether this is acceleration or
    // deceleration.
    double projection =
        pose.linear_acceleration().x() * pose.linear_velocity().x() +
        pose.linear_acceleration().y() * pose.linear_velocity().y();

    // Calculates the magnitude of the acceleration. Negate the value if
    // it is indeed a deceleration.
    double magnitude = std::hypot(pose.linear_acceleration().x(),
                                  pose.linear_acceleration().y());
    point.set_a(std::signbit(projection) ? -magnitude : magnitude);
  }
  SetStartPoint(point);
}


void ControlGTCore::SetStartPoint(const TrajectoryPoint& start_point) {
  next_point_ = start_point;
  prev_point_index_ = next_point_index_ = 0;
  received_planning_ = false;
}


void ControlGTCore::InternalReset() {
  ClearPlanning();
}

void ControlGTCore::ClearPlanning() {
  current_trajectory_->Clear();
  received_planning_ = false;
}

void ControlGTCore::Start(double v) {
  std::lock_guard<std::mutex> lock(mutex_);
  initial_speed_ = v;
  if (!enabled_) {
    // When there is no localization yet, Init(true) will use a
    // dummy point from the current map as an arbitrary start.
    // When localization is already available, we do not need to
    // reset/override the start point.
    localization_reader_->Observe();
    InitStartPoint(next_point_.has_v() ? next_point_.v() : 0.0,
         next_point_.has_a() ? next_point_.a() : 0.0);

    // InternalReset();
    sim_control_timer_->Start();
    if(_is_enable_prediction_gt){
      sim_prediction_timer_->Start();
    }
    enabled_ = true;
  }
}

void ControlGTCore::Stop() {
  std::lock_guard<std::mutex> lock(mutex_);

  if (enabled_) {
    sim_control_timer_->Stop();
    if(_is_enable_prediction_gt){
      sim_prediction_timer_->Stop();
    }
    enabled_ = false;
  }
}

void ControlGTCore::OnPlanning(const std::shared_ptr<ADCTrajectory>& trajectory) {
  std::lock_guard<std::mutex> lock(mutex_);
  if (!enabled_ || !start_point_from_localization_ || trajectory->trajectory_point().size()==0||std::abs(trajectory->trajectory_point().Get(0).v() - next_point_.v())>5) {
    return;
  }
  if (!received_planning_){
      auto control_cmd = std::make_shared<ControlCommand>();
      FillHeader("SimControl", control_cmd.get());
      control_cmd_writer_->Write(control_cmd);
      AERROR << "simulation start!";
  }
  // if (received_planning_ && trajectory->header().timestamp_sec()-current_trajectory_->header().timestamp_sec() < kSimControlIntervalMs*0.001 ){
  //    AERROR << "trajectory's time interval is too small";
  //    return;
  // }
  // auto current_time = Clock::NowInSeconds();
  // if(trajectory->header().timestamp_sec()>current_time){
  //   AERROR << "trajectory->header().timestamp_sec() is larger than current time ";
  // }
    current_trajectory_ = trajectory;
    prev_point_index_ = 0;
    next_point_index_ = 0;
    received_planning_ = true;
    send_dummy_prediction_ = false;
    if(_is_enable_prediction_gt){
      sim_prediction_timer_->Stop();
    }
}

  void ControlGTCore::OnPredictionObstacles(const std::shared_ptr<apollo::prediction::PredictionObstacles> &obstacles){
    std::lock_guard<std::mutex> lock(mutex_);
    if (obstacles->prediction_obstacle().size()>0)
      received_simulator_signal_=true;
  }

  void ControlGTCore::OnPerceptionObstacles(const std::shared_ptr<apollo::perception::PerceptionObstacles> &obstacles){
    std::lock_guard<std::mutex> lock(mutex_);
    if(obstacles->perception_obstacle().size()>0){
      received_simulator_signal_=true;
    }
  }


void ControlGTCore::Freeze() {
  AERROR << "Freeze: next_point_.v:"<<next_point_.v();
  next_point_.set_v(0.0);
  next_point_.set_a(0.0);
  prev_point_ = next_point_;
}

void ControlGTCore::RunOnce() {
  std::lock_guard<std::mutex> lock(mutex_);
    if(!start_point_from_localization_) {
        localization_reader_->Observe();
            InitStartPoint(next_point_.has_v() ? next_point_.v() : 0.0,
         next_point_.has_a() ? next_point_.a() : 0.0);
        return;
    }
  TrajectoryPoint trajectory_point;
  Chassis::GearPosition gear_position;

  if (!PerfectControlModel(&trajectory_point, &gear_position)) {
    //AERROR << "Failed to calculate next point with perfect control model";
    return;
  }

  PublishChassis(trajectory_point.v(), gear_position);
  PublishLocalization(trajectory_point);
  if(received_planning_&&received_simulator_signal_){
    LogTrajectoryPoint(&trajectory_point);
  }
}

void ControlGTCore::LogTrajectoryPoint(TrajectoryPoint* point){
    AERROR << "LogTrajectoryPoint x:"<<point->path_point().x()<<", y:"<<point->path_point().y()<<",z:"<<point->path_point().z();
    FILE *fp = fopen(_log_file_name, "a+");
    fprintf(fp,"######\nindex:%d\ntimestamp:%lf\nposition_x:%lf\nposition_y:%lf\nposition_z:%lf\nheading:%lf\nspeed:%lf\n",
        _waypoint_index,
        Clock::NowInSeconds(),
        point->path_point().x(),
        point->path_point().y(),
        point->path_point().z(),
        point->path_point().theta(),
        point->v()
        );
    _waypoint_index = _waypoint_index + 1;
    fclose(fp);
}

bool ControlGTCore::PerfectControlModel(TrajectoryPoint* point,
                                     Chassis::GearPosition* gear_position) {
                                
  // Result of the interpolation.
  auto current_time = Clock::NowInSeconds();
  const auto& trajectory = current_trajectory_->trajectory_point();
  *gear_position = current_trajectory_->gear();
  //AERROR << "PerfectControlModel trajectory.size():"<<trajectory.size();
  if (!(received_planning_&&received_simulator_signal_)) {
    prev_point_ = next_point_;
    AERROR << "received_simulator_signal_:"<<received_simulator_signal_;
  } else {
    if (current_trajectory_->estop().is_estop() ||
        next_point_index_ >= trajectory.size()) {
      // Freeze the car when there's an estop or the current trajectory has
      // been exhausted.
      Freeze();
    } else {
      while (next_point_index_ < trajectory.size() &&
             current_time > trajectory.Get(next_point_index_).relative_time() +
                                current_trajectory_->header().timestamp_sec()) {
        ++next_point_index_;
      }

      if (next_point_index_ >= trajectory.size()) {
        next_point_index_ = trajectory.size() - 1;
      }

      if (next_point_index_ == 0) {
        return false;
      }

      prev_point_index_ = next_point_index_ - 1;
      next_point_ = trajectory.Get(next_point_index_);
      prev_point_ = trajectory.Get(prev_point_index_);
    }
  }

  if (current_time > next_point_.relative_time() +
                         current_trajectory_->header().timestamp_sec()) {
    // Don't try to extrapolate if relative_time passes last point
    AERROR << "Don't try to extrapolate if relative_time passes last point";
    *point = next_point_;
  } else {
   AERROR << "InterpolateUsingLinearApproximation prev_point_.v:"<<prev_point_.v()<<", next_point_.v():"<<next_point_.v();
    *point = InterpolateUsingLinearApproximation(
        prev_point_, next_point_,
        current_time - current_trajectory_->header().timestamp_sec());
  }
  return true;
}

void ControlGTCore::PublishChassis(double cur_speed,
                                Chassis::GearPosition gear_position) {
  auto chassis = std::make_shared<Chassis>();
  FillHeader("SimControl", chassis.get());

  chassis->set_engine_started(true);
  chassis->set_driving_mode(Chassis::COMPLETE_AUTO_DRIVE);
  chassis->set_gear_location(gear_position);

  chassis->set_speed_mps(static_cast<float>(cur_speed));
  if (gear_position == canbus::Chassis::GEAR_REVERSE) {
    chassis->set_speed_mps(-chassis->speed_mps());
  }

  chassis->set_throttle_percentage(0.0);
  chassis->set_brake_percentage(0.0);

  chassis_writer_->Write(chassis);
}

void ControlGTCore::PublishLocalization(const TrajectoryPoint& point) {
  auto localization = std::make_shared<LocalizationEstimate>();
  FillHeader("ControlGT", localization.get());

  auto* pose = localization->mutable_pose();
  auto prev = prev_point_.path_point();
  auto next = next_point_.path_point();
  // Set position
  pose->mutable_position()->set_x(point.path_point().x());
  pose->mutable_position()->set_y(point.path_point().y());
  pose->mutable_position()->set_z(point.path_point().z());
  // Set orientation and heading
  double cur_theta = point.path_point().theta();

  if (FLAGS_use_navigation_mode) {
    double flu_x = point.path_point().x();
    double flu_y = point.path_point().y();

    Eigen::Vector2d enu_coordinate =
        common::math::RotateVector2d({flu_x, flu_y}, cur_theta);

    enu_coordinate.x() += adc_position_.x();
    enu_coordinate.y() += adc_position_.y();
    pose->mutable_position()->set_x(enu_coordinate.x());
    pose->mutable_position()->set_y(enu_coordinate.y());
  }

  Eigen::Quaternion<double> cur_orientation =
      HeadingToQuaternion<double>(cur_theta);
  pose->mutable_orientation()->set_qw(cur_orientation.w());
  pose->mutable_orientation()->set_qx(cur_orientation.x());
  pose->mutable_orientation()->set_qy(cur_orientation.y());
  pose->mutable_orientation()->set_qz(cur_orientation.z());
  pose->set_heading(cur_theta);

  // Set linear_velocity
  pose->mutable_linear_velocity()->set_x(std::cos(cur_theta) * point.v());
  pose->mutable_linear_velocity()->set_y(std::sin(cur_theta) * point.v());
  pose->mutable_linear_velocity()->set_z(0);

  // Set angular_velocity in both map reference frame and vehicle reference
  // frame
  pose->mutable_angular_velocity()->set_x(0);
  pose->mutable_angular_velocity()->set_y(0);
  pose->mutable_angular_velocity()->set_z(point.v() *
                                          point.path_point().kappa());

  TransformToVRF(pose->angular_velocity(), pose->orientation(),
                 pose->mutable_angular_velocity_vrf());

  // Set linear_acceleration in both map reference frame and vehicle reference
  // frame
  auto* linear_acceleration = pose->mutable_linear_acceleration();
  linear_acceleration->set_x(std::cos(cur_theta) * point.a());
  linear_acceleration->set_y(std::sin(cur_theta) * point.a());
  linear_acceleration->set_z(0);

  TransformToVRF(pose->linear_acceleration(), pose->orientation(),
                 pose->mutable_linear_acceleration_vrf());

  localization_writer_->Write(localization);

  adc_position_.set_x(pose->position().x());
  adc_position_.set_y(pose->position().y());
  adc_position_.set_z(pose->position().z());

}

void ControlGTCore::PublishDummyPrediction() {
  auto prediction = std::make_shared<PredictionObstacles>();
  {
    std::lock_guard<std::mutex> lock(mutex_);
    if (!send_dummy_prediction_) {
      return;
    }
    FillHeader("SimPrediction", prediction.get());
  }
  prediction_writer_->Write(prediction);
}
}  // namespace controlGT
}  // namespace apollo
