"""
mutator.mutator
"""
import abc
import copy
from hashlib import new
import random
import config
import math

from callable import Callable

from dataparser import scenario
import logger 

log = logger.get_logger("autoware-fuzzer")

def prob(num):
    if random.randint(0, 99) < num:
        return True
    else:
        return False

# define different atomic operations for mutate different kinds of elements in the scenario


class Mutator(metaclass=abc.ABCMeta):
    def __init__(self):
        pass

    @abc.abstractmethod
    def mutation(self, _obj):
        pass


class MutatorFactory:
    """ The factory class for creating different mutators """
    registry = {}

    @classmethod
    def register(cls, name: str) -> Callable:
        def inner_wrapper(wrapper_class: Mutator) -> Callable:
            cls.registry[name] = wrapper_class
            return wrapper_class

        return inner_wrapper

    @classmethod
    def create_mutator(cls, name: str, **kwargs) -> 'Mutator':
        exec_class = cls.registry[name]
        mutator = exec_class(**kwargs)
        return mutator


@MutatorFactory.register('pedestrian')
class PedestrianMutator(Mutator):
    def __init__(self):
        self.variant = [
            "Deer",
            "Turkey",
            "Bill",
            "Bob",
            "EntrepreneurFemale",
            "Howard",
            "Johny",
            "Pamela",
            "Presley",
            "Robin",
            "Stephen",
            "Zoe"
        ]
    def mutation(self, _config):
        obj_list = []
        for obj in _config.element:
            new_obj = copy.deepcopy(obj)

            if prob(config.PROB_PedestrianVariant):
                new_obj.variant = random.choice(self.variant)

            if prob(config.PROB_PedestrianSpeed):
                prev_speed = 1.5
                for wayp in new_obj.wayPoints:
                    prev_speed += random.uniform(-0.5, 0.5)
                    if prev_speed < 0:
                        prev_speed = 0
                    wayp.speed = prev_speed

            obj_list.append(obj)

        return obj_list

@MutatorFactory.register('obstacle')
class StaticObstacleMutator(Mutator):
    def mutation(self, _config):
        obj_list = []

        # only mutate traffic cone yet
        for obj in _config.element:
            new_obj = copy.deepcopy(obj)
            if new_obj.type == "TrafficCone":
                if prob(config.PROB_TrafficConePosition):
                    log.info("mutation strategy: Change cone position")
                    new_obj.transform.position['x'] += random.uniform(-3.0, 3.0)
                    new_obj.transform.position['z'] += random.uniform(-3.0, 3.0)
                    new_obj.transform.rotation['y'] += random.uniform(-360.0, 360.0)
                    new_obj.transform.rotation['y'] %= 360.0
            obj_list.append(new_obj)
        
        if prob(config.PROB_TrafficConeAdd) and len(obj_list) < config.MAX_OBSTACLE_NUM:
            print("[+] Cone Add")
            log.info("mutation strategy: Add cone")
            tmp_obj = scenario.StaticObstacle()
            tmp_obj.uid = ""
            tmp_obj.type = "TrafficCone"
            tmp_obj.spawned = True
            tmp_obj.policies = []

            tmp_point = random.choice(random.choice(_config.lane)._lane_position)
            tmp_position = dict(x = tmp_point[0], y = tmp_point[2], z = tmp_point[1])
            tmp_rotation = dict(x = 0.0, y = 0.0, z = 0.0)
            tmp_obj.transform = scenario.Transform(tmp_position, tmp_rotation)
            obj_list.append(tmp_obj)

        if prob(config.PROB_TrafficConeDelete) and len(obj_list) > 0:
            print("[+] Cone Delete")
            log.info("mutation strategy: Delete cone")
            obj_list.remove(random.choice(obj_list))
            
        return obj_list

@MutatorFactory.register('weather')
class WeatherMutator(Mutator):
    def mutation(self, _config):
        assert isinstance(_config.element[0], scenario.Weather)
        new_obj = copy.deepcopy(_config.element[0])
        
        if prob(config.PROB_WeatherRain):
            new_obj.rain = random.uniform(0, 1)
            log.info("mutation strategy: reset rain")
        if prob(config.PROB_WeatherFog):
            new_obj.fog = random.uniform(0, 1)
            log.info("mutation strategy: reset fog")
        if prob(config.PROB_WeatherWetness):
            new_obj.wetness = random.uniform(0, 1)
            log.info("mutation strategy: reset wetness")
        if prob(config.PROB_WeatherCloudiness):
            new_obj.cloudiness = random.uniform(0, 1)
            log.info("mutation strategy: reset cloudiness")
        
        # damage not mutate yet
        new_obj.damage = 0
        
        return [new_obj]

@MutatorFactory.register('time')
class TimeMutator(Mutator):
    def mutation(self, _config):
        assert isinstance(_config.element[0], scenario.Time)
        new_obj = copy.deepcopy(_config.element[0])
        changed = False
        if prob(config.PROB_TimeMonth):
            changed = True
            new_obj.month = random.randint(1, 12)
        if prob(config.PROB_TimeDay):
            changed = True
            new_obj.day = random.randint(1, 28)
        if prob(config.PROB_TimeHour):
            changed = True
            new_obj.hour = random.randint(0, 23)
        if prob(config.PROB_TimeMinute):
            changed = True
            new_obj.minute = random.randint(0, 59)
        if prob(config.PROB_TimeSecond):
            changed = True
            new_obj.second = random.randint(0, 59)
        if changed == True:
            log.info("mutation strategy: Change time")
        return [new_obj]

@MutatorFactory.register('npc')
class NPCMutator(Mutator):
    def __init__(self):
        self.variant = [
            "Bicyclist",
            "BoxTruck",
            "Hatchback",
            "Jeep",
            "SUV",
            "SchoolBus",
            "Sedan"
        ]

    def mutation(self, _config):
        obj_list = copy.deepcopy(_config.element)

        if prob(config.PROB_NPCAdd) and len(obj_list) < config.MAX_NPC_NUM:
            log.info("mutation strategy: Add npc")
            tmp_obj = scenario.NPCVehicle()
            tmp_obj.uid = ""
            tmp_obj.variant = random.choice(self.variant)
            tmp_obj.parameterType = ""
            tmp_obj.color = dict(r = 0, g = 0, b = 0)
            tmp_obj.wayPoints = []

            while True:
                tmp_lane_pos = random.choice(_config.lane)._lane_position
                tmp_idx = random.choice(range(len(tmp_lane_pos)))
                tmp_point = tmp_lane_pos[tmp_idx]
                tmp_position = dict(x = tmp_point[0], y = tmp_point[2], z = tmp_point[1])
                tmp_rotation = dict(x = 0.0, y = 0.0, z = 0.0)

                def calc_distance(car1, car2):
                    x1 = car1["x"]
                    y1 = car1["y"]
                    z1 = car1["z"]
                    x2 = car2["x"]
                    y2 = car2["y"]
                    z2 = car2["z"]
                    dis = math.pow(x1 - x2, 2) + math.pow(y1 - y2, 2) + math.pow(z1 - z2, 2)
                    dis = math.sqrt(dis)
                    return dis

                # prevent not overlapping with EGO / NPCs
                dis = calc_distance(tmp_position, ego_info.transform.position)
                print(dis)
                if dis < 2:
                    continue

                for npc in npc_info:
                    dis = min(dis, calc_distance(tmp_position, npc.transform.position))
                if dis < 2:
                    continue

                if tmp_idx + 1 < len(tmp_lane_pos):
                    delta_x = tmp_lane_pos[tmp_idx + 1][0] - tmp_point[0]
                    delta_z = tmp_lane_pos[tmp_idx + 1][2] - tmp_point[2]
                    tmp_rotation['y'] = math.atan2(delta_x, delta_z) * 180 / math.pi
                tmp_obj.transform = scenario.Transform(tmp_position, tmp_rotation)
                break

            tmp_obj.behaviour = scenario.NPCBehaviour()
            tmp_obj.behaviour.name = "NPCLaneFollowBehaviour"
            tmp_obj.behaviour.maxSpeed = random.uniform(0, 10)
            tmp_obj.behaviour.isLaneChange = random.choice([True, False])
            obj_list.append(tmp_obj)
        
        if prob(config.PROB_NPCDelete) and len(obj_list) > 0:
            # print("[+] NPC Delete")
            log.info("mutation strategy: Delete npc")
            obj_list.remove(random.choice(obj_list))
        
        for obj in obj_list:
            if prob(config.PROB_NPCVariant):
                obj.variant = random.choice(self.variant)
            
            if prob(config.PROB_NPCColor):
                obj.color["r"] = random.uniform(0, 1)
                obj.color["g"] = random.uniform(0, 1)
                obj.color["b"] = random.uniform(0, 1)

            if obj.behaviour.name == "NPCLaneFollowBehaviour":
                if prob(config.PROB_NPCMaxSpeed):
                    obj.behaviour.maxSpeed = random.uniform(0, 10)
                    obj.behaviour.isLaneChange = random.choice([True, False])
                    log.info("mutation strategy: Update npc max speed to {}".format(obj.behaviour.maxSpeed))
            elif obj.behaviour.name == "NPCWaypointBehaviour":
                if prob(config.PROB_NPCSpeed):
                    new_speed = random.uniform(0, 10)
                    log.info("mutation strategy: Change npc waypoint speed to {}".format(new_speed))
                    for wayp in obj.wayPoints:
                        wayp.speed = new_speed

        return obj_list

@MutatorFactory.register('ego')
class EgoMutator(Mutator):
    def mutation(self, _config):
        assert isinstance(_config.element[0], scenario.EgoVehicle)
        new_obj = copy.deepcopy(_config.element[0])

        return [new_obj]

@MutatorFactory.register('trafficlight')
class TrafficLightMutator(Mutator):
    def mutation(self, _config):
        obj_list = []
        for obj in _config.element:
            new_obj = copy.deepcopy(obj)

            for policy in new_obj.policies:
                prev_color = ""

                if policy.action == 'state':
                    if prob(config.PROB_TrafficLightState):
                        policy.value = random.choice(['green', 'red', 'yellow'])
                        log.info("mutation strategy: Change traffic light state")
                    prev_color = policy.value
                    
                if policy.action == 'wait':
                    if prob(config.PROB_TrafficLightValue):
                        if prev_color == 'yellow':
                            policy.value = random.randint(0, 3)
                        else:
                            policy.value = random.randint(0, 30)

            obj_list.append(new_obj)

        return obj_list

class MutationConfig:
    def __init__(self, _obj, _lane):
        self.element = _obj
        self.lane = _lane


def mutation(_scenario, _lane=None, _rand=1):
    # the mutation on npc vehicles should make it still follow the lane
    new_scenarios = []
    assert isinstance(_scenario, scenario.Scenario)
    # generate 5 new testcases by default
    for i in range(_rand):
        curr_elements = _scenario.elements
        new_scenario = copy.deepcopy(_scenario)
        new_scenario.hash = None
        new_scenario.json_obj = None
        new_scenario.elements = {}

        # print("---------------------")
        global ego_info, npc_info
        ego_info = curr_elements["ego"][0]
        npc_info = curr_elements["ego"]
        for name, ele_list in curr_elements.items():
            mutator = MutatorFactory.create_mutator(name)
            new_list = mutator.mutation(MutationConfig(ele_list, _lane))
            new_scenario.elements[name] = new_list
            # print(name, len(new_list))
            # log.info("Added element: {} . Current length: ")

        # print(new_scenario.get_hash())
        new_scenarios.append(new_scenario)
        log.info("mutation output: {}".format(new_scenario.get_hash()))
    return new_scenarios

# if __name__ == "__main__":
#    elements = [scenario.Weather()]
#    mutation(elements, '')
