import rospy
import roslaunch
import os
from time import sleep
from carla_msgs.msg import CarlaCollisionEvent
import carla
from math import sqrt
import pexpect
from rosgraph_msgs.msg import Log

HOST = '127.0.0.1'
PORT = 2000

AUTOWARE_FUZZING = os.getenv("AUTOWARE_FUZZER")
odr_list = []
with open(AUTOWARE_FUZZING + 'commonroad_osm/list.txt', 'r') as f:
    for odr in f.readlines():
        if not odr.startswith('# '):
            odr_list.append(odr.strip('\n') + '.xodr')

class pcd_generator:
    def __init__(self,odr_list):
        assert(len(odr_list)>0)
        self.odr_list = odr_list
        self.current_odr_id = 0 
        self.current_odr = self.odr_list[self.current_odr_id]
        self.uuid = None
        self.role_name = 'ego_vehicle'
        self.ego = None
        self.launch_pcl_recorder = None
        self.pcl_recorder_running = False
        self.pcl_recorder_path = '/xxxxx/pcd_map_generation.launch'
        self.spawn_points = None
        self.carla_client = None
        self.in_collision = False
        self.carla_process = None
        self.collision_subscriber = rospy.Subscriber(
            "/carla/{}/collision".format(self.role_name), CarlaCollisionEvent, self.on_collision)
    
    def run_carla(self):
        self.carla_process = pexpect.spawn('/opt/carla-simulator/CarlaUE4.sh')

    def stop_carla(self):
        if self.carla_process != None:
            self.carla_process.close(force=True)
            self.carla_process = None

    def on_collision(self):
        print("collision detect!")
        if self.in_collision == False:
            self.in_collision = True

    def clean_records(self):
        cmd = 'rm /tmp/pcl_capture/*'
        os.system(cmd)

    def run_pcl_recorder(self,spawn_point):
        if self.pcl_recorder_running == False:
            self.pcl_recorder_running = True
            if self.uuid == None:
                self.uuid = roslaunch.rlutil.get_or_generate_uuid(None, False)
            self.launch_pcl_recorder = roslaunch.scriptapi.ROSLaunch()
            odr_path = AUTOWARE_FUZZING + 'opendrive/' + self.current_odr
            args = [self.pcl_recorder_path,"odr_path:={}".format(odr_path),"spawn_point:={}".format(spawn_point)]
            launch_file = roslaunch.rlutil.resolve_launch_arguments(args)[0]
            self.launch_pcl_recorder.parent = roslaunch.parent.ROSLaunchParent(self.uuid,[( launch_file, args[1:] )])
            self.launch_pcl_recorder.start()

    def stop_pcl_recorder(self):
        if self.pcl_recorder_running:
            self.launch_pcl_recorder.parent.shutdown()
            self.launch_pcl_recorder.stop()
            del self.launch_pcl_recorder
            self.launch_pcl_recorder = None
            self.pcl_recorder_running = False
        
    def generate_pcd_map_after_record(self):
        cmd = "pcl_concatenate_points_pcd /tmp/pcl_capture/point_*.pcd && \\"
        cmd +="pcl_voxel_grid -leaf 0.1,0.1,0.1 output.pcd {}.pcd".format(AUTOWARE_FUZZING + 'pcd/' + self.current_odr.strip('.xodr'))
        os.system(cmd)
        self.clean_records()
    
    def generate_pcd_map_one_point(self,flag):
        cmd = "pcl_concatenate_points_pcd /tmp/pcl_capture/capture*.pcd"
        os.system(cmd)
        cmd = "pcl_voxel_grid -leaf 0.1,0.1,0.1 output.pcd /tmp/pcl_capture/point_{}.pcd".format(flag)
        os.system(cmd)
        cmd = "rm /tmp/pcl_capture/capture*.pcd"
        os.system(cmd)

    def change_xodr(self):
        return

    def spin(self):
        rospy.spin()

    def init_client(self):
        self.carla_client = carla.Client(HOST,PORT)

    def get_spawn_points(self):
        if self.carla_client != None:
            opendrive = open(AUTOWARE_FUZZING + 'opendrive/' + self.current_odr).read()
            self.carla_client.generate_opendrive_world(opendrive)
            world = self.carla_client.get_world()
            _map = world.get_map()
            self.spawn_points = _map.get_spawn_points()

    def get_ego_handle(self):
        if self.carla_client != None:
            self.world = self.carla_client.get_world()
            ego_list = self.world.get_actors().filter('vehicle.*')
            if len(ego_list) != 0:
                self.ego = ego_list[0]
            else:
                self.ego = None
            if self.ego == None:
                return False
            else:
                return True
        return True

    def ego_existing_and_moving(self):
        if self.carla_client != None:
            if self.get_ego_handle() == False :
                return False      
            if self.ego == None:
                return False
            return self.ego.is_alive         
    
    def dist(self,loc1,loc2):
        d_2 = (loc1.x-loc2.x)**2+(loc1.y-loc2.y)**2
        return sqrt(d_2)
    
    def restart_carla(self):
        if self.carla_process!=None:
            self.stop_carla()
        self.run_carla()
        self.carla_process.expect("Disabling core dumps.")
        sleep(10)
    

    def run_one_map(self):
        self.init_client()
        self.get_spawn_points()
        self.clean_records()
        for point in self.spawn_points:
            try:
                self.carla_client.reload_world()
            except:
                self.restart_carla()
                self.init_client()
                opendrive = open(AUTOWARE_FUZZING + 'opendrive/' + self.current_odr).read()
                self.carla_client.generate_opendrive_world(opendrive)
                sleep(5)
            x = point.location.x
            y = point.location.y
            z = point.location.z
            pitch = point.rotation.pitch
            yaw = point.rotation.yaw
            roll = point.rotation.roll
            sp_str = "{},{},{},{},{},{}".format(x,y,z,roll,pitch,yaw)
            print(sp_str)
            self.run_pcl_recorder(sp_str)
            time = 0
            timeout = 600
            # wait_for_ego:
            self.ego = None
            while time < 20:
                if self.get_ego_handle():
                    break
                sleep(1)
                time += 1
            if self.ego==None and time == 10:
                break
            time = 0
            self.in_collision = False
            last_transform = None
            last10_transform = None
            if_break = False
            while time < timeout and self.carla_process.isalive() and if_break==False:
                if not self.ego_existing_and_moving():
                    break
                # Falling check every 1 sec
                for i in range(10):
                    current_transform = self.ego.get_transform()
                    if last_transform == None:
                        last_transform = current_transform
                    else:
                        if abs(current_transform.location.z - last_transform.location.z) >= 5:
                            if_break = True
                            break
                        last_transform = current_transform
                    sleep(1)
                    time += 1
                # moving check every 10 sec
                if last10_transform == None:
                    last10_transform = current_transform
                elif self.dist(current_transform.location, last10_transform.location) < 0.05:
                    break
                last10_transform = current_transform
            self.stop_pcl_recorder()
            point_flag = "{}_{}_{}_{}_{}_{}".format(x,y,z,roll,pitch,yaw)
            self.generate_pcd_map_one_point(point_flag)
        self.generate_pcd_map_after_record()
    
    def run(self):
        for odr in self.odr_list:
            self.current_odr = odr
            while not os.path.isfile('{}.pcd'.format(AUTOWARE_FUZZING + 'pcd/' + self.current_odr.strip('.xodr'))):
                try:
                    self.restart_carla()
                    self.run_one_map()
                except :
                    # destroy all nodes and re-start
                    self.destroy()
        self.destroy()

    def destroy(self):
        if self.carla_process != None:
            self.stop_carla()
        # self.collision_subscriber.unregister()
        self.stop_pcl_recorder()
        self.clean_records()

        

if __name__ == '__main__':
    generator = pcd_generator(odr_list)
    generator.run()
            