# coding=utf-8
# !/usr/bin/env python3
"""
Entry point of the apollo-fuzzer
"""
import os

import click
import heapq
import sys
import timeout_decorator
import time
import subprocess

import config as _Config
import logger as _Logger
import mutator as _Mutator
import dataparser as _Parser
from run_vse import VSERunner
LOG = _Logger.get_logger(_Config.__prog__)


class Fuzzer:
    def __init__(self, _corpus, _lane, _workdir, _feedback):
        self.corpus = _Parser.corpus_parser(_corpus)
        self.lane = _Parser.lane_parser(_lane)
        self.workdir = _workdir
        self.feedback = _feedback
        self.init_workdir()
        # the seed queue
        self.queue = []
        # seeds that can trigger collision
        self.collisions = []
        # testcases that have already been executed, used for testcase deduplication
        self.hashes = set([])

        self.simtime = 0
        self.simulator_time = 0
        self.simprocess = None
        self.env = os.environ.copy()
        self.env["DISPLAY"] = ":0"
        # self.simdir = os.path.join(os.path.expanduser('~'), "Build/simulator")
        self.simdir = os.path.join(os.path.expanduser('~'), "svlsimulator-linux64-2021.2.2/simulator")

    def init_workdir(self):
        if os.listdir(self.workdir):
            LOG.info("Working Directory Not Empty!")

            # resume mode
            LOG.info("Resuming working directory...")
            assert os.path.exists(self.workdir + "/queue")
            assert os.path.exists(self.workdir + "/collision")
            assert os.path.exists(self.workdir + "/info")
            assert not os.path.exists(self.workdir + "/collision_old")
            os.rename(self.workdir + "/collision", self.workdir + "/collision_old")
            os.mkdir(self.workdir + "/collision")
            return
        # create dirs for queue/collisions
        os.mkdir(self.workdir + "/queue")
        os.mkdir(self.workdir + "/collision")
        # create fuzzing info file (for logging fuzzing status)
        f = open(self.workdir + "/info", "w")
        f.close()

    def run_instance(self, _seed, _timeout=None):

        # restart simulation every 1h
        if time.time() - self.simtime >= 3600:
            LOG.info("Restart LGSVL simulation")
            # subprocess.run(["/bin/bash", "-x", os.path.join(os.path.expanduser('~'), "apollo-fuzzer/tools/restart_simulation.sh")])
            # subprocess.run(["python3", os.path.join(os.path.expanduser('~'), "apollo-fuzzer/tools/wise.py"), "t640", "restart"])
            subprocess.run(["/usr/bin/zsh", "-c", "resim"])
            time.sleep(10)
            self.simtime = time.time()

        # check if it is a already-verified testcase
        if _seed.get_hash() in self.hashes:
            return None
        # run a seed with lgsvl simulation
        # define different feedback functions & use args to determine which one to use during runtime
        # on collision?
        score = 0
        vse_runner = VSERunner(_seed, self.workdir, self.feedback)
        try:
            score = vse_runner.run(30)
            if self.feedback != "none":
                LOG.info("Score: " + str(score))
        except timeout_decorator.timeout_decorator.TimeoutError:
            LOG.info("vse_runner is timeout!")
            self.simtime = -1
            self.simulator_time = -1

        return score

    def queue_push(self, _seed):
        # update the priority queue
        heapq.heappush(self.queue, (_seed.score, _seed))
        # update the queue dir
        _seed.store(self.workdir + "/queue/" + str(_seed.get_hash()))
        return

    def queue_pop(self):
        # update the priority queue
        (curr_score, curr_seed) = heapq.heappop(self.queue)
        # update the queue dir, if needed
        # if os.path.isfile(self.workdir + "/queue/" + str(curr_seed.get_hash())):
        return curr_seed

    def execute(self):
        # run a seed to init environment
        if len(self.corpus) > 0:
            LOG.info("running initial seed")
            self.run_instance(self.corpus[0])
            os.system("rm -f " + self.workdir + "/collision/* 2>/dev/null")

        # first initialize the seed queue of fuzzing by executing the initial seeds in the corpus
        for scenario in self.corpus:
            LOG.info(scenario.seed_path)
            res = self.run_instance(scenario)
            if res is not None:
                scenario.verified = True
                scenario.score = res
                self.queue_push(scenario)
                self.hashes.add(scenario.get_hash())

        # the fuzzing loop
        while len(self.queue) > 0:
            if len(self.queue) < 1000:
                mutate_cnt = 5
            else:
                mutate_cnt = 1
            curr = self.queue_pop()
            # mutation on curr scenario
            for new_scenario in _Mutator.mutation(curr, self.lane, mutate_cnt):
                res = self.run_instance(new_scenario)
                if res is not None:
                    LOG.info("Original seed: {}".format(new_scenario.seed_path))
                    LOG.info("Saved seed: {}".format(new_scenario.get_hash()))
                    LOG.info("Queue size: {}".format(len(self.queue)))
                    new_scenario.verified = True
                    new_scenario.score = res
                    self.queue_push(new_scenario)
                    self.hashes.add(new_scenario.get_hash())
        return


@click.group(help=_Config.__description__)
@click.version_option(version=_Config.__version__, prog_name=_Config.__prog__)
def cli():
    pass


@cli.command()
# Each seed is in a json-format, specifying the initial simulation scenarios for fuzzing @GPF
@click.option('--corpus',
              type=click.Path(file_okay=False, exists=True),
              required=True,
              help='The directory which contains the seeds(scenarios) for simulation-fuzzing.'
              )
# The lane info is in a json format which indicates the lane info of the map @GPF
@click.option('--lane',
              type=click.Path(dir_okay=False, exists=True),
              required=True,
              help='The file which contains the lane info of the map.'
              )
# The working directory for fuzzer
@click.option('--out',
              type=click.Path(file_okay=False, exists=True),
              required=True,
              help='The working directory for fuzzer'
              )
# The feedback strategy for fuzzer
@click.option('--feedback',
              type=click.Choice(['none', 'avfuzzer']),
              default='none',
              help='The feedback strategy for fuzzer')
@timeout_decorator.timeout(10800, use_signals=False)
def fuzz(corpus, lane, out, feedback):
    """ Execute Simulation Fuzzing """
    LOG.info("Fuzzing Start.")
    LOG.info("Feedback: " + feedback)
    fuzzer = Fuzzer(corpus, lane, out, feedback)
    fuzzer.execute()
    LOG.info("Fuzzing End.")


if __name__ == "__main__":
    # for pycharm debugging
    fuzz(sys.argv[1:])
    # for bash
    # cli()
