# Copyright Iris contributors
#
# This file is part of Iris and is released under the BSD license.
# See LICENSE in the root of the repository for full licensing details.
"""Warnings specific to the :mod:`iris` package.

PLEASE NAMESPACE ALL WARNING CLASSES (i.e. prefix with Iris...).
"""


class IrisUserWarning(UserWarning):
    r"""Base class for :class:`UserWarning` generated by Iris."""

    pass


class IrisLoadWarning(IrisUserWarning):
    """Any warning relating to loading."""

    pass


class IrisSaveWarning(IrisUserWarning):
    """Any warning relating to saving."""

    pass


class IrisCfWarning(IrisUserWarning):
    """Any warning relating to :term:`CF Conventions` ."""

    pass


class IrisIgnoringWarning(IrisUserWarning):
    """Any warning that involves an Iris operation not using some information.

    E.g. :class:`~iris.aux_factory.AuxCoordFactory` generation disregarding
    bounds.
    """

    pass


class IrisDefaultingWarning(IrisUserWarning):
    """Any warning that involves Iris changing invalid/missing information.

    E.g. creating a :class:`~iris.coords.AuxCoord` from an invalid
    :class:`~iris.coords.DimCoord` definition.
    """

    pass


class IrisVagueMetadataWarning(IrisUserWarning):
    """Warnings where object metadata may not be fully descriptive."""

    pass


class IrisUnsupportedPlottingWarning(IrisUserWarning):
    """Warnings where support for a plotting module/function is not guaranteed."""

    pass


class IrisImpossibleUpdateWarning(IrisUserWarning):
    """Warnings where it is not possible to update an object.

    Mainly generated during regridding where the necessary information for
    updating an :class:`~iris.aux_factory.AuxCoordFactory` is no longer
    present.
    """

    pass


class IrisGeometryExceedWarning(IrisUserWarning):
    """:mod:`iris.analysis.geometry` warnings about geometry exceeding dimensions."""

    pass


class IrisMaskValueMatchWarning(IrisUserWarning):
    """Warnings where the value representing masked data is actually present in data."""

    pass


class IrisCfLoadWarning(IrisCfWarning, IrisLoadWarning):
    """Any warning relating to both loading and :term:`CF Conventions` ."""

    pass


class IrisCfSaveWarning(IrisCfWarning, IrisSaveWarning):
    """Any warning relating to both saving and :term:`CF Conventions` ."""

    pass


class IrisCfInvalidCoordParamWarning(IrisCfLoadWarning):
    """Warnings where incorrect information for CF coord construction is in a file."""

    pass


class IrisCfMissingVarWarning(IrisCfLoadWarning):
    """Warnings where a CF variable references another variable that is not in the file."""

    pass


class IrisCfLabelVarWarning(IrisCfLoadWarning, IrisIgnoringWarning):
    """Warnings where a CF string/label variable is being used inappropriately."""

    pass


class IrisCfNonSpanningVarWarning(IrisCfLoadWarning, IrisIgnoringWarning):
    """Warnings where a CF variable is ignored because it does not span the required dimension."""

    pass


class IrisIgnoringBoundsWarning(IrisIgnoringWarning):
    """Warnings where bounds information has not been used by an Iris operation."""

    pass


class IrisCannotAddWarning(IrisIgnoringWarning):
    """Warnings where a member object cannot be added to a :class:`~iris.cube.Cube` ."""

    pass


class IrisGuessBoundsWarning(IrisDefaultingWarning):
    """Warnings where Iris has filled absent bounds information with a best estimate."""

    pass


class IrisPpClimModifiedWarning(IrisSaveWarning, IrisDefaultingWarning):
    """Warnings where a climatology has been modified while saving :term:`Post Processing (PP) Format` ."""

    pass


class IrisFactoryCoordNotFoundWarning(IrisLoadWarning):
    """Warnings where a referenced factory coord can not be found when loading a variable in :term:`NetCDF Format`."""

    pass


class IrisNimrodTranslationWarning(IrisLoadWarning):
    """For unsupported vertical coord types in :mod:`iris.file_formats.nimrod_load_rules`.

    (Pre-dates the full categorisation of Iris UserWarnings).
    """

    pass


class IrisUnknownCellMethodWarning(IrisCfLoadWarning):
    """If a loaded :class:`~iris.coords.CellMethod` is not one the method names known to Iris.

    (Pre-dates the full categorisation of Iris UserWarnings).
    """

    pass


class IrisSaverFillValueWarning(IrisMaskValueMatchWarning, IrisSaveWarning):
    """For fill value complications during Iris file saving :term:`NetCDF Format`.

    (Pre-dates the full categorisation of Iris UserWarnings).
    """

    pass
