# Copyright Iris contributors
#
# This file is part of Iris and is released under the BSD license.
# See LICENSE in the root of the repository for full licensing details.
"""Unit tests for the `iris.cube.Cube` class aggregated_by method."""

from unittest import mock

from cf_units import Unit
import numpy as np
from numpy import ma
import pytest

from iris._lazy_data import as_lazy_data
import iris.analysis
from iris.analysis import MEAN, SUM, Aggregator, WeightedAggregator
import iris.aux_factory
import iris.coords
from iris.coords import AncillaryVariable, AuxCoord, CellMeasure, DimCoord
from iris.cube import Cube
from iris.tests import _shared_utils
from iris.tests.stock import realistic_4d


class Test_aggregated_by:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.cube = Cube(np.arange(44).reshape(4, 11))

        val_coord = AuxCoord([0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 1], long_name="val")
        label_coord = AuxCoord(
            [
                "alpha",
                "alpha",
                "beta",
                "beta",
                "alpha",
                "gamma",
                "alpha",
                "alpha",
                "alpha",
                "gamma",
                "beta",
            ],
            long_name="label",
            units="no_unit",
        )
        data = ma.arange(11)
        data.mask = data.data % 2
        mask_coord = AuxCoord(data, long_name="mask")
        unmask_coord = AuxCoord(ma.arange(11), long_name="unmask")
        simple_agg_coord = AuxCoord([1, 1, 2, 2], long_name="simple_agg")
        spanning_coord = AuxCoord(np.arange(44).reshape(4, 11), long_name="spanning")
        spanning_label_coord = AuxCoord(
            np.arange(1, 441, 10).reshape(4, 11).astype(str),
            long_name="span_label",
            units="no_unit",
        )
        data = ma.array(np.arange(44).reshape(4, 11), mask=True)
        spanning_mask_coord = AuxCoord(data, long_name="spanning_mask")
        spanning_unmask_coord = AuxCoord(
            ma.arange(44).reshape(4, 11), long_name="spanning_unmask"
        )

        self.cube.add_aux_coord(simple_agg_coord, 0)
        self.cube.add_aux_coord(val_coord, 1)
        self.cube.add_aux_coord(label_coord, 1)
        self.cube.add_aux_coord(mask_coord, 1)
        self.cube.add_aux_coord(unmask_coord, 1)
        self.cube.add_aux_coord(spanning_coord, (0, 1))
        self.cube.add_aux_coord(spanning_label_coord, (0, 1))
        self.cube.add_aux_coord(spanning_mask_coord, (0, 1))
        self.cube.add_aux_coord(spanning_unmask_coord, (0, 1))

        self.mock_agg = mock.Mock(spec=Aggregator)
        self.mock_agg.cell_method = []
        self.mock_agg.aggregate = mock.Mock(return_value=np.arange(4))
        self.mock_agg.aggregate_shape = mock.Mock(return_value=())
        self.mock_agg.lazy_func = None
        self.mock_agg.post_process = mock.Mock(side_effect=lambda x, y, z: x)

        self.mock_weighted_agg = mock.Mock(spec=WeightedAggregator)
        self.mock_weighted_agg.cell_method = []

        def mock_weighted_aggregate(*_, **kwargs):
            if kwargs.get("returned", False):
                return (np.arange(11), np.ones(11))
            return np.arange(4)

        self.mock_weighted_agg.aggregate = mock.Mock(
            side_effect=mock_weighted_aggregate
        )
        self.mock_weighted_agg.aggregate_shape = mock.Mock(return_value=())
        self.mock_weighted_agg.lazy_func = None
        self.mock_weighted_agg.post_process = mock.Mock(
            side_effect=lambda x, y, z, **kwargs: y
        )

        self.ancillary_variable = AncillaryVariable([0, 1, 2, 3], long_name="foo")
        self.cube.add_ancillary_variable(self.ancillary_variable, 0)
        self.cell_measure = CellMeasure([0, 1, 2, 3], long_name="bar")
        self.cube.add_cell_measure(self.cell_measure, 0)

        self.simple_weights = np.array([1.0, 0.0, 2.0, 2.0])
        self.val_weights = np.ones_like(self.cube.data, dtype=np.float32)

    @pytest.mark.parametrize("dataless", [True, False])
    def test_2d_coord_simple_agg(self, dataless):
        if dataless:
            self.cube.data = None
        # For 2d coords, slices of aggregated coord should be the same as
        # aggregated slices.
        res_cube = self.cube.aggregated_by("simple_agg", self.mock_agg)
        for res_slice, cube_slice in zip(
            res_cube.slices("simple_agg"), self.cube.slices("simple_agg")
        ):
            cube_slice_agg = cube_slice.aggregated_by("simple_agg", self.mock_agg)
            assert res_slice.coord("spanning") == cube_slice_agg.coord("spanning")
            assert res_slice.coord("span_label") == cube_slice_agg.coord("span_label")
            assert res_slice.coord("spanning_mask") == cube_slice_agg.coord(
                "spanning_mask"
            )
            actual = res_slice.coord("spanning_unmask")
            assert actual == cube_slice_agg.coord("spanning_unmask")
            assert not ma.isMaskedArray(actual.points)
            assert not ma.isMaskedArray(actual.bounds)

    @pytest.mark.parametrize("dataless", [True, False])
    def test_agg_by_label(self, dataless):
        if dataless:
            self.cube.data = None
        # Aggregate a cube on a string coordinate label where label
        # and val entries are not in step; the resulting cube has a val
        # coord of bounded cells and a label coord of single string entries.
        res_cube = self.cube.aggregated_by("label", self.mock_agg)
        val_coord = AuxCoord(
            np.array([1.0, 0.5, 1.0]),
            bounds=np.array([[0, 2], [0, 1], [0, 2]]),
            long_name="val",
        )
        label_coord = AuxCoord(
            np.array(["alpha", "beta", "gamma"]),
            long_name="label",
            units="no_unit",
        )
        mask_coord = AuxCoord(
            ma.array([4.0, 6.0, np.nan], mask=[False, False, True]),
            bounds=ma.array(
                [[0.0, 8.0], [2.0, 10.0], [np.nan, np.nan]],
                mask=[[False, False], [False, False], [True, True]],
            ),
            long_name="mask",
        )
        unmask_coord = AuxCoord(
            np.array([4.0, 6.0, 7.0]),
            bounds=np.array([[0.0, 8.0], [2.0, 10.0], [5.0, 9.0]]),
            long_name="unmask",
        )
        assert res_cube.coord("val") == val_coord
        assert res_cube.coord("label") == label_coord
        assert res_cube.coord("mask") == mask_coord
        assert res_cube.coord("unmask") == unmask_coord

    @pytest.mark.parametrize("dataless", [True, False])
    def test_agg_by_label_bounded(self, dataless):
        if dataless:
            self.cube.data = None
        # Aggregate a cube on a string coordinate label where label
        # and val entries are not in step; the resulting cube has a val
        # coord of bounded cells and a label coord of single string entries.
        val_points = self.cube.coord("val").points
        self.cube.coord("val").bounds = np.array([val_points - 0.5, val_points + 0.5]).T
        mask_points = self.cube.coord("mask").points
        self.cube.coord("mask").bounds = ma.array(
            [mask_points - 0.5, mask_points + 0.5]
        ).T
        unmask_points = self.cube.coord("unmask").points
        self.cube.coord("unmask").bounds = ma.array(
            [unmask_points - 0.5, unmask_points + 0.5]
        ).T
        res_cube = self.cube.aggregated_by("label", self.mock_agg)
        val_coord = AuxCoord(
            np.array([1.0, 0.5, 1.0]),
            bounds=np.array([[-0.5, 2.5], [-0.5, 1.5], [-0.5, 2.5]]),
            long_name="val",
        )
        label_coord = AuxCoord(
            np.array(["alpha", "beta", "gamma"]),
            long_name="label",
            units="no_unit",
        )
        mask_coord = AuxCoord(
            ma.array([4.0, 6.0, np.nan], mask=[False, False, True]),
            bounds=ma.array(
                [[-0.5, 8.5], [1.5, 10.5], [np.nan, np.nan]],
                mask=[[False, False], [False, False], [True, True]],
            ),
            long_name="mask",
        )
        unmask_coord = AuxCoord(
            np.array([4.0, 6.0, 7.0]),
            bounds=np.array([[-0.5, 8.5], [1.5, 10.5], [4.5, 9.5]]),
            long_name="unmask",
        )
        assert res_cube.coord("val") == val_coord
        assert res_cube.coord("label") == label_coord
        assert res_cube.coord("mask") == mask_coord
        assert res_cube.coord("unmask") == unmask_coord

    @pytest.mark.parametrize("dataless", [True, False])
    def test_2d_agg_by_label(self, dataless):
        if dataless:
            self.cube.data = None
        res_cube = self.cube.aggregated_by("label", self.mock_agg)
        # For 2d coord, slices of aggregated coord should be the same as
        # aggregated slices.
        for res_slice, cube_slice in zip(
            res_cube.slices("val"), self.cube.slices("val")
        ):
            cube_slice_agg = cube_slice.aggregated_by("label", self.mock_agg)
            assert res_slice.coord("spanning") == cube_slice_agg.coord("spanning")
            assert res_slice.coord("spanning_mask") == cube_slice_agg.coord(
                "spanning_mask"
            )
            actual = res_slice.coord("spanning_unmask")
            assert actual == cube_slice_agg.coord("spanning_unmask")
            assert not ma.isMaskedArray(actual.points)
            assert not ma.isMaskedArray(actual.bounds)

    @pytest.mark.parametrize("dataless", [True, False])
    def test_agg_by_val(self, dataless):
        if dataless:
            self.cube.data = None
        # Aggregate a cube on a numeric coordinate val where label
        # and val entries are not in step; the resulting cube has a label
        # coord with serialised labels from the aggregated cells.
        res_cube = self.cube.aggregated_by("val", self.mock_agg)
        val_coord = AuxCoord(np.array([0, 1, 2]), long_name="val")
        exp0 = "alpha|alpha|beta|alpha|alpha|gamma"
        exp1 = "beta|alpha|beta"
        exp2 = "gamma|alpha"
        label_coord = AuxCoord(
            np.array((exp0, exp1, exp2)), long_name="label", units="no_unit"
        )
        mask_coord = AuxCoord(
            np.array([3.0, 7.0, 8.0]),
            bounds=np.array([[0.0, 6.0], [4.0, 10.0], [8.0, 8.0]]),
            long_name="mask",
        )
        unmask_coord = AuxCoord(
            np.array([4.5, 6.5, 6.5]),
            bounds=np.array([[0.0, 9.0], [3.0, 10.0], [5.0, 8.0]]),  #
            long_name="unmask",
        )
        assert res_cube.coord("val") == val_coord
        assert res_cube.coord("label") == label_coord
        assert res_cube.coord("mask") == mask_coord
        assert res_cube.coord("unmask") == unmask_coord

    @pytest.mark.parametrize("dataless", [True, False])
    def test_2d_agg_by_val(self, dataless):
        if dataless:
            self.cube.data = None
        res_cube = self.cube.aggregated_by("val", self.mock_agg)
        # For 2d coord, slices of aggregated coord should be the same as
        # aggregated slices.
        for res_slice, cube_slice in zip(
            res_cube.slices("val"), self.cube.slices("val")
        ):
            cube_slice_agg = cube_slice.aggregated_by("val", self.mock_agg)
            assert res_slice.coord("spanning") == cube_slice_agg.coord("spanning")
            assert res_slice.coord("spanning_mask") == cube_slice_agg.coord(
                "spanning_mask"
            )
            actual = res_slice.coord("spanning_unmask")
            assert actual == cube_slice_agg.coord("spanning_unmask")
            assert not ma.isMaskedArray(actual.points)
            assert not ma.isMaskedArray(actual.bounds)

    @pytest.mark.parametrize("dataless", [True, False])
    def test_single_string_aggregation(self, dataless):
        if dataless:
            self.cube.data = None
        aux_coords = [
            (AuxCoord(["a", "b", "a"], long_name="foo"), 0),
            (AuxCoord(["a", "a", "a"], long_name="bar"), 0),
        ]
        cube = iris.cube.Cube(
            np.arange(12).reshape(3, 4), aux_coords_and_dims=aux_coords
        )
        result = cube.aggregated_by("foo", MEAN)
        assert result.shape == (2, 4)
        assert result.coord("bar") == AuxCoord(["a|a", "a"], long_name="bar")

    @pytest.mark.parametrize("dataless", [True, False])
    def test_ancillary_variables_and_cell_measures_kept(self, dataless):
        if dataless:
            self.cube.data = None
        cube_agg = self.cube.aggregated_by("val", self.mock_agg)
        assert cube_agg.ancillary_variables() == [self.ancillary_variable]
        assert cube_agg.cell_measures() == [self.cell_measure]

    @pytest.mark.parametrize("dataless", [True, False])
    def test_ancillary_variables_and_cell_measures_removed(self, dataless):
        if dataless:
            self.cube.data = None
        cube_agg = self.cube.aggregated_by("simple_agg", self.mock_agg)
        assert cube_agg.ancillary_variables() == []
        assert cube_agg.cell_measures() == []

    def test_1d_weights(self):
        self.cube.aggregated_by(
            "simple_agg", self.mock_weighted_agg, weights=self.simple_weights
        )
        assert self.mock_weighted_agg.aggregate.call_count == 2

        # A simple mock.assert_called_with does not work due to ValueError: The
        # truth value of an array with more than one element is ambiguous. Use
        # a.any() or a.all()
        call_1 = self.mock_weighted_agg.aggregate.mock_calls[0]
        _shared_utils.assert_array_equal(
            call_1.args[0],
            np.array(
                [
                    [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10],
                    [11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21],
                ]
            ),
        )
        assert call_1.kwargs["axis"] == 0
        _shared_utils.assert_array_almost_equal(
            call_1.kwargs["weights"],
            np.array(
                [
                    [1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0],
                    [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
                ]
            ),
        )

        call_2 = self.mock_weighted_agg.aggregate.mock_calls[1]
        _shared_utils.assert_array_equal(
            call_2.args[0],
            np.array(
                [
                    [22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32],
                    [33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43],
                ]
            ),
        )
        assert call_2.kwargs["axis"] == 0
        _shared_utils.assert_array_almost_equal(
            call_2.kwargs["weights"],
            np.array(
                [
                    [2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0],
                    [2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0, 2.0],
                ]
            ),
        )

    def test_2d_weights(self):
        self.cube.aggregated_by("val", self.mock_weighted_agg, weights=self.val_weights)

        assert self.mock_weighted_agg.aggregate.call_count == 3

        # A simple mock.assert_called_with does not work due to ValueError: The
        # truth value of an array with more than one element is ambiguous. Use
        # a.any() or a.all()
        call_1 = self.mock_weighted_agg.aggregate.mock_calls[0]
        _shared_utils.assert_array_equal(
            call_1.args[0],
            np.array(
                [
                    [0, 1, 2, 6, 7, 9],
                    [11, 12, 13, 17, 18, 20],
                    [22, 23, 24, 28, 29, 31],
                    [33, 34, 35, 39, 40, 42],
                ]
            ),
        )
        assert call_1.kwargs["axis"] == 1
        _shared_utils.assert_array_almost_equal(
            call_1.kwargs["weights"], np.ones((4, 6))
        )

        call_2 = self.mock_weighted_agg.aggregate.mock_calls[1]
        _shared_utils.assert_array_equal(
            call_2.args[0],
            np.array([[3, 4, 10], [14, 15, 21], [25, 26, 32], [36, 37, 43]]),
        )
        assert call_2.kwargs["axis"] == 1
        _shared_utils.assert_array_almost_equal(
            call_2.kwargs["weights"], np.ones((4, 3))
        )

        call_3 = self.mock_weighted_agg.aggregate.mock_calls[2]
        _shared_utils.assert_array_equal(
            call_3.args[0], np.array([[5, 8], [16, 19], [27, 30], [38, 41]])
        )
        assert call_3.kwargs["axis"] == 1
        _shared_utils.assert_array_almost_equal(
            call_3.kwargs["weights"], np.ones((4, 2))
        )

    def test_returned(self):
        output = self.cube.aggregated_by(
            "simple_agg", self.mock_weighted_agg, returned=True
        )

        assert isinstance(output, tuple)
        assert len(output) == 2
        assert output[0].shape == (2, 11)
        assert output[1].shape == (2, 11)

    def test_fail_1d_weights_wrong_len(self):
        wrong_weights = np.array([1.0, 2.0])
        msg = (
            r"1D weights must have the same length as the dimension that is "
            r"aggregated, got 2, expected 11"
        )
        with pytest.raises(ValueError, match=msg):
            self.cube.aggregated_by(
                "val", self.mock_weighted_agg, weights=wrong_weights
            )

    def test_fail_weights_wrong_shape(self):
        wrong_weights = np.ones((42, 1))
        msg = (
            r"Weights must either be 1D or have the same shape as the cube, "
            r"got shape \(42, 1\) for weights, \(4, 11\) for cube"
        )
        with pytest.raises(ValueError, match=msg):
            self.cube.aggregated_by(
                "val", self.mock_weighted_agg, weights=wrong_weights
            )


class Test_aggregated_by__lazy:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.data = np.arange(44).reshape(4, 11)
        self.lazydata = as_lazy_data(self.data)
        self.cube = Cube(self.lazydata)

        val_coord = AuxCoord([0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 1], long_name="val")
        label_coord = AuxCoord(
            [
                "alpha",
                "alpha",
                "beta",
                "beta",
                "alpha",
                "gamma",
                "alpha",
                "alpha",
                "alpha",
                "gamma",
                "beta",
            ],
            long_name="label",
            units="no_unit",
        )
        simple_agg_coord = AuxCoord([1, 1, 2, 2], long_name="simple_agg")

        self.label_mean = np.array(
            [
                [4.0 + 1.0 / 3.0, 5.0, 7.0],
                [15.0 + 1.0 / 3.0, 16.0, 18.0],
                [26.0 + 1.0 / 3.0, 27.0, 29.0],
                [37.0 + 1.0 / 3.0, 38.0, 40.0],
            ]
        )
        self.val_mean = np.array(
            [
                [4.0 + 1.0 / 6.0, 5.0 + 2.0 / 3.0, 6.5],
                [15.0 + 1.0 / 6.0, 16.0 + 2.0 / 3.0, 17.5],
                [26.0 + 1.0 / 6.0, 27.0 + 2.0 / 3.0, 28.5],
                [37.0 + 1.0 / 6.0, 38.0 + 2.0 / 3.0, 39.5],
            ]
        )

        self.cube.add_aux_coord(simple_agg_coord, 0)
        self.cube.add_aux_coord(val_coord, 1)
        self.cube.add_aux_coord(label_coord, 1)

        self.simple_weights = np.array([1.0, 0.0, 2.0, 2.0])
        self.val_weights = 2.0 * np.ones(self.cube.shape, dtype=np.float32)

    def test_agg_by_label__lazy(self):
        # Aggregate a cube on a string coordinate label where label
        # and val entries are not in step; the resulting cube has a val
        # coord of bounded cells and a label coord of single string entries.
        res_cube = self.cube.aggregated_by("label", MEAN)
        val_coord = AuxCoord(
            np.array([1.0, 0.5, 1.0]),
            bounds=np.array([[0, 2], [0, 1], [0, 2]]),
            long_name="val",
        )
        label_coord = AuxCoord(
            np.array(["alpha", "beta", "gamma"]),
            long_name="label",
            units="no_unit",
        )
        assert res_cube.has_lazy_data()
        assert res_cube.coord("val") == val_coord
        assert res_cube.coord("label") == label_coord
        _shared_utils.assert_array_equal(res_cube.data, self.label_mean)
        assert not res_cube.has_lazy_data()

    def test_agg_by_val__lazy(self):
        # Aggregate a cube on a numeric coordinate val where label
        # and val entries are not in step; the resulting cube has a label
        # coord with serialised labels from the aggregated cells.
        res_cube = self.cube.aggregated_by("val", MEAN)
        val_coord = AuxCoord(np.array([0, 1, 2]), long_name="val")
        exp0 = "alpha|alpha|beta|alpha|alpha|gamma"
        exp1 = "beta|alpha|beta"
        exp2 = "gamma|alpha"
        label_coord = AuxCoord(
            np.array((exp0, exp1, exp2)), long_name="label", units="no_unit"
        )
        assert res_cube.has_lazy_data()
        assert res_cube.coord("val") == val_coord
        assert res_cube.coord("label") == label_coord
        _shared_utils.assert_array_equal(res_cube.data, self.val_mean)
        assert not res_cube.has_lazy_data()

    def test_single_string_aggregation__lazy(self):
        aux_coords = [
            (AuxCoord(["a", "b", "a"], long_name="foo"), 0),
            (AuxCoord(["a", "a", "a"], long_name="bar"), 0),
        ]
        cube = iris.cube.Cube(
            as_lazy_data(np.arange(12).reshape(3, 4)),
            aux_coords_and_dims=aux_coords,
        )
        means = np.array([[4.0, 5.0, 6.0, 7.0], [4.0, 5.0, 6.0, 7.0]])
        result = cube.aggregated_by("foo", MEAN)
        assert result.has_lazy_data()
        assert result.shape == (2, 4)
        assert result.coord("bar") == AuxCoord(["a|a", "a"], long_name="bar")
        _shared_utils.assert_array_equal(result.data, means)
        assert not result.has_lazy_data()

    def test_1d_weights__lazy(self):
        assert self.cube.has_lazy_data()

        cube_agg = self.cube.aggregated_by(
            "simple_agg", SUM, weights=self.simple_weights
        )

        assert self.cube.has_lazy_data()
        assert cube_agg.has_lazy_data()
        assert cube_agg.shape == (2, 11)

        row_0 = [0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0]
        row_1 = [
            110.0,
            114.0,
            118.0,
            122.0,
            126.0,
            130.0,
            134.0,
            138.0,
            142.0,
            146.0,
            150.0,
        ]
        _shared_utils.assert_array_almost_equal(cube_agg.data, np.array([row_0, row_1]))

    def test_2d_weights__lazy(self):
        assert self.cube.has_lazy_data()

        cube_agg = self.cube.aggregated_by("val", SUM, weights=self.val_weights)

        assert self.cube.has_lazy_data()
        assert cube_agg.has_lazy_data()

        assert cube_agg.shape == (4, 3)
        _shared_utils.assert_array_almost_equal(
            cube_agg.data,
            np.array(
                [
                    [50.0, 34.0, 26.0],
                    [182.0, 100.0, 70.0],
                    [314.0, 166.0, 114.0],
                    [446.0, 232.0, 158.0],
                ]
            ),
        )

    def test_returned__lazy(self):
        assert self.cube.has_lazy_data()

        output = self.cube.aggregated_by(
            "simple_agg", SUM, weights=self.simple_weights, returned=True
        )

        assert self.cube.has_lazy_data()

        assert isinstance(output, tuple)
        assert len(output) == 2

        cube = output[0]
        assert isinstance(cube, Cube)
        assert cube.has_lazy_data()
        assert cube.shape == (2, 11)
        row_0 = [0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0]
        row_1 = [
            110.0,
            114.0,
            118.0,
            122.0,
            126.0,
            130.0,
            134.0,
            138.0,
            142.0,
            146.0,
            150.0,
        ]
        _shared_utils.assert_array_almost_equal(cube.data, np.array([row_0, row_1]))

        weights = output[1]
        assert weights.shape == (2, 11)
        _shared_utils.assert_array_almost_equal(
            weights,
            np.array(
                [
                    [1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0],
                    [4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0, 4.0],
                ]
            ),
        )


class Test_aggregated_by__climatology:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.data = np.arange(100).reshape(20, 5)
        self.aggregator = iris.analysis.MEAN

    def get_result(
        self,
        transpose: bool = False,
        second_categorised: bool = False,
        bounds: bool = False,
        partially_aligned: bool = False,
        partially_aligned_timelike: bool = False,
        invalid_units: bool = False,
        already_climatological: bool = False,
        climatological_op: bool = True,
    ) -> Cube:
        cube_data = self.data
        if transpose:
            cube_data = cube_data.T
            axes = [1, 0]
        else:
            axes = [0, 1]
        if not invalid_units:
            units = Unit("days since 1970-01-01")
        else:
            units = Unit("m")
        if partially_aligned_timelike:
            pa_units = Unit("days since 1970-01-01")
        else:
            pa_units = Unit("m")

        # DimCoords
        aligned_coord = DimCoord(
            np.arange(20),
            long_name="aligned",
            units=units,
        )
        orthogonal_coord = DimCoord(np.arange(5), long_name="orth")

        if bounds:
            aligned_coord.guess_bounds()

        aligned_coord.climatological = already_climatological

        dim_coords_and_dims = zip([aligned_coord, orthogonal_coord], axes)

        # AuxCoords
        categorised_coord1 = AuxCoord(
            np.tile([0, 1], 10), long_name="cat1", units=Unit("month")
        )

        if second_categorised:
            categorised_coord2 = AuxCoord(np.tile([0, 1, 2, 3, 4], 4), long_name="cat2")
            categorised_coords: AuxCoord | list[AuxCoord] = [
                categorised_coord1,
                categorised_coord2,
            ]
        else:
            categorised_coords = categorised_coord1

        aux_coords_and_dims: list[tuple[AuxCoord, int | tuple[int, ...]]] = [
            (categorised_coord1, axes[0]),
        ]

        if second_categorised:
            aux_coords_and_dims.append((categorised_coord2, axes[0]))

        if partially_aligned:
            partially_aligned_coord = AuxCoord(
                cube_data + 1,
                long_name="part_aligned",
                units=pa_units,
            )
            aux_coords_and_dims.append((partially_aligned_coord, (0, 1)))

        # Build cube
        in_cube = iris.cube.Cube(
            cube_data,
            long_name="wibble",
            dim_coords_and_dims=dim_coords_and_dims,
            aux_coords_and_dims=aux_coords_and_dims,
        )

        out_cube = in_cube.aggregated_by(
            categorised_coords,
            self.aggregator,
            climatological=climatological_op,
        )

        return out_cube

    def test_basic(self):
        """Check the least complicated version works (set climatological, set
        points correctly).
        """
        result = self.get_result()

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(2))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[0, 18], [1, 19]])
        )
        assert aligned_coord.climatological
        assert aligned_coord in result.dim_coords

        categorised_coord = result.coord("cat1")
        _shared_utils.assert_array_equal(categorised_coord.points, np.arange(2))
        assert categorised_coord.bounds is None
        assert not categorised_coord.climatological

    def test_2d_other_coord(self):
        """Check that we can handle aggregation applying to a 2d AuxCoord that
        covers the aggregation dimension and another one.
        """
        result = self.get_result(partially_aligned=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(2))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[0, 18], [1, 19]])
        )
        assert aligned_coord.climatological

        part_aligned_coord = result.coord("part_aligned")
        _shared_utils.assert_array_equal(
            part_aligned_coord.points, np.arange(46, 56).reshape(2, 5)
        )
        _shared_utils.assert_array_equal(
            part_aligned_coord.bounds,
            np.array([np.arange(1, 11), np.arange(91, 101)]).T.reshape(2, 5, 2),
        )
        assert not part_aligned_coord.climatological

    def test_2d_timelike_other_coord(self):
        """Check that we can handle aggregation applying to a 2d AuxCoord that
        covers the aggregation dimension and another one.
        """
        result = self.get_result(
            partially_aligned=True, partially_aligned_timelike=True
        )

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(2))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[0, 18], [1, 19]])
        )
        assert aligned_coord.climatological

        part_aligned_coord = result.coord("part_aligned")
        _shared_utils.assert_array_equal(
            part_aligned_coord.points, np.arange(1, 11).reshape(2, 5)
        )
        _shared_utils.assert_array_equal(
            part_aligned_coord.bounds,
            np.array([np.arange(1, 11), np.arange(91, 101)]).T.reshape(2, 5, 2),
        )
        assert part_aligned_coord.climatological

    def test_transposed(self):
        """Check that we can handle the axis of aggregation being a different one."""
        result = self.get_result(transpose=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(2))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[0, 18], [1, 19]])
        )
        assert aligned_coord.climatological

        categorised_coord = result.coord("cat1")
        _shared_utils.assert_array_equal(categorised_coord.points, np.arange(2))
        assert categorised_coord.bounds is None
        assert not categorised_coord.climatological

    def test_bounded(self):
        """Check that we handle bounds correctly."""
        result = self.get_result(bounds=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, [-0.5, 0.5])
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[-0.5, 18.5], [0.5, 19.5]])
        )
        assert aligned_coord.climatological

    def test_multiple_agg_coords(self):
        """Check that we can aggregate on multiple coords on the same axis."""
        result = self.get_result(second_categorised=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(10))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds,
            np.array([np.arange(10), np.arange(10, 20)]).T,
        )
        assert aligned_coord.climatological

        categorised_coord1 = result.coord("cat1")
        _shared_utils.assert_array_equal(
            categorised_coord1.points, np.tile(np.arange(2), 5)
        )
        assert categorised_coord1.bounds is None
        assert not categorised_coord1.climatological

        categorised_coord2 = result.coord("cat2")
        _shared_utils.assert_array_equal(
            categorised_coord2.points, np.tile(np.arange(5), 2)
        )
        assert categorised_coord2.bounds is None
        assert not categorised_coord2.climatological

    def test_non_climatological_units(self):
        """Check that the failure to set the climatological flag on an incompatible
        unit is handled quietly.
        """
        result = self.get_result(invalid_units=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(9, 11))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[0, 18], [1, 19]])
        )
        assert not aligned_coord.climatological

    def test_clim_in_clim_op(self):
        """Check the least complicated version works (set climatological, set
        points correctly). For the input coordinate to be climatological, it
        must have bounds.
        """
        result = self.get_result(bounds=True, already_climatological=True)

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, [-0.5, 0.5])
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[-0.5, 18.5], [0.5, 19.5]])
        )
        assert aligned_coord.climatological

        categorised_coord = result.coord("cat1")
        _shared_utils.assert_array_equal(categorised_coord.points, np.arange(2))
        assert categorised_coord.bounds is None
        assert not categorised_coord.climatological

    def test_clim_in_no_clim_op(self):
        """Check the least complicated version works (set climatological, set
        points correctly). For the input coordinate to be climatological, it
        must have bounds.
        """
        result = self.get_result(
            bounds=True, already_climatological=True, climatological_op=False
        )

        aligned_coord = result.coord("aligned")
        _shared_utils.assert_array_equal(aligned_coord.points, np.arange(9, 11))
        _shared_utils.assert_array_equal(
            aligned_coord.bounds, np.array([[-0.5, 18.5], [0.5, 19.5]])
        )
        assert aligned_coord.climatological

        categorised_coord = result.coord("cat1")
        _shared_utils.assert_array_equal(categorised_coord.points, np.arange(2))
        assert categorised_coord.bounds is None
        assert not categorised_coord.climatological


class Test_aggregated_by__derived:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.cube = realistic_4d()[:, :10, :6, :8]
        self.time_cat_coord = AuxCoord([0, 0, 1, 1, 2, 2], long_name="time_cat")
        self.cube.add_aux_coord(self.time_cat_coord, 0)
        height_data = np.zeros(self.cube.shape[1])
        height_data[5:] = 1
        self.height_cat_coord = AuxCoord(height_data, long_name="height_cat")
        self.cube.add_aux_coord(self.height_cat_coord, 1)
        self.aggregator = iris.analysis.MEAN

    def test_grouped_dim(self):
        """Check that derived coordinates are maintained when the coordinates they
        derive from are aggregated.
        """
        result = self.cube.aggregated_by(
            self.height_cat_coord,
            self.aggregator,
        )
        assert len(result.aux_factories) == 1
        altitude = result.coord("altitude")
        assert altitude.shape == (2, 6, 8)

        # Check the bounds are derived as expected.
        orig_alt_bounds = self.cube.coord("altitude").bounds
        bounds_0 = orig_alt_bounds[0::5, :, :, 0]
        bounds_1 = orig_alt_bounds[4::5, :, :, 1]
        expected_bounds = np.stack([bounds_0, bounds_1], axis=-1)
        assert np.array_equal(expected_bounds, result.coord("altitude").bounds)

    def test_ungrouped_dim(self):
        """Check that derived coordinates are preserved when aggregating along a
        different axis.
        """
        result = self.cube.aggregated_by(
            self.time_cat_coord,
            self.aggregator,
        )
        assert len(result.aux_factories) == 1
        altitude = result.coord("altitude")
        assert altitude == self.cube.coord("altitude")
