# Copyright Iris contributors
#
# This file is part of Iris and is released under the BSD license.
# See LICENSE in the root of the repository for full licensing details.
"""Unit tests for the :class:`iris.coords.Coord` class."""

import collections
from datetime import datetime

import cf_units
import dask.array as da
import numpy as np
import numpy.ma as ma
import pytest

import iris
from iris.coords import AuxCoord, Coord, DimCoord
from iris.cube import Cube
from iris.exceptions import UnitConversionError
from iris.tests import _shared_utils
from iris.tests.unit.coords import CoordTestMixin
from iris.warnings import IrisVagueMetadataWarning

Pair = collections.namedtuple("Pair", "points bounds")


class Test_nearest_neighbour_index__ascending:
    @pytest.fixture(autouse=True)
    def _setup(self):
        points = [0.0, 90.0, 180.0, 270.0]
        self.coord = DimCoord(points, circular=False, units="degrees")

    def _test_nearest_neighbour_index(self, target, bounds=None, circular=False):
        _bounds = [[-20, 10], [10, 100], [100, 260], [260, 340]]
        ext_pnts = [-70, -10, 110, 275, 370]
        if bounds is True:
            self.coord.bounds = _bounds
        else:
            self.coord.bounds = bounds
        self.coord.circular = circular
        results = [self.coord.nearest_neighbour_index(ind) for ind in ext_pnts]
        assert results == target

    def test_nobounds(self):
        target = [0, 0, 1, 3, 3]
        self._test_nearest_neighbour_index(target)

    def test_nobounds_circular(self):
        target = [3, 0, 1, 3, 0]
        self._test_nearest_neighbour_index(target, circular=True)

    def test_bounded(self):
        target = [0, 0, 2, 3, 3]
        self._test_nearest_neighbour_index(target, bounds=True)

    def test_bounded_circular(self):
        target = [3, 0, 2, 3, 0]
        self._test_nearest_neighbour_index(target, bounds=True, circular=True)

    def test_bounded_overlapping(self):
        _bounds = [[-20, 50], [10, 150], [100, 300], [260, 340]]
        target = [0, 0, 1, 2, 3]
        self._test_nearest_neighbour_index(target, bounds=_bounds)

    def test_bounded_disjointed(self):
        _bounds = [[-20, 10], [80, 170], [180, 190], [240, 340]]
        target = [0, 0, 1, 3, 3]
        self._test_nearest_neighbour_index(target, bounds=_bounds)

    def test_scalar(self):
        self.coord = DimCoord([0], circular=False, units="degrees")
        target = [0, 0, 0, 0, 0]
        self._test_nearest_neighbour_index(target)

    def test_bounded_float_point(self):
        coord = DimCoord(1, bounds=[0, 2])
        result = coord.nearest_neighbour_index(2.5)
        assert result == 0


class Test_nearest_neighbour_index__descending:
    @pytest.fixture(autouse=True)
    def _setup(self):
        points = [270.0, 180.0, 90.0, 0.0]
        self.coord = DimCoord(points, circular=False, units="degrees")

    def _test_nearest_neighbour_index(self, target, bounds=False, circular=False):
        _bounds = [[340, 260], [260, 100], [100, 10], [10, -20]]
        ext_pnts = [-70, -10, 110, 275, 370]
        if bounds:
            self.coord.bounds = _bounds
        self.coord.circular = circular
        results = [self.coord.nearest_neighbour_index(ind) for ind in ext_pnts]
        assert results == target

    def test_nobounds(self):
        target = [3, 3, 2, 0, 0]
        self._test_nearest_neighbour_index(target)

    def test_nobounds_circular(self):
        target = [0, 3, 2, 0, 3]
        self._test_nearest_neighbour_index(target, circular=True)

    def test_bounded(self):
        target = [3, 3, 1, 0, 0]
        self._test_nearest_neighbour_index(target, bounds=True)

    def test_bounded_circular(self):
        target = [0, 3, 1, 0, 3]
        self._test_nearest_neighbour_index(target, bounds=True, circular=True)


class Test_guess_bounds:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.coord = DimCoord(
            np.array([-160, -120, 0, 30, 150, 170]),
            units="degree",
            standard_name="longitude",
            circular=True,
        )

    def test_non_circular(self):
        self.coord.circular = False
        self.coord.guess_bounds()
        target = np.array(
            [
                [-180.0, -140.0],
                [-140.0, -60.0],
                [-60.0, 15.0],
                [15.0, 90.0],
                [90.0, 160.0],
                [160.0, 180.0],
            ]
        )
        _shared_utils.assert_array_equal(target, self.coord.bounds)

    def test_circular_increasing(self):
        self.coord.guess_bounds()
        target = np.array(
            [
                [-175.0, -140.0],
                [-140.0, -60.0],
                [-60.0, 15.0],
                [15.0, 90.0],
                [90.0, 160.0],
                [160.0, 185.0],
            ]
        )
        _shared_utils.assert_array_equal(target, self.coord.bounds)

    def test_circular_decreasing(self):
        self.coord.points = self.coord.points[::-1]
        self.coord.guess_bounds()
        target = np.array(
            [
                [185.0, 160.0],
                [160.0, 90.0],
                [90.0, 15.0],
                [15.0, -60.0],
                [-60.0, -140.0],
                [-140.0, -175.0],
            ]
        )
        _shared_utils.assert_array_equal(target, self.coord.bounds)

    def test_circular_increasing_alt_range(self):
        self.coord.points = np.array([10, 30, 90, 150, 210, 220])
        self.coord.guess_bounds()
        target = np.array(
            [
                [-65.0, 20.0],
                [20.0, 60.0],
                [60.0, 120.0],
                [120.0, 180.0],
                [180.0, 215.0],
                [215.0, 295.0],
            ]
        )
        _shared_utils.assert_array_equal(target, self.coord.bounds)

    def test_circular_decreasing_alt_range(self):
        self.coord.points = np.array([10, 30, 90, 150, 210, 220])[::-1]
        self.coord.guess_bounds()
        target = np.array(
            [
                [295, 215],
                [215, 180],
                [180, 120],
                [120, 60],
                [60, 20],
                [20, -65],
            ]
        )
        _shared_utils.assert_array_equal(target, self.coord.bounds)

    def test_circular_n216_float32(self):
        # Values represent longitude taken from an N216 pp-file.
        coord = DimCoord.from_regular(
            -0.41666666, 0.8333333, 432, units="degrees", circular=True
        )
        coord.guess_bounds()
        # The last bound should match up to the first.
        assert coord.bounds[-1, -1] - coord.bounds[0, 0] == 360


class Test_guess_bounds__default_enabled_latitude_clipping:
    def test_all_inside(self):
        lat = DimCoord([-10, 0, 20], units="degree", standard_name="latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(lat.bounds, [[-15, -5], [-5, 10], [10, 30]])

    def test_points_inside_bounds_outside(self):
        lat = DimCoord([-80, 0, 70], units="degree", standard_name="latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(lat.bounds, [[-90, -40], [-40, 35], [35, 90]])

    def test_points_inside_bounds_outside_grid_latitude(self):
        lat = DimCoord([-80, 0, 70], units="degree", standard_name="grid_latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(lat.bounds, [[-90, -40], [-40, 35], [35, 90]])

    def test_points_to_edges_bounds_outside(self):
        lat = DimCoord([-90, 0, 90], units="degree", standard_name="latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(lat.bounds, [[-90, -45], [-45, 45], [45, 90]])

    def test_points_outside(self):
        lat = DimCoord([-100, 0, 120], units="degree", standard_name="latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(
            lat.bounds, [[-150, -50], [-50, 60], [60, 180]]
        )

    def test_points_inside_bounds_outside_wrong_unit(self):
        lat = DimCoord([-80, 0, 70], units="feet", standard_name="latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(
            lat.bounds, [[-120, -40], [-40, 35], [35, 105]]
        )

    def test_points_inside_bounds_outside_wrong_name(self):
        lat = DimCoord([-80, 0, 70], units="degree", standard_name="longitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(
            lat.bounds, [[-120, -40], [-40, 35], [35, 105]]
        )

    def test_points_inside_bounds_outside_wrong_name_2(self):
        lat = DimCoord([-80, 0, 70], units="degree", long_name="other_latitude")
        lat.guess_bounds()
        _shared_utils.assert_array_equal(
            lat.bounds, [[-120, -40], [-40, 35], [35, 105]]
        )


def test_guess_bounds_monthly_and_yearly():
    units = cf_units.Unit("days since epoch", calendar="gregorian")
    points = units.date2num(
        [
            datetime(1990, 1, 1),
            datetime(1990, 2, 1),
            datetime(1990, 3, 1),
        ]
    )
    coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
    with pytest.raises(
        ValueError,
        match="Cannot guess monthly and yearly bounds simultaneously.",
    ):
        coord.guess_bounds(monthly=True, yearly=True)


class Test_Guess_Bounds_Monthly:
    def test_monthly_multiple_points_in_month(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        points = units.date2num(
            [
                datetime(1990, 1, 3),
                datetime(1990, 1, 28),
                datetime(1990, 2, 13),
            ]
        )
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        with pytest.raises(
            ValueError,
            match="Cannot guess monthly bounds for a coordinate with multiple points "
            "in a month.",
        ):
            coord.guess_bounds(monthly=True)

    def test_monthly_non_contiguous(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = units.date2num(
            [
                [datetime(1990, 1, 1), datetime(1990, 2, 1)],
                [datetime(1990, 2, 1), datetime(1990, 3, 1)],
                [datetime(1990, 5, 1), datetime(1990, 6, 1)],
            ]
        )
        points = expected.mean(axis=1)
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        with pytest.raises(
            ValueError, match="Cannot guess bounds for a non-contiguous coordinate."
        ):
            coord.guess_bounds(monthly=True)

    def test_monthly_end_of_month(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = units.date2num(
            [
                [datetime(1990, 1, 1), datetime(1990, 2, 1)],
                [datetime(1990, 2, 1), datetime(1990, 3, 1)],
                [datetime(1990, 3, 1), datetime(1990, 4, 1)],
            ]
        )
        points = units.date2num(
            [
                datetime(1990, 1, 31),
                datetime(1990, 2, 28),
                datetime(1990, 3, 31),
            ]
        )
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        coord.guess_bounds(monthly=True)
        dates = units.num2date(coord.bounds)
        expected_dates = units.num2date(expected)
        _shared_utils.assert_array_equal(dates, expected_dates)

    def test_monthly_multiple_years(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = [
            [datetime(1990, 10, 1), datetime(1990, 11, 1)],
            [datetime(1990, 11, 1), datetime(1990, 12, 1)],
            [datetime(1990, 12, 1), datetime(1991, 1, 1)],
        ]
        expected_points = units.date2num(expected)
        points = expected_points.mean(axis=1)
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        coord.guess_bounds(monthly=True)
        dates = units.num2date(coord.bounds)
        _shared_utils.assert_array_equal(dates, expected)

    def test_monthly_single_point(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = [
            [datetime(1990, 1, 1), datetime(1990, 2, 1)],
        ]
        expected_points = units.date2num(expected)
        points = expected_points.mean(axis=1)
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        coord.guess_bounds(monthly=True)
        dates = units.num2date(coord.bounds)
        _shared_utils.assert_array_equal(dates, expected)


class Test_Guess_Bounds_Yearly:
    def test_yearly_multiple_points_in_year(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        points = units.date2num(
            [
                datetime(1990, 1, 1),
                datetime(1990, 2, 1),
                datetime(1991, 1, 1),
            ]
        )
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        with pytest.raises(
            ValueError,
            match="Cannot guess yearly bounds for a coordinate with multiple points "
            "in a year.",
        ):
            coord.guess_bounds(yearly=True)

    def test_yearly_non_contiguous(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = units.date2num(
            [
                [datetime(1990, 1, 1), datetime(1990, 1, 1)],
                [datetime(1991, 1, 1), datetime(1991, 1, 1)],
                [datetime(1994, 1, 1), datetime(1994, 1, 1)],
            ]
        )
        points = expected.mean(axis=1)
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        with pytest.raises(
            ValueError, match="Cannot guess bounds for a non-contiguous coordinate."
        ):
            coord.guess_bounds(yearly=True)

    def test_yearly_end_of_year(self):
        units = cf_units.Unit("days since epoch", calendar="gregorian")
        expected = units.date2num(
            [
                [datetime(1990, 1, 1), datetime(1991, 1, 1)],
                [datetime(1991, 1, 1), datetime(1992, 1, 1)],
                [datetime(1992, 1, 1), datetime(1993, 1, 1)],
            ]
        )
        points = units.date2num(
            [
                datetime(1990, 12, 31),
                datetime(1991, 12, 31),
                datetime(1992, 12, 31),
            ]
        )
        coord = iris.coords.AuxCoord(points=points, units=units, standard_name="time")
        coord.guess_bounds(yearly=True)
        dates = units.num2date(coord.bounds)
        expected_dates = units.num2date(expected)
        _shared_utils.assert_array_equal(dates, expected_dates)


class Test_cell:
    def _mock_coord(self, mocker):
        coord = mocker.Mock(
            spec=Coord,
            ndim=1,
        )
        coord.core_points = lambda: np.array([mocker.sentinel.time])
        coord.core_bounds = lambda: np.array(
            [[mocker.sentinel.lower, mocker.sentinel.upper]]
        )

        return coord

    def test_time_as_object(self, mocker):
        # Ensure Coord.cell() converts the point/bound values to
        # "datetime" objects.
        coord = self._mock_coord(mocker)
        coord.units.num2date = mocker.Mock(
            side_effect=[
                mocker.sentinel.datetime,
                (mocker.sentinel.datetime_lower, mocker.sentinel.datetime_upper),
            ]
        )
        cell = Coord.cell(coord, 0)
        assert cell.point is mocker.sentinel.datetime
        assert cell.bound == (
            mocker.sentinel.datetime_lower,
            mocker.sentinel.datetime_upper,
        )
        assert coord.units.num2date.call_args_list == [
            mocker.call((mocker.sentinel.time,)),
            mocker.call((mocker.sentinel.lower, mocker.sentinel.upper)),
        ]


class Test_collapsed(CoordTestMixin):
    def test_serialize(self):
        # Collapse a string AuxCoord, causing it to be serialised.
        string = Pair(
            np.array(["two", "four", "six", "eight"]),
            np.array(
                [
                    ["one", "three"],
                    ["three", "five"],
                    ["five", "seven"],
                    ["seven", "nine"],
                ]
            ),
        )
        string_nobounds = Pair(np.array(["ecks", "why", "zed"]), None)
        string_multi = Pair(
            np.array(["three", "six", "nine"]),
            np.array(
                [
                    ["one", "two", "four", "five"],
                    ["four", "five", "seven", "eight"],
                    ["seven", "eight", "ten", "eleven"],
                ]
            ),
        )

        def _serialize(data):
            return "|".join(str(item) for item in data.flatten())

        for units in ["unknown", "no_unit"]:
            for points, bounds in [string, string_nobounds, string_multi]:
                coord = AuxCoord(points=points, bounds=bounds, units=units)
                collapsed_coord = coord.collapsed()
                _shared_utils.assert_array_equal(
                    collapsed_coord.points, _serialize(points)
                )
                if bounds is not None:
                    for index in np.ndindex(bounds.shape[1:]):
                        index_slice = (slice(None),) + tuple(index)
                        _shared_utils.assert_array_equal(
                            collapsed_coord.bounds[index_slice],
                            _serialize(bounds[index_slice]),
                        )

    def test_dim_1d(self):
        # Numeric coords should not be serialised.
        coord = DimCoord(
            points=np.array([2, 4, 6, 8]),
            bounds=np.array([[1, 3], [3, 5], [5, 7], [7, 9]]),
        )
        for units in ["unknown", "no_unit", 1, "K"]:
            coord.units = units
            with _shared_utils.assert_no_warnings_regexp():
                collapsed_coord = coord.collapsed()
            _shared_utils.assert_array_equal(
                collapsed_coord.points, np.mean(coord.points)
            )
            _shared_utils.assert_array_equal(
                collapsed_coord.bounds,
                [[coord.bounds.min(), coord.bounds.max()]],
            )

    def test_lazy_points(self):
        # Lazy points should stay lazy after collapse.
        coord = AuxCoord(points=da.from_array(np.arange(5), chunks=5))
        collapsed_coord = coord.collapsed()
        assert collapsed_coord.has_lazy_bounds()
        assert collapsed_coord.has_lazy_points()

    def test_numeric_masked_all(self):
        coord = AuxCoord(
            points=ma.array(
                [[1, 2, 4, 5], [4, 5, 7, 8], [7, 8, 10, 11]],
                mask=True,
            ),
        )

        collapsed_coord = coord.collapsed()
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            ma.array([-1], mask=True), bounds=ma.array([[-1, -1]], mask=[[True, True]])
        )
        assert collapsed_coord == expected

        collapsed_coord = coord.collapsed(dims_to_collapse=1)
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            points=ma.array([-1, -1, -1], mask=[True, True, True]),
            bounds=ma.array(
                [[-1, -1], [-1, -1], [-1, -1]],
                mask=[[True, True], [True, True], [True, True]],
            ),
        )
        assert collapsed_coord == expected

        collapsed_coord = coord.collapsed(dims_to_collapse=0)
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            points=ma.array([-1, -1, -1, -1], mask=[True, True, True, True]),
            bounds=ma.array(
                [[-1, -1], [-1, -1], [-1, -1], [-1, -1]],
                mask=[[True, True], [True, True], [True, True], [True, True]],
            ),
        )
        assert collapsed_coord == expected

    def test_numeric_nd(self):
        coord = AuxCoord(points=np.array([[1, 2, 4, 5], [4, 5, 7, 8], [7, 8, 10, 11]]))

        collapsed_coord = coord.collapsed()
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([6]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, np.array([[1, 11]]))

        # Test partially collapsing one dimension...
        collapsed_coord = coord.collapsed(1)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([3.0, 6.0, 9.0])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[1, 5], [4, 8], [7, 11]])
        )

        # ... and the other
        collapsed_coord = coord.collapsed(0)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([4, 5, 7, 8]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds,
            np.array([[1, 7], [2, 8], [4, 10], [5, 11]]),
        )

    def test_numeric_masked(self):
        coord = AuxCoord(
            points=ma.array(
                [[1, 2, 4, 5], [4, 5, 7, 8], [7, 8, 10, 11]],
                mask=[
                    [True, False, False, True],
                    [True, True, True, True],
                    [False, True, True, False],
                ],
            ),
        )

        collapsed_coord = coord.collapsed()
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([6]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, np.array([[2, 11]]))

        collapsed_coord = coord.collapsed(dims_to_collapse=1)
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            points=ma.array([3, -1, 9], mask=[False, True, False]),
            bounds=ma.array(
                [[2, 4], [-1, -1], [7, 11]],
                mask=[[False, False], [True, True], [False, False]],
            ),
        )
        assert collapsed_coord == expected

        collapsed_coord = coord.collapsed(dims_to_collapse=0)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([7, 2, 4, 11])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[7, 7], [2, 2], [4, 4], [11, 11]])
        )

    def test_numeric_nd_bounds_all(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real)

        collapsed_coord = coord.collapsed()
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([55]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, np.array([[-2, 112]]))

    def test_numeric_no_masked_bounds_all(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.no_masked_pts_real, bounds=self.no_masked_bds_real)

        collapsed_coord = coord.collapsed()
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([55]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, np.array([[-2, 112]]))

    def test_numeric_masked_bounds_all(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.masked_pts_real, bounds=self.masked_bds_real)

        collapsed_coord = coord.collapsed()
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([75]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, np.array([[38, 112]]))

    def test_numeric_nd_bounds_second(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real)
        collapsed_coord = coord.collapsed(1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_numeric_no_masked_bounds_second(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.no_masked_pts_real, bounds=self.no_masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_numeric_masked_bounds_second(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.masked_pts_real, bounds=self.masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=1)
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            ma.array([-1, 55, 95], mask=[True, False, False]),
            bounds=ma.array(
                [[-1, -1], [38, 72], [78, 112]],
                mask=[[True, True], [False, False], [False, False]],
            ),
        )
        assert collapsed_coord == expected

    def test_numeric_nd_bounds_first(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real)
        # ... and the other..
        collapsed_coord = coord.collapsed(0)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([40, 50, 60, 70])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds,
            np.array([[-2, 82], [8, 92], [18, 102], [28, 112]]),
        )

    def test_numeric_no_masked_bounds_first(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.no_masked_pts_real, bounds=self.no_masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=0)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([40, 50, 60, 70])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 82], [8, 92], [18, 102], [28, 112]])
        )

    def test_numeric_masked_bounds_first(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.masked_pts_real, bounds=self.masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=0)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([60, 70, 80, 90])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[38, 82], [48, 92], [58, 102], [68, 112]])
        )

    def test_numeric_nd_bounds_last(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real)
        # ... and again with -ve dimension specification.
        collapsed_coord = coord.collapsed(-1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_numeric_no_masked_bounds_last(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.no_masked_pts_real, bounds=self.no_masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=-1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_numeric_masked_bounds_last(self):
        self.setup_test_arrays((3, 4), masked=True)
        coord = AuxCoord(self.masked_pts_real, bounds=self.masked_bds_real)
        collapsed_coord = coord.collapsed(dims_to_collapse=-1)
        self.assert_is_masked_array(collapsed_coord.points)
        self.assert_is_masked_array(collapsed_coord.bounds)
        expected = AuxCoord(
            ma.array([-1, 55, 95], mask=[True, False, False]),
            bounds=ma.array(
                [[-1, -1], [38, 72], [78, 112]],
                mask=[[True, True], [False, False], [False, False]],
            ),
        )
        assert collapsed_coord == expected

    def test_lazy_nd_bounds_all(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_lazy)

        collapsed_coord = coord.collapsed()

        # Note that the new points get recalculated from the lazy bounds
        #  and so end up as lazy
        assert collapsed_coord.has_lazy_points()
        assert collapsed_coord.has_lazy_bounds()

        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([55]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, da.array([[-2, 112]]))

    def test_lazy_nd_bounds_second(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_lazy)

        collapsed_coord = coord.collapsed(1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_lazy_nd_bounds_first(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_lazy)

        collapsed_coord = coord.collapsed(0)
        _shared_utils.assert_array_equal(
            collapsed_coord.points, np.array([40, 50, 60, 70])
        )
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds,
            np.array([[-2, 82], [8, 92], [18, 102], [28, 112]]),
        )

    def test_lazy_nd_bounds_last(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_lazy)

        collapsed_coord = coord.collapsed(-1)
        _shared_utils.assert_array_equal(collapsed_coord.points, np.array([15, 55, 95]))
        _shared_utils.assert_array_equal(
            collapsed_coord.bounds, np.array([[-2, 32], [38, 72], [78, 112]])
        )

    def test_lazy_nd_points_and_bounds(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_lazy, bounds=self.bds_lazy)

        collapsed_coord = coord.collapsed()

        assert collapsed_coord.has_lazy_points()
        assert collapsed_coord.has_lazy_bounds()

        _shared_utils.assert_array_equal(collapsed_coord.points, da.array([55]))
        _shared_utils.assert_array_equal(collapsed_coord.bounds, da.array([[-2, 112]]))

    def test_numeric_nd_multidim_bounds_warning(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real, long_name="y")

        msg = (
            "Collapsing a multi-dimensional coordinate. "
            "Metadata may not be fully descriptive for 'y'."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            coord.collapsed()

    def test_lazy_nd_multidim_bounds_warning(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_lazy, bounds=self.bds_lazy, long_name="y")

        msg = (
            "Collapsing a multi-dimensional coordinate. "
            "Metadata may not be fully descriptive for 'y'."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            coord.collapsed()

    def test_numeric_nd_noncontiguous_bounds_warning(self):
        self.setup_test_arrays((3))
        coord = AuxCoord(self.pts_real, bounds=self.bds_real, long_name="y")

        msg = (
            "Collapsing a non-contiguous coordinate. "
            "Metadata may not be fully descriptive for 'y'."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            coord.collapsed()

    def test_lazy_nd_noncontiguous_bounds_warning(self):
        self.setup_test_arrays((3))
        coord = AuxCoord(self.pts_lazy, bounds=self.bds_lazy, long_name="y")

        msg = (
            "Collapsing a non-contiguous coordinate. "
            "Metadata may not be fully descriptive for 'y'."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            coord.collapsed()

    def test_numeric_3_bounds(self):
        points = np.array([2.0, 6.0, 4.0])
        bounds = np.array([[1.0, 0.0, 3.0], [5.0, 4.0, 7.0], [3.0, 2.0, 5.0]])

        coord = AuxCoord(points, bounds=bounds, long_name="x")

        msg = (
            r"Cannot check if coordinate is contiguous: Invalid operation for "
            r"'x', with 3 bound\(s\). Contiguous bounds are only defined for "
            r"1D coordinates with 2 bounds. Metadata may not be fully "
            r"descriptive for 'x'. Ignoring bounds."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            collapsed_coord = coord.collapsed()

        assert not collapsed_coord.has_lazy_points()
        assert not collapsed_coord.has_lazy_bounds()

        _shared_utils.assert_array_almost_equal(collapsed_coord.points, np.array([4.0]))
        _shared_utils.assert_array_almost_equal(
            collapsed_coord.bounds, np.array([[2.0, 6.0]])
        )

    def test_lazy_3_bounds(self):
        points = da.arange(3) * 2.0
        bounds = da.arange(3 * 3).reshape(3, 3)

        coord = AuxCoord(points, bounds=bounds, long_name="x")

        msg = (
            r"Cannot check if coordinate is contiguous: Invalid operation for "
            r"'x', with 3 bound\(s\). Contiguous bounds are only defined for "
            r"1D coordinates with 2 bounds. Metadata may not be fully "
            r"descriptive for 'x'. Ignoring bounds."
        )
        with pytest.warns(IrisVagueMetadataWarning, match=msg):
            collapsed_coord = coord.collapsed()

        assert collapsed_coord.has_lazy_points()
        assert collapsed_coord.has_lazy_bounds()

        _shared_utils.assert_array_almost_equal(collapsed_coord.points, da.array([2.0]))
        _shared_utils.assert_array_almost_equal(
            collapsed_coord.bounds, da.array([[0.0, 4.0]])
        )

    def test_string_masked(self):
        points = ma.array(["foo", "bar", "bing"], mask=[0, 1, 0], dtype=str)
        coord = AuxCoord(points)

        collapsed_coord = coord.collapsed(0)

        expected = "foo|--|bing"
        assert collapsed_coord.points == expected

    def test_string_nd_first(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str))

        collapsed_coord = coord.collapsed(0)
        expected = [
            "0.0|40.0|80.0",
            "10.0|50.0|90.0",
            "20.0|60.0|100.0",
            "30.0|70.0|110.0",
        ]

        _shared_utils.assert_array_equal(collapsed_coord.points, expected)

    def test_string_nd_second(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str))

        collapsed_coord = coord.collapsed(1)
        expected = [
            "0.0|10.0|20.0|30.0",
            "40.0|50.0|60.0|70.0",
            "80.0|90.0|100.0|110.0",
        ]

        _shared_utils.assert_array_equal(collapsed_coord.points, expected)

    def test_string_nd_both(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str))

        collapsed_coord = coord.collapsed()
        expected = ["0.0|10.0|20.0|30.0|40.0|50.0|60.0|70.0|80.0|90.0|100.0|110.0"]

        _shared_utils.assert_array_equal(collapsed_coord.points, expected)

    def test_string_nd_bounds_first(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str), bounds=self.bds_real.astype(str))

        collapsed_coord = coord.collapsed(0)

        # Points handling is as for non bounded case.  So just check bounds.
        expected_lower = [
            "-2.0|38.0|78.0",
            "8.0|48.0|88.0",
            "18.0|58.0|98.0",
            "28.0|68.0|108.0",
        ]

        expected_upper = [
            "2.0|42.0|82.0",
            "12.0|52.0|92.0",
            "22.0|62.0|102.0",
            "32.0|72.0|112.0",
        ]

        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 0], expected_lower)
        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 1], expected_upper)

    def test_string_nd_bounds_second(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str), bounds=self.bds_real.astype(str))

        collapsed_coord = coord.collapsed(1)

        # Points handling is as for non bounded case.  So just check bounds.
        expected_lower = [
            "-2.0|8.0|18.0|28.0",
            "38.0|48.0|58.0|68.0",
            "78.0|88.0|98.0|108.0",
        ]

        expected_upper = [
            "2.0|12.0|22.0|32.0",
            "42.0|52.0|62.0|72.0",
            "82.0|92.0|102.0|112.0",
        ]

        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 0], expected_lower)
        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 1], expected_upper)

    def test_string_nd_bounds_both(self):
        self.setup_test_arrays((3, 4))
        coord = AuxCoord(self.pts_real.astype(str), bounds=self.bds_real.astype(str))

        collapsed_coord = coord.collapsed()

        # Points handling is as for non bounded case.  So just check bounds.
        expected_lower = ["-2.0|8.0|18.0|28.0|38.0|48.0|58.0|68.0|78.0|88.0|98.0|108.0"]
        expected_upper = [
            "2.0|12.0|22.0|32.0|42.0|52.0|62.0|72.0|82.0|92.0|102.0|112.0"
        ]

        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 0], expected_lower)
        _shared_utils.assert_array_equal(collapsed_coord.bounds[:, 1], expected_upper)


class Test_is_compatible:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.test_coord = AuxCoord([1.0])
        self.other_coord = self.test_coord.copy()

    def test_noncommon_array_attrs_compatible(self):
        # Non-common array attributes should be ok.
        self.test_coord.attributes["array_test"] = np.array([1.0, 2, 3])
        assert self.test_coord.is_compatible(self.other_coord)

    def test_matching_array_attrs_compatible(self):
        # Matching array attributes should be ok.
        self.test_coord.attributes["array_test"] = np.array([1.0, 2, 3])
        self.other_coord.attributes["array_test"] = np.array([1.0, 2, 3])
        assert self.test_coord.is_compatible(self.other_coord)

    def test_different_array_attrs_incompatible(self):
        # Differing array attributes should make coords incompatible.
        self.test_coord.attributes["array_test"] = np.array([1.0, 2, 3])
        self.other_coord.attributes["array_test"] = np.array([1.0, 2, 777.7])
        assert not self.test_coord.is_compatible(self.other_coord)

    def test_misshaped_array_attrs_incompatible(self):
        # Comparison should avoid broadcast failures and return False.
        self.test_coord.attributes["array_test"] = np.array([1.0, 2, 3])
        self.other_coord.attributes["array_test"] = np.array([1.0, 2])
        assert not self.test_coord.is_compatible(self.other_coord)


class Test_contiguous_bounds:
    def test_1d_coord_no_bounds_warning(self):
        coord = DimCoord([0, 1, 2], standard_name="latitude")
        msg = "Coordinate 'latitude' is not bounded, guessing contiguous bounds."
        with pytest.warns(match=msg):
            coord.contiguous_bounds()

    def test_2d_coord_no_bounds_error(self):
        coord = AuxCoord(np.array([[0, 0], [5, 5]]), standard_name="latitude")
        emsg = "Guessing bounds of 2D coords is not currently supported"
        with pytest.raises(ValueError, match=emsg):
            coord.contiguous_bounds()

    def test__sanity_check_bounds_call(self, mocker):
        coord = DimCoord([5, 15, 25], bounds=[[0, 10], [10, 20], [20, 30]])
        bounds_check = mocker.patch("iris.coords.Coord._sanity_check_bounds")
        coord.contiguous_bounds()
        bounds_check.assert_called_once()

    def test_1d_coord(self):
        coord = DimCoord(
            [2, 4, 6],
            standard_name="latitude",
            bounds=[[1, 3], [3, 5], [5, 7]],
        )
        expected = np.array([1, 3, 5, 7])
        result = coord.contiguous_bounds()
        _shared_utils.assert_array_equal(result, expected)

    def test_1d_coord_discontiguous(self):
        coord = DimCoord(
            [2, 4, 6],
            standard_name="latitude",
            bounds=[[1, 3], [4, 5], [5, 7]],
        )
        expected = np.array([1, 4, 5, 7])
        result = coord.contiguous_bounds()
        _shared_utils.assert_array_equal(result, expected)

    def test_2d_lon_bounds(self):
        coord = AuxCoord(
            np.array([[1, 3], [1, 3]]),
            bounds=np.array(
                [[[0, 2, 2, 0], [2, 4, 4, 2]], [[0, 2, 2, 0], [2, 4, 4, 2]]]
            ),
        )
        expected = np.array([[0, 2, 4], [0, 2, 4], [0, 2, 4]])
        result = coord.contiguous_bounds()
        _shared_utils.assert_array_equal(result, expected)

    def test_2d_lat_bounds(self):
        coord = AuxCoord(
            np.array([[1, 1], [3, 3]]),
            bounds=np.array(
                [[[0, 0, 2, 2], [0, 0, 2, 2]], [[2, 2, 4, 4], [2, 2, 4, 4]]]
            ),
        )
        expected = np.array([[0, 0, 0], [2, 2, 2], [4, 4, 4]])
        result = coord.contiguous_bounds()
        _shared_utils.assert_array_equal(result, expected)


class Test_is_contiguous:
    def test_no_bounds(self):
        coord = DimCoord([1, 3])
        result = coord.is_contiguous()
        assert not result

    def test__discontiguity_in_bounds_call(self, mocker):
        # Check that :meth:`iris.coords.Coord._discontiguity_in_bounds` is
        # called.
        coord = DimCoord([1, 3], bounds=[[0, 2], [2, 4]])

        discontiguity_check = mocker.patch("iris.coords.Coord._discontiguity_in_bounds")
        # Discontiguity returns two objects that are unpacked in
        # `coord.is_contiguous`.
        discontiguity_check.return_value = [None, None]
        coord.is_contiguous(rtol=1e-1, atol=1e-3)
        discontiguity_check.assert_called_with(rtol=1e-1, atol=1e-3)


class Test__discontiguity_in_bounds:
    @pytest.fixture(autouse=True)
    def _setup(self):
        self.points_3by3 = np.array([[1, 2, 3], [1, 2, 3], [1, 2, 3]])
        self.lon_bounds_3by3 = np.array(
            [
                [[0, 2, 2, 0], [2, 4, 4, 2], [4, 6, 6, 4]],
                [[0, 2, 2, 0], [2, 4, 4, 2], [4, 6, 6, 4]],
                [[0, 2, 2, 0], [2, 4, 4, 2], [4, 6, 6, 4]],
            ]
        )
        self.lat_bounds_3by3 = np.array(
            [
                [[0, 0, 2, 2], [0, 0, 2, 2], [0, 0, 2, 2]],
                [[2, 2, 4, 4], [2, 2, 4, 4], [2, 2, 4, 4]],
                [[4, 4, 6, 6], [4, 4, 6, 6], [4, 4, 6, 6]],
            ]
        )

    def test_1d_contiguous(self):
        coord = DimCoord([-20, 0, 20], bounds=[[-30, -10], [-10, 10], [10, 30]])
        contiguous, diffs = coord._discontiguity_in_bounds()
        assert contiguous
        _shared_utils.assert_array_equal(diffs, np.zeros(2))

    def test_1d_discontiguous(self):
        coord = DimCoord([10, 20, 40], bounds=[[5, 15], [15, 25], [35, 45]])
        contiguous, diffs = coord._discontiguity_in_bounds()
        assert not contiguous
        _shared_utils.assert_array_equal(diffs, np.array([False, True]))

    def test_1d_one_cell(self):
        # Test a 1D coord with a single cell.
        coord = DimCoord(20, bounds=[[10, 30]])
        contiguous, diffs = coord._discontiguity_in_bounds()
        assert contiguous
        _shared_utils.assert_array_equal(diffs, np.array([]))

    def test_2d_contiguous_both_dirs(self):
        coord = AuxCoord(self.points_3by3, bounds=self.lon_bounds_3by3)
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        assert not diffs_along_x.any()
        assert not diffs_along_y.any()

    def test_2d_discontiguous_along_x(self):
        coord = AuxCoord(
            self.points_3by3[:, ::2], bounds=self.lon_bounds_3by3[:, ::2, :]
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert not contiguous
        _shared_utils.assert_array_equal(
            diffs_along_x, np.array([True, True, True]).reshape(3, 1)
        )
        assert not diffs_along_y.any()

    def test_2d_discontiguous_along_y(self):
        coord = AuxCoord(
            self.points_3by3[::2, :], bounds=self.lat_bounds_3by3[::2, :, :]
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert not contiguous
        assert not diffs_along_x.any()
        _shared_utils.assert_array_equal(diffs_along_y, np.array([[True, True, True]]))

    def test_2d_discontiguous_along_x_and_y(self):
        coord = AuxCoord(
            np.array([[1, 5], [3, 5]]),
            bounds=np.array(
                [[[0, 2, 2, 0], [4, 6, 6, 4]], [[2, 4, 4, 2], [4, 6, 6, 4]]]
            ),
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        exp_x_diffs = np.array([True, False]).reshape(2, 1)
        exp_y_diffs = np.array([True, False]).reshape(1, 2)
        assert not contiguous
        _shared_utils.assert_array_equal(diffs_along_x, exp_x_diffs)
        _shared_utils.assert_array_equal(diffs_along_y, exp_y_diffs)

    def test_2d_contiguous_along_x_atol(self):
        coord = AuxCoord(
            self.points_3by3[:, ::2], bounds=self.lon_bounds_3by3[:, ::2, :]
        )
        # Set a high atol that allows small discontiguities.
        contiguous, diffs = coord._discontiguity_in_bounds(atol=5)
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        _shared_utils.assert_array_equal(
            diffs_along_x, np.array([False, False, False]).reshape(3, 1)
        )
        assert not diffs_along_y.any()

    def test_2d_one_cell(self):
        # Test a 2D coord with a single cell, where the coord has shape (1, 1).
        coord = AuxCoord(
            self.points_3by3[:1, :1], bounds=self.lon_bounds_3by3[:1, :1, :]
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        expected_diffs = np.array([], dtype=np.int64)
        assert contiguous
        _shared_utils.assert_array_equal(diffs_along_x, expected_diffs.reshape(1, 0))
        _shared_utils.assert_array_equal(diffs_along_y, expected_diffs.reshape(0, 1))

    def test_2d_one_cell_along_x(self):
        # Test a 2D coord with a single cell along the x axis, where the coord
        # has shape (2, 1).
        coord = AuxCoord(self.points_3by3[:, :1], bounds=self.lat_bounds_3by3[:, :1, :])
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        assert not diffs_along_x.any()
        _shared_utils.assert_array_equal(diffs_along_y, np.array([0, 0]).reshape(2, 1))

    def test_2d_one_cell_along_y(self):
        # Test a 2D coord with a single cell along the y axis, where the coord
        # has shape (1, 2).
        coord = AuxCoord(self.points_3by3[:1, :], bounds=self.lon_bounds_3by3[:1, :, :])
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        assert not diffs_along_x.any()
        assert not diffs_along_y.any()

    def test_2d_contiguous_mod_360(self):
        # Test that longitude coordinates are adjusted by the 360 modulus when
        # calculating the discontiguities in contiguous bounds.
        coord = AuxCoord(
            [[175, -175], [175, -175]],
            standard_name="longitude",
            bounds=np.array(
                [
                    [[170, 180, 180, 170], [-180, -170, -170, -180]],
                    [[170, 180, 180, 170], [-180, -170, -170, -180]],
                ]
            ),
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        assert not diffs_along_x.any()
        assert not diffs_along_y.any()

    def test_2d_discontiguous_mod_360(self):
        # Test that longitude coordinates are adjusted by the 360 modulus when
        # calculating the discontiguities in contiguous bounds.
        coord = AuxCoord(
            [[175, -175], [175, -175]],
            standard_name="longitude",
            bounds=np.array(
                [
                    [[170, 180, 180, 170], [10, 20, 20, 10]],
                    [[170, 180, 180, 170], [10, 20, 20, 10]],
                ]
            ),
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert not contiguous
        _shared_utils.assert_array_equal(diffs_along_x, np.array([[True], [True]]))
        assert not diffs_along_y.any()

    def test_2d_contiguous_mod_360_not_longitude(self):
        # Test that non-longitude coordinates are not adjusted by the 360
        # modulus when calculating the discontiguities in contiguous bounds.
        coord = AuxCoord(
            [[-150, 350], [-150, 350]],
            standard_name="height",
            bounds=np.array(
                [
                    [[-400, 100, 100, -400], [100, 600, 600, 100]],
                    [[-400, 100, 100, -400], [100, 600, 600, 100]],
                ]
            ),
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert contiguous
        assert not diffs_along_x.any()
        assert not diffs_along_y.any()

    def test_2d_discontiguous_mod_360_not_longitude(self):
        # Test that non-longitude coordinates are not adjusted by the 360
        # modulus when calculating the discontiguities in discontiguous bounds.
        coord = AuxCoord(
            [[-150, 350], [-150, 350]],
            standard_name="height",
            bounds=np.array(
                [
                    [[-400, 100, 100, -400], [200, 600, 600, 200]],
                    [[-400, 100, 100, -400], [200, 600, 600, 200]],
                ]
            ),
        )
        contiguous, diffs = coord._discontiguity_in_bounds()
        diffs_along_x, diffs_along_y = diffs
        assert not contiguous
        _shared_utils.assert_array_equal(diffs_along_x, np.array([[True], [True]]))
        assert not diffs_along_y.any()


class Test__sanity_check_bounds:
    def test_coord_1d_2_bounds(self):
        # Check that a 1d coord with 2 bounds does not raise an error.
        coord = iris.coords.DimCoord(
            [0, 1], standard_name="latitude", bounds=[[0, 1], [1, 2]]
        )
        coord._sanity_check_bounds()

    def test_coord_1d_no_bounds(self):
        coord = iris.coords.DimCoord([0, 1], standard_name="latitude")
        emsg = "Contiguous bounds are only defined for 1D coordinates with 2 bounds."
        with pytest.raises(ValueError, match=emsg):
            coord._sanity_check_bounds()

    def test_coord_1d_1_bounds(self):
        coord = iris.coords.DimCoord(
            [0, 1], standard_name="latitude", bounds=np.array([[0], [1]])
        )
        emsg = "Contiguous bounds are only defined for 1D coordinates with 2 bounds."
        with pytest.raises(ValueError, match=emsg):
            coord._sanity_check_bounds()

    def test_coord_2d_4_bounds(self):
        coord = iris.coords.AuxCoord(
            [[0, 0], [1, 1]],
            standard_name="latitude",
            bounds=np.array(
                [[[0, 0, 1, 1], [0, 0, 1, 1]], [[1, 1, 2, 2], [1, 1, 2, 2]]]
            ),
        )
        coord._sanity_check_bounds()

    def test_coord_2d_no_bounds(self):
        coord = iris.coords.AuxCoord([[0, 0], [1, 1]], standard_name="latitude")
        emsg = "Contiguous bounds are only defined for 2D coordinates with 4 bounds."
        with pytest.raises(ValueError, match=emsg):
            coord._sanity_check_bounds()

    def test_coord_2d_2_bounds(self):
        coord = iris.coords.AuxCoord(
            [[0, 0], [1, 1]],
            standard_name="latitude",
            bounds=np.array([[[0, 1], [0, 1]], [[1, 2], [1, 2]]]),
        )
        emsg = "Contiguous bounds are only defined for 2D coordinates with 4 bounds."
        with pytest.raises(ValueError, match=emsg):
            coord._sanity_check_bounds()

    def test_coord_3d(self):
        coord = iris.coords.AuxCoord(np.zeros((2, 2, 2)), standard_name="height")
        emsg = (
            "Contiguous bounds are not defined for coordinates with more "
            "than 2 dimensions."
        )
        with pytest.raises(ValueError, match=emsg):
            coord._sanity_check_bounds()


class Test_convert_units:
    def test_convert_unknown_units(self):
        coord = iris.coords.AuxCoord(1, units="unknown")
        emsg = (
            "Cannot convert from unknown units. "
            'The "units" attribute may be set directly.'
        )
        with pytest.raises(UnitConversionError, match=emsg):
            coord.convert_units("degrees")

    @pytest.mark.parametrize(
        "attribute",
        [
            "valid_min",
            "valid_max",
            "valid_range",
            "actual_range",
        ],
    )
    def test_convert_attributes(self, attribute):
        coord = iris.coords.AuxCoord(1, units="m")
        value = np.array([0, 10]) if attribute.endswith("range") else 0
        coord.attributes[attribute] = value
        coord.convert_units("ft")
        converted_value = cf_units.Unit("m").convert(value, "ft")
        _shared_utils.assert_array_all_close(
            coord.attributes[attribute], converted_value
        )


class Test___str__:
    def test_short_time_interval(self):
        coord = DimCoord([5], standard_name="time", units="days since 1970-01-01")
        expected = "\n".join(
            [
                "DimCoord :  time / (days since 1970-01-01, standard calendar)",
                "    points: [1970-01-06 00:00:00]",
                "    shape: (1,)",
                "    dtype: int64",
                "    standard_name: 'time'",
            ]
        )
        result = coord.__str__()
        assert expected == result

    def test_short_time_interval__bounded(self):
        coord = DimCoord([5, 6], standard_name="time", units="days since 1970-01-01")
        coord.guess_bounds()
        expected = "\n".join(
            [
                "DimCoord :  time / (days since 1970-01-01, standard calendar)",
                "    points: [1970-01-06 00:00:00, 1970-01-07 00:00:00]",
                "    bounds: [",
                "        [1970-01-05 12:00:00, 1970-01-06 12:00:00],",
                "        [1970-01-06 12:00:00, 1970-01-07 12:00:00]]",
                "    shape: (2,)  bounds(2, 2)",
                "    dtype: int64",
                "    standard_name: 'time'",
            ]
        )
        result = coord.__str__()
        assert expected == result

    def test_long_time_interval(self):
        coord = DimCoord([5], standard_name="time", units="years since 1970-01-01")
        expected = "\n".join(
            [
                "DimCoord :  time / (years since 1970-01-01, standard calendar)",
                "    points: [5]",
                "    shape: (1,)",
                "    dtype: int64",
                "    standard_name: 'time'",
            ]
        )
        result = coord.__str__()
        assert expected == result

    def test_long_time_interval__bounded(self):
        coord = DimCoord([5, 6], standard_name="time", units="years since 1970-01-01")
        coord.guess_bounds()
        expected = "\n".join(
            [
                "DimCoord :  time / (years since 1970-01-01, standard calendar)",
                "    points: [5, 6]",
                "    bounds: [",
                "        [4.5, 5.5],",
                "        [5.5, 6.5]]",
                "    shape: (2,)  bounds(2, 2)",
                "    dtype: int64",
                "    standard_name: 'time'",
            ]
        )
        result = coord.__str__()
        assert expected == result

    def test_non_time_unit(self):
        coord = DimCoord([1.0])
        expected = "\n".join(
            [
                "DimCoord :  unknown / (unknown)",
                "    points: [1.]",
                "    shape: (1,)",
                "    dtype: float64",
            ]
        )
        result = coord.__str__()
        assert expected == result


class TestClimatology:
    # Variety of tests for the climatological property of a coord.
    # Only using AuxCoord since there is no different behaviour between Aux
    # and DimCoords for this property.

    def test_create(self):
        coord = AuxCoord(
            points=[0, 1],
            bounds=[[0, 1], [1, 2]],
            units="days since 1970-01-01",
            climatological=True,
        )
        assert coord.climatological

    def test_create_no_bounds_no_set(self):
        with pytest.raises(ValueError, match="Cannot set.*no bounds exist"):
            AuxCoord(
                points=[0, 1],
                units="days since 1970-01-01",
                climatological=True,
            )

    def test_create_no_time_no_set(self):
        emsg = "Cannot set climatological .* valid time reference units.*"
        with pytest.raises(TypeError, match=emsg):
            AuxCoord(points=[0, 1], bounds=[[0, 1], [1, 2]], climatological=True)

    def test_absent(self):
        coord = AuxCoord(points=[0, 1], bounds=[[0, 1], [1, 2]])
        assert not coord.climatological

    def test_absent_no_bounds_no_set(self):
        coord = AuxCoord(points=[0, 1], units="days since 1970-01-01")
        with pytest.raises(ValueError, match="Cannot set.*no bounds exist"):
            coord.climatological = True

    def test_absent_no_time_no_set(self):
        coord = AuxCoord(points=[0, 1], bounds=[[0, 1], [1, 2]])
        emsg = "Cannot set climatological .* valid time reference units.*"
        with pytest.raises(TypeError, match=emsg):
            coord.climatological = True

    def test_absent_no_bounds_unset(self):
        coord = AuxCoord(points=[0, 1])
        coord.climatological = False
        assert not coord.climatological

    def test_bounds_set(self):
        coord = AuxCoord(
            points=[0, 1],
            bounds=[[0, 1], [1, 2]],
            units="days since 1970-01-01",
        )
        coord.climatological = True
        assert coord.climatological

    def test_bounds_unset(self):
        coord = AuxCoord(
            points=[0, 1],
            bounds=[[0, 1], [1, 2]],
            units="days since 1970-01-01",
            climatological=True,
        )
        coord.climatological = False
        assert not coord.climatological

    def test_remove_bounds(self):
        coord = AuxCoord(
            points=[0, 1],
            bounds=[[0, 1], [1, 2]],
            units="days since 1970-01-01",
            climatological=True,
        )
        coord.bounds = None
        assert not coord.climatological

    def test_change_units(self):
        coord = AuxCoord(
            points=[0, 1],
            bounds=[[0, 1], [1, 2]],
            units="days since 1970-01-01",
            climatological=True,
        )
        assert coord.climatological
        coord.units = "K"
        assert not coord.climatological


class TestIgnoreAxis:
    def test_default(self, sample_coord):
        assert sample_coord.ignore_axis is False

    def test_set_true(self, sample_coord):
        sample_coord.ignore_axis = True
        assert sample_coord.ignore_axis is True

    def test_set_random_value(self, sample_coord):
        with pytest.raises(
            ValueError,
            match=r"'ignore_axis' can only be set to 'True' or 'False'",
        ):
            sample_coord.ignore_axis = "foo"

    @pytest.mark.parametrize(
        ("ignore_axis", "copy_or_from", "result"),
        [
            (True, "copy", True),
            (True, "from_coord", True),
            (False, "copy", False),
            (False, "from_coord", False),
        ],
    )
    def test_copy_coord(self, ignore_axis, copy_or_from, result, sample_coord):
        sample_coord.ignore_axis = ignore_axis
        if copy_or_from == "copy":
            new_coord = sample_coord.copy()
        elif copy_or_from == "from_coord":
            new_coord = sample_coord.from_coord(sample_coord)
        assert new_coord.ignore_axis is result


class Test___init____abstractmethod:
    def test_abstract(self):
        emsg = "Can't instantiate abstract class Coord"
        with pytest.raises(TypeError, match=emsg):
            _ = Coord(points=[0, 1])


class Test_cube_dims:
    def test_cube_dims(self, mocker):
        # Check that "coord.cube_dims(cube)" calls "cube.coord_dims(coord)".
        mock_dims_result = mocker.sentinel.COORD_DIMS
        mock_dims_call = mocker.Mock(return_value=mock_dims_result)
        mock_cube = mocker.Mock(Cube, coord_dims=mock_dims_call)
        test_coord = AuxCoord([1], long_name="test_name")

        result = test_coord.cube_dims(mock_cube)
        assert result == mock_dims_result
        assert mock_dims_call.call_args_list == [mocker.call(test_coord)]
