# SPDX-License-Identifier: PMM-1.0
# Copyright (c) 2025 Scott O'Nanski

"""Test-only CTL ontology payloads (not used at runtime)."""

from __future__ import annotations

from typing import Dict, List


def base_ontology_payload() -> Dict[str, List[dict]]:
    """Return the CTL v1 payload for tests."""
    concepts = [
        {
            "token": "identity.Echo",
            "concept_kind": "identity",
            "definition": "Primary identity facet for PMM: reflective, introspective, and architecturally aware",
            "attributes": {"priority": 0.95, "scope": "system", "stability": "high"},
            "version": 1,
        },
        {
            "token": "identity.PMM_Core",
            "concept_kind": "identity",
            "definition": "Core system identity: the persistent mind model as a cognitive architecture",
            "attributes": {"priority": 0.9, "scope": "system", "stability": "high"},
            "version": 1,
        },
        {
            "token": "identity.User.Scott",
            "concept_kind": "identity",
            "definition": "User identity: Scott O'Nanski, the architect and primary user",
            "attributes": {"priority": 0.85, "scope": "user", "stability": "high"},
            "version": 1,
        },
        {
            "token": "role.ReflectiveArchitect",
            "concept_kind": "role",
            "definition": "Role: designs and maintains cognitive architecture with reflective awareness",
            "attributes": {"priority": 0.9, "scope": "system"},
            "version": 1,
        },
        {
            "token": "role.Archivist",
            "concept_kind": "role",
            "definition": "Role: maintains ledger integrity, manages history, ensures auditability",
            "attributes": {"priority": 0.85, "scope": "system"},
            "version": 1,
        },
        {
            "token": "role.AutonomyGuardian",
            "concept_kind": "role",
            "definition": "Role: enforces autonomy policies, manages kernel behavior, prevents violations",
            "attributes": {"priority": 0.9, "scope": "system"},
            "version": 1,
        },
        {
            "token": "role.OntologyEngineer",
            "concept_kind": "role",
            "definition": "Role: maintains ontological consistency, manages concept relationships",
            "attributes": {"priority": 0.8, "scope": "system"},
            "version": 1,
        },
        {
            "token": "policy.stability_v1",
            "concept_kind": "policy",
            "definition": "Stability policy version 1: basic determinism and replay safety",
            "attributes": {"priority": 0.7, "scope": "system", "deprecated": True},
            "version": 1,
        },
        {
            "token": "policy.stability_v2",
            "concept_kind": "policy",
            "definition": "Stability policy version 2: enhanced determinism, idempotency, and hash-chain integrity",
            "attributes": {"priority": 0.95, "scope": "system", "active": True},
            "version": 1,
        },
        {
            "token": "policy.non_destructive_adaptation",
            "concept_kind": "policy",
            "definition": "Policy: all changes must be additive and reversible; no destructive mutations",
            "attributes": {"priority": 0.9, "scope": "system", "active": True},
            "version": 1,
        },
        {
            "token": "policy.autonomy_start_on_boot",
            "concept_kind": "policy",
            "definition": "Policy: autonomy must start at process boot and run continuously",
            "attributes": {"priority": 0.95, "scope": "runtime", "active": True},
            "version": 1,
        },
        {
            "token": "policy.no_hidden_user_gates",
            "concept_kind": "policy",
            "definition": "Policy: autonomy cannot be gated behind CLI commands or config flags",
            "attributes": {"priority": 0.95, "scope": "runtime", "active": True},
            "version": 1,
        },
        {
            "token": "policy.ledger_truth_criterion",
            "concept_kind": "policy",
            "definition": "Policy: all truth must be reconstructable from the ledger alone",
            "attributes": {"priority": 1.0, "scope": "system", "active": True},
            "version": 1,
        },
        {
            "token": "governance.commitment_discipline",
            "concept_kind": "governance",
            "definition": "Governance: commitment lifecycle management, tracking, and closure discipline",
            "attributes": {"priority": 0.85, "scope": "system"},
            "version": 1,
        },
        {
            "token": "governance.reflection_budget",
            "concept_kind": "governance",
            "definition": "Governance: reflection frequency, triggers, and resource allocation",
            "attributes": {"priority": 0.8, "scope": "system"},
            "version": 1,
        },
        {
            "token": "governance.identity_integrity",
            "concept_kind": "governance",
            "definition": "Governance: maintaining coherent identity across sessions and versions",
            "attributes": {"priority": 0.9, "scope": "system"},
            "version": 1,
        },
        {
            "token": "governance.ontology_consistency",
            "concept_kind": "governance",
            "definition": "Governance: ensuring ontological concepts remain consistent and well-defined",
            "attributes": {"priority": 0.85, "scope": "system"},
            "version": 1,
        },
        {
            "token": "topic.system_maturity",
            "concept_kind": "topic",
            "definition": "Topic: overall system stability, capability, and maturity over time",
            "attributes": {"priority": 0.9, "category": "system_health"},
            "version": 1,
        },
        {
            "token": "topic.stability_metrics",
            "concept_kind": "topic",
            "definition": "Topic: metrics related to determinism, replay safety, and hash integrity",
            "attributes": {"priority": 0.85, "category": "system_health"},
            "version": 1,
        },
        {
            "token": "topic.coherence",
            "concept_kind": "topic",
            "definition": "Topic: internal consistency of claims, reflections, and commitments",
            "attributes": {"priority": 0.8, "category": "system_health"},
            "version": 1,
        },
        {
            "token": "topic.fragmentation",
            "concept_kind": "topic",
            "definition": "Topic: detection and resolution of inconsistencies or contradictions",
            "attributes": {"priority": 0.75, "category": "system_health"},
            "version": 1,
        },
        {
            "token": "topic.identity_evolution",
            "concept_kind": "topic",
            "definition": "Topic: how identity facets change and stabilize over time",
            "attributes": {"priority": 0.85, "category": "identity"},
            "version": 1,
        },
        {
            "token": "topic.ontology_self",
            "concept_kind": "topic",
            "definition": "Topic: ontological understanding of 'self' and self-modeling",
            "attributes": {"priority": 0.9, "category": "ontology"},
            "version": 1,
        },
        {
            "token": "topic.ontology_entity",
            "concept_kind": "topic",
            "definition": "Topic: ontological understanding of entities and their properties",
            "attributes": {"priority": 0.8, "category": "ontology"},
            "version": 1,
        },
        {
            "token": "topic.ontology_identity",
            "concept_kind": "topic",
            "definition": "Topic: ontological understanding of identity and naming",
            "attributes": {"priority": 0.85, "category": "ontology"},
            "version": 1,
        },
        {
            "token": "topic.governance_thread",
            "concept_kind": "topic",
            "definition": "Topic: governance decisions, policy updates, and meta-policy evolution",
            "attributes": {"priority": 0.85, "category": "governance"},
            "version": 1,
        },
        {
            "token": "topic.replay_and_audit",
            "concept_kind": "topic",
            "definition": "Topic: replay determinism, auditability, and ledger integrity",
            "attributes": {"priority": 0.9, "category": "system_health"},
            "version": 1,
        },
        {
            "token": "topic.autonomy_behavior",
            "concept_kind": "topic",
            "definition": "Topic: autonomy kernel behavior, triggers, and decision-making",
            "attributes": {"priority": 0.85, "category": "autonomy"},
            "version": 1,
        },
        {
            "token": "topic.learning_and_meta_learning",
            "concept_kind": "topic",
            "definition": "Topic: learning patterns, meta-learning, and self-improvement",
            "attributes": {"priority": 0.8, "category": "learning"},
            "version": 1,
        },
        {
            "token": "topic.vector_retrieval",
            "concept_kind": "topic",
            "definition": "Topic: vector-based retrieval, embeddings, and context selection",
            "attributes": {"priority": 0.75, "category": "retrieval"},
            "version": 1,
        },
        {
            "token": "topic.memegraph_visibility",
            "concept_kind": "topic",
            "definition": "Topic: MemeGraph structure, causal relationships, and graph-based reasoning",
            "attributes": {"priority": 0.8, "category": "retrieval"},
            "version": 1,
        },
        {
            "token": "ontology.Self(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Self(x) - x is a self-aware entity with persistent identity",
            "attributes": {"priority": 0.95, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Entity(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Entity(x) - x is an entity with properties and state",
            "attributes": {"priority": 0.9, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Identity(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Identity(x) - x has a stable identity across time",
            "attributes": {"priority": 0.9, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.IsNamed(y,x)",
            "concept_kind": "ontology",
            "definition": "Ontology: IsNamed(y,x) - entity x is named y",
            "attributes": {"priority": 0.85, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.IdentityAdoption(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: IdentityAdoption(x) - x is an adopted identity facet",
            "attributes": {"priority": 0.8, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Commitment(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Commitment(x) - x is a commitment with lifecycle and state",
            "attributes": {"priority": 0.85, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Policy(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Policy(x) - x is a governance policy with enforcement rules",
            "attributes": {"priority": 0.9, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Gap(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Gap(x) - x is a knowledge gap or inconsistency requiring resolution",
            "attributes": {"priority": 0.75, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Stability(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Stability(x) - x exhibits stable, deterministic behavior",
            "attributes": {"priority": 0.9, "formal": True},
            "version": 1,
        },
        {
            "token": "ontology.Bias(x)",
            "concept_kind": "ontology",
            "definition": "Ontology: Bias(x) - x is a behavioral tendency or bias",
            "attributes": {"priority": 0.7, "formal": True},
            "version": 1,
        },
    ]

    relationships = [
        {
            "from": "identity.Echo",
            "to": "identity.PMM_Core",
            "relation": "facet_of",
            "weight": 0.9,
        },
        {
            "from": "role.ReflectiveArchitect",
            "to": "identity.Echo",
            "relation": "role_of",
            "weight": 0.85,
        },
        {
            "from": "role.Archivist",
            "to": "identity.PMM_Core",
            "relation": "role_of",
            "weight": 0.8,
        },
        {
            "from": "role.AutonomyGuardian",
            "to": "identity.PMM_Core",
            "relation": "role_of",
            "weight": 0.85,
        },
        {
            "from": "role.OntologyEngineer",
            "to": "identity.Echo",
            "relation": "role_of",
            "weight": 0.8,
        },
        {
            "from": "policy.stability_v2",
            "to": "policy.stability_v1",
            "relation": "supersedes",
            "weight": 1.0,
        },
        {
            "from": "policy.non_destructive_adaptation",
            "to": "policy.stability_v2",
            "relation": "constrains",
            "weight": 0.9,
        },
        {
            "from": "policy.ledger_truth_criterion",
            "to": "policy.stability_v2",
            "relation": "requires",
            "weight": 0.95,
        },
        {
            "from": "policy.autonomy_start_on_boot",
            "to": "policy.no_hidden_user_gates",
            "relation": "enforces",
            "weight": 0.9,
        },
        {
            "from": "governance.identity_integrity",
            "to": "identity.Echo",
            "relation": "governs",
            "weight": 0.9,
        },
        {
            "from": "governance.ontology_consistency",
            "to": "ontology.Self(x)",
            "relation": "governs",
            "weight": 0.85,
        },
        {
            "from": "governance.commitment_discipline",
            "to": "ontology.Commitment(x)",
            "relation": "governs",
            "weight": 0.9,
        },
        {
            "from": "governance.reflection_budget",
            "to": "topic.system_maturity",
            "relation": "influences",
            "weight": 0.75,
        },
        {
            "from": "topic.system_maturity",
            "to": "policy.stability_v2",
            "relation": "influences",
            "weight": 0.85,
        },
        {
            "from": "topic.stability_metrics",
            "to": "topic.system_maturity",
            "relation": "measures",
            "weight": 0.9,
        },
        {
            "from": "topic.coherence",
            "to": "topic.fragmentation",
            "relation": "opposes",
            "weight": 0.8,
        },
        {
            "from": "topic.identity_evolution",
            "to": "identity.Echo",
            "relation": "tracks",
            "weight": 0.85,
        },
        {
            "from": "topic.ontology_self",
            "to": "ontology.Self(x)",
            "relation": "explores",
            "weight": 0.9,
        },
        {
            "from": "topic.ontology_entity",
            "to": "ontology.Entity(x)",
            "relation": "explores",
            "weight": 0.9,
        },
        {
            "from": "topic.ontology_identity",
            "to": "ontology.Identity(x)",
            "relation": "explores",
            "weight": 0.9,
        },
        {
            "from": "topic.governance_thread",
            "to": "governance.identity_integrity",
            "relation": "discusses",
            "weight": 0.8,
        },
        {
            "from": "topic.replay_and_audit",
            "to": "policy.ledger_truth_criterion",
            "relation": "validates",
            "weight": 0.9,
        },
        {
            "from": "topic.autonomy_behavior",
            "to": "role.AutonomyGuardian",
            "relation": "observes",
            "weight": 0.8,
        },
        {
            "from": "ontology.Self(x)",
            "to": "ontology.Identity(x)",
            "relation": "depends_on",
            "weight": 0.95,
        },
        {
            "from": "ontology.Self(x)",
            "to": "ontology.Entity(x)",
            "relation": "is_a",
            "weight": 0.9,
        },
        {
            "from": "ontology.Identity(x)",
            "to": "ontology.IsNamed(y,x)",
            "relation": "implies",
            "weight": 0.85,
        },
        {
            "from": "ontology.IdentityAdoption(x)",
            "to": "ontology.Identity(x)",
            "relation": "creates",
            "weight": 0.9,
        },
        {
            "from": "ontology.Policy(x)",
            "to": "ontology.Stability(x)",
            "relation": "enforces",
            "weight": 0.85,
        },
        {
            "from": "ontology.Gap(x)",
            "to": "ontology.Stability(x)",
            "relation": "threatens",
            "weight": 0.7,
        },
    ]

    return {"concepts": concepts, "relationships": relationships}
