#!/usr/bin/env python3
# SPDX-License-Identifier: PMM-1.0
# Copyright (c) 2025 Scott O'Nanski
#
# PMM — Triple Exporter (Readable / Telemetry / Ledger)
# Generates linked exports:
#  1. Human-readable chat transcript
#  2. Telemetry summary for auditing
#  3. Compressed raw ledger JSON for AI verification

import sqlite3
import json
import gzip
import hashlib
from datetime import datetime, timezone
from pathlib import Path
from collections import Counter
import textwrap


def sha256(data: str) -> str:
    """Return SHA256 digest of provided string."""
    return hashlib.sha256(data.encode("utf-8")).hexdigest()


def _sql_literal(value: str | None) -> str:
    """Render a Python string as a simple SQL literal for telemetry snippets.

    Intended for human-readable inspection; not an exact .dump re-encoder.
    Newlines are compacted to spaces; single quotes are doubled.
    """
    if value is None:
        return "NULL"
    cleaned = value.replace("\n", " ").replace("\r", " ")
    cleaned = cleaned.strip()
    cleaned = cleaned.replace("'", "''")
    return f"'{cleaned}'"


def export_session():
    repo_root = Path(__file__).resolve().parent.parent
    db_path = repo_root / ".data" / "pmmdb" / "pmm.db"
    if not db_path.exists():
        print(f"[ERROR] PMM database not found at {db_path}")
        return

    # Common timestamp key
    now = datetime.now(timezone.utc).strftime("%Y-%m-%d_%H-%M-%S")
    readable_path = repo_root / f"chat_session_{now}_readable.md"
    telemetry_path = repo_root / f"chat_session_{now}_telemetry.md"
    ledger_path = repo_root / f"chat_session_{now}_ledger.json.gz"

    # Query all events
    conn = sqlite3.connect(db_path)
    cursor = conn.cursor()
    cursor.execute(
        """
        SELECT id, ts, kind, content, meta, prev_hash, hash
        FROM events
        ORDER BY id ASC
    """
    )
    rows = cursor.fetchall()
    conn.close()

    # Build counters and state trackers
    kinds = Counter()
    breaks = []
    continuity = []
    chat_msgs = []

    # --- Ledger export data (for gzip) ---
    raw_ledger = []

    for eid, ts, kind, content, meta, prev_hash, hsh in rows:
        kinds[kind] += 1
        raw_ledger.append(
            {
                "id": eid,
                "ts": ts,
                "kind": kind,
                "content": content,
                "meta": meta,
                "prev_hash": prev_hash,
                "hash": hsh,
            }
        )

        # Detect continuity gaps
        if continuity and prev_hash and prev_hash != continuity[-1]:
            breaks.append((eid, continuity[-1], prev_hash))
        continuity.append(hsh)

        # Collect visible chat messages
        if kind in {"user_message", "assistant_message"}:
            visible = textwrap.dedent(content).strip()
            if kind == "assistant_message":
                hidden_prefixes = ("COMMIT:", "CLOSE:", "CLAIM:", "REFLECT:")
                visible = "\n".join(
                    line
                    for line in visible.splitlines()
                    if not line.startswith(hidden_prefixes)
                ).strip()
            chat_msgs.append((kind, ts, visible))

    # ============================================================
    # 1. Human-readable Markdown (chat only)
    # ============================================================
    readable = []
    readable.append("# Persistent Mind Model — Readable Chat Log\n")
    readable.append(f"**Exported:** {now.replace('_',' ')} UTC\n")
    readable.append(f"**Linked Telemetry:** `{telemetry_path.name}`  \n")
    readable.append(f"**Linked Ledger:** `{ledger_path.name}`\n")
    readable.append("---\n")

    for idx, (kind, ts, content) in enumerate(chat_msgs, start=1):
        role = "👤 User" if kind == "user_message" else "🤖 Echo"
        readable.append(f"### Turn {idx}: {role}\n*{ts}*\n\n```text\n{content}\n```\n")

    readable.append(
        "\n---\n_End of readable log — see telemetry or ledger for verification._\n"
    )

    with open(readable_path, "w", encoding="utf-8") as f:
        f.write("\n".join(readable))

    # ============================================================
    # 2. Telemetry Markdown (condensed)
    # ============================================================
    telemetry = []
    telemetry.append("# Persistent Mind Model — Telemetry Summary\n")
    telemetry.append(f"**Exported:** {now.replace('_',' ')} UTC\n")
    telemetry.append(f"**Linked Readable:** `{readable_path.name}`  \n")
    telemetry.append(f"**Linked Ledger:** `{ledger_path.name}`\n")
    telemetry.append("---\n")

    telemetry.append("## 📜 Event Summary\n\n")
    telemetry.append("| ID | Kind | Meta Preview | Prev Hash | Hash |\n")
    telemetry.append("|----|------|--------------|-----------|------|\n")

    for eid, ts, kind, content, meta, prev_hash, hsh in rows:
        # Single-line meta preview for readability (mirrors small_telemetry style)
        meta_display = ""
        if meta and meta.strip() not in ("{}", "null", "NULL"):
            meta_str = meta.replace("\n", " ").replace("\r", " ").strip()
            meta_display = meta_str[:300] + ("…" if len(meta_str) > 300 else "")

        telemetry.append(
            f"| {eid} | {kind} | {meta_display or '—'} | `{prev_hash or '∅'}` | `{hsh or '∅'}` |\n"
        )

    telemetry.append("\n## 📊 Statistics\n\n")
    telemetry.append(f"- **Total Events:** {len(rows)}\n")
    telemetry.append(f"- **Event Types:** {len(kinds)}\n")
    telemetry.append("\n| Kind | Count |\n|------|-------:|\n")
    for k, c in kinds.items():
        telemetry.append(f"| {k} | {c} |\n")
    telemetry.append("\n")

    if breaks:
        telemetry.append("⚠️ **Continuity Breaks Detected:**\n")
        for eid, prev, actual in breaks:
            telemetry.append(f"- Event {eid}: expected `{prev}` but saw `{actual}`\n")
    else:
        telemetry.append("✅ **No hash continuity breaks detected.**\n")

    # ============================================================
    # 2a. SQL-style snippets for last few events
    # ============================================================
    telemetry.append("\n## 🔎 SQL Snippets (Last 10 Events)\n\n")
    telemetry.append("```sql\n")
    tail = rows[-10:] if len(rows) > 10 else rows
    for eid, ts, kind, content, meta, prev_hash, hsh in tail:
        ts_lit = _sql_literal(ts)
        kind_lit = _sql_literal(kind)
        content_lit = _sql_literal(content)
        meta_lit = _sql_literal(meta)
        prev_lit = _sql_literal(prev_hash)
        hash_lit = _sql_literal(hsh)
        telemetry.append(
            "INSERT INTO events (id, ts, kind, content, meta, prev_hash, hash) "
            f"VALUES({eid}, {ts_lit}, {kind_lit}, {content_lit}, "
            f"{meta_lit}, {prev_lit}, {hash_lit});\n"
        )
    telemetry.append("```\n")

    # ============================================================
    # 2b. Adaptive Projections & Metrics (Stability / Coherence / Policy)
    # ============================================================
    # Surface the latest stability/coherence metrics and adaptive policy events
    # in a compact summary, to match the new subsystems.
    latest_stability = None
    latest_coherence = None
    latest_policy_update = None
    latest_meta_policy = None
    for eid, ts, kind, content, meta, prev_hash, hsh in reversed(rows):
        if kind == "stability_metrics" and latest_stability is None:
            latest_stability = content
        elif kind == "coherence_check" and latest_coherence is None:
            latest_coherence = content
        elif kind == "policy_update" and latest_policy_update is None:
            latest_policy_update = content
        elif kind == "meta_policy_update" and latest_meta_policy is None:
            latest_meta_policy = content
        if (
            latest_stability
            and latest_coherence
            and latest_policy_update
            and latest_meta_policy
        ):
            break

    telemetry.append("\n## 🧩 Adaptive Metrics Snapshot\n\n")
    if latest_stability:
        try:
            s = json.loads(latest_stability)
        except Exception:
            s = {}
        telemetry.append("### Stability Metrics (latest)\n\n")
        if isinstance(s, dict):
            stability_score = s.get("stability_score")
            telemetry.append(f"- **Stability Score:** `{stability_score}`\n")
            telemetry.append(f"- **Window Size:** `{s.get('window_size')}`\n")
            metrics = s.get("metrics") or {}
            if isinstance(metrics, dict):
                for key, val in metrics.items():
                    telemetry.append(f"  - `{key}` = `{val}`\n")
        else:
            telemetry.append(f"- Raw content: `{latest_stability}`\n")
        telemetry.append("\n")

    if latest_coherence:
        try:
            c = json.loads(latest_coherence)
        except Exception:
            c = {}
        telemetry.append("### Coherence Check (latest)\n\n")
        if isinstance(c, dict):
            telemetry.append(f"- **Coherence Score:** `{c.get('coherence_score')}`\n")
            telemetry.append(
                f"- **Resolution Needed:** `{c.get('resolution_needed')}`\n"
            )
            scope = c.get("scope") or []
            if scope:
                telemetry.append(f"- **Domains:** {', '.join(scope)}\n")
            conflicts = c.get("conflicts") or []
            telemetry.append(f"- **Conflicts:** `{len(conflicts)}`\n")
        else:
            telemetry.append(f"- Raw content: `{latest_coherence}`\n")
        telemetry.append("\n")

    if latest_policy_update:
        telemetry.append("### Policy Update (latest)\n\n")
        try:
            p = json.loads(latest_policy_update)
        except Exception:
            p = {}
        if isinstance(p, dict):
            changes = p.get("changes") or {}
            telemetry.append(f"- **Changes:** `{changes}`\n")
        else:
            telemetry.append(f"- Raw content: `{latest_policy_update}`\n")
        telemetry.append("\n")

    if latest_meta_policy:
        telemetry.append("### Meta-Policy Update (latest)\n\n")
        try:
            mp = json.loads(latest_meta_policy)
        except Exception:
            mp = {}
        if isinstance(mp, dict):
            changes = mp.get("changes") or {}
            telemetry.append(f"- **Meta Changes:** `{changes}`\n")
        else:
            telemetry.append(f"- Raw content: `{latest_meta_policy}`\n")
        telemetry.append("\n")

    # Concept Graph Snapshot (computed from full history)
    # We can't easily import pmm.core.concept_graph here if we want this script to be standalone-ish,
    # but we can do a quick scan of concept_define events.
    concept_defs = [r for r in rows if r[2] == "concept_define"]
    concept_binds = [
        r for r in rows if r[2] in ("concept_bind_event", "concept_bind_async")
    ]
    async_binds = [r for r in rows if r[2] == "concept_bind_async"]
    async_claims = [r for r in rows if r[2] == "claim_from_text"]

    telemetry.append("### Concept Graph Snapshot\n\n")
    telemetry.append(f"- **Total Definitions:** `{len(concept_defs)}`\n")
    telemetry.append(f"- **Total Bindings:** `{len(concept_binds)}`\n")
    if async_binds or async_claims:
        telemetry.append("\n### 📚 Archivist Activity\n\n")
        telemetry.append(f"- Async Concept Bindings: `{len(async_binds)}`\n")
        telemetry.append(f"- Async Claims Extracted: `{len(async_claims)}`\n")

    if concept_defs:
        # Show last 3 defined concepts
        last_3 = concept_defs[-3:]
        telemetry.append("- **Recent Definitions:**\n")
        for _, _, _, content, _, _, _ in last_3:
            try:
                d = json.loads(content)
                token = d.get("token", "unknown")
                kind = d.get("concept_kind", "unknown")
                telemetry.append(f"  - `{token}` ({kind})\n")
            except Exception:
                pass
    telemetry.append("\n")

    # Manifest for AI verification
    manifest = {
        "export_timestamp": now,
        "linked_readable": readable_path.name,
        "linked_ledger": ledger_path.name,
        "total_events": len(rows),
        "event_type_counts": dict(kinds),
        "continuity_breaks": len(breaks),
        # Digest over event hashes (column 6), not prev_hash
        "sha256_full_digest": sha256("".join(r[6] or "" for r in rows)),
    }
    telemetry.append("\n## 🧾 Verification Manifest\n\n```json\n")
    telemetry.append(json.dumps(manifest, indent=2, ensure_ascii=False))
    telemetry.append("\n```\n\n---\n_End of telemetry summary._\n")

    with open(telemetry_path, "w", encoding="utf-8") as f:
        f.write("\n".join(telemetry))

    # ============================================================
    # 3. Compressed JSON Ledger
    # ============================================================
    with gzip.open(ledger_path, "wt", encoding="utf-8") as f:
        json.dump(raw_ledger, f, indent=2, ensure_ascii=False)

    print(
        f"[OK] Export complete →\n  {readable_path}\n  {telemetry_path}\n  {ledger_path}"
    )


if __name__ == "__main__":
    export_session()
