# ⛅ Optional Packages

Most of the examples included within `trulens` require additional packages
not installed alongside `trulens`. You may be prompted to install them
(with pip). The requirements file `trulens/requirements.optional.txt`
contains the list of optional packages and their use if you'd like to install
them all in one go.

## Dev Notes

To handle optional packages and provide clearer instructions to the user, we
employ a context-manager-based scheme (see `utils/imports.py`) to import
packages that may not be installed. The basic form of such imports can be seen
in `__init__.py`:

```python
with OptionalImports(messages=REQUIREMENT_LLAMA):
    from trulens.apps.llamaindex import TruLlama
```

This ensures that `TruLlama` gets defined subsequently even if the import
fails (because `tru_llama` imports `llama_index` which may not be installed).
However, if the user imports TruLlama (via `__init__.py`) and tries to use it
(call it, look up attribute, etc), they will be presented with a message telling them
that `llama-index` is optional and how to install it:

```python
ModuleNotFoundError:
llama-index package is required for instrumenting llama_index apps.
You should be able to install it with pip:

    pip install "llama-index>=v0.9.14.post3"
```

If a user imports directly from the TruLlama module (not by way of `__init__.py`),
they will get that message immediately instead of upon use due to this line inside
`tru_llama.py`:

```python
OptionalImports(messages=REQUIREMENT_LLAMA).assert_installed(llama_index)
```

This checks that the optional import system did not return a replacement for
`llama_index` (under a context manager earlier in the file).

If used in conjunction, the optional imports context manager and
`assert_installed` check can be simplified by storing a reference to the
`OptionalImports` instance which is returned by the context manager entrance:

```python
with OptionalImports(messages=REQUIREMENT_LLAMA) as opt:
    import llama_index
    ...

opt.assert_installed(llama_index)
```

`assert_installed` also returns the `OptionalImports` instance on success so
assertions can be chained:

```python
opt.assert_installed(package1).assert_installed(package2)
# or
opt.assert_installed[[package1, package2]]
```

### When to Fail

As implied above, imports from a general package that does not imply an
optional package (e.g. `from trulens ...`) should not produce the error
immediately. However, imports from packages that do imply the use of optional imports
(e.g. `tru_llama.py`) should produce the error immediately.
