From iris.proofmode Require Import base proofmode.
From iris.base_logic.lib Require Export fancy_updates.
From iris.program_logic Require Import weakestpre.
From iris.algebra Require Import gset gmap frac.

From dislog.utils Require Import graph more_iris.
From dislog.lang Require Import syntax reducible.
From dislog.newlang Require Export semantics invert_step atomic pureinv.

Set Implicit Arguments.

(* We define the WP w.r.t to an abstract [interp] predicate,
   quantified here. This predicate is realized in [interp.v] *)
Class dislogGS (has_escape:bool) (Σ : gFunctors) :=
  DislogGS {
      (* No later credits, allowing a simple statement of [wpg_par.give_plain] *)
      iinvgs : invGS_gen HasNoLc Σ;
      interp : store -> amap -> graph -> iProp Σ;
    }.

#[global] Existing Instance iinvgs.

(* Some ltac *)
Ltac wpg_intros := iIntros (G α σ) "Hi".
Ltac intros_mod := iApply fupd_mask_intro; [ set_solver | iIntros "Hclose"].
Ltac intros_post := iIntros (???? Hstep).
Ltac mod_all H Z := iMod (H with "[%//]") as H; do 2 iModIntro; iMod H as Z.

Section wpg.
Context `{!dislogGS has_escape Σ}.

(* ------------------------------------------------------------------------ *)
(* The actual wpg *)

(* As usual, the WP is defined as a guarded fixpoint. [wpg_pre] is
   the "factored-out" version. *)

Definition wpg_pre
  (wpg : coPset -d> task_tree -d> expr -d> (val -d> iPropO Σ) -d> iPropO Σ) :
  coPset -d> task_tree -d> expr -d> (val -d> iPropO Σ) -d> iPropO Σ := λ E T e Q,
  match to_val e with
  | Some v => |={E}=> ⌜is_leaf T⌝ ∗ Q v
  | None => ∀ G α σ,
      ⌜pureinv G α σ T e⌝ ∗ interp σ α G ={E,∅}=∗ ⌜reducible has_escape σ α G T e⌝ ∗
      (∀ α' σ' T' e', ⌜step G σ α T e σ' α' T' e'⌝  ={∅}=∗ ▷ |={∅,E}=>
      ⌜pureinv G α' σ' T' e'⌝ ∗ interp σ' α' G ∗ wpg E T' e' Q)
  end%I.

Local Instance wpg_pre_contractive : Contractive wpg_pre.
Proof.
  rewrite /wpg_pre /= => n wpg wp' Hwp ? ? e Q.
  repeat (f_contractive || f_equiv); apply Hwp.
Qed.

(* wpg as the fixpoint of wpg_pre *)
Definition wpg : coPset -> task_tree -> expr -> (val -> iProp Σ) -> iProp Σ :=
  fixpoint wpg_pre.

Lemma wpg_unfold E t e Q :
  wpg E t e Q ⊣⊢ wpg_pre wpg E t e Q.
Proof. apply (fixpoint_unfold wpg_pre). Qed.

(* ------------------------------------------------------------------------ *)
(* Structural rules *)

Lemma fupd_wpg E T e Q :
  (|={E}=> wpg E T e Q) -∗
  wpg E T e Q.
Proof.
  iIntros "Hwp".
  rewrite !wpg_unfold /wpg_pre.
  destruct (to_val e) eqn:He.
  { by iMod "Hwp". }
  iIntros. iMod "Hwp".
  iApply ("Hwp" with "[$]").
Qed.

Lemma bupd_wpg E T e Q :
  (|==> wpg E T e Q) -∗
  wpg E T e Q.
Proof.
  iIntros "Hwp". iApply fupd_wpg. done.
Qed.

Lemma wpg_val_eq E T (v:val) Q :
  wpg E T v Q ⊣⊢ |={E}=> ⌜is_leaf T⌝ ∗ Q v.
Proof.
  rewrite wpg_unfold /wpg_pre //.
Qed.

Lemma wpg_fupd E T e Q :
  wpg E T e (fun v => |={E}=> Q v) -∗
  wpg E T e Q.
Proof.
  iIntros "Hwp".
  iLöb as "IH" forall (T e).
  rewrite !wpg_unfold /wpg_pre.
  destruct (to_val e) eqn:He.
  { iMod "Hwp" as "(?&?)". by iFrame. }
  iIntros. iMod ("Hwp" with "[$]") as "(Hr&Hwp)". iModIntro.
  iFrame "Hr". intros_post.
  mod_all "Hwp" "(?&?&Hwp)". iFrame.
  by iApply "IH".
Qed.

Lemma wpg_strong_mono E T e P Q :
  wpg E T e P -∗
  (∀ v, P v ={E}=∗ Q v) -∗
  wpg E T e Q.
Proof.
  iIntros "Hwp HPQ".
  iLöb as "IH" forall (T e).
  rewrite !wpg_unfold /wpg_pre.
  destruct (to_val e).
  { iMod "Hwp" as "(?&?)".
    iMod ("HPQ" with "[$]"). by iFrame. }
  { iIntros. iMod ("Hwp" with "[$]") as "(Hr&Hwp)". iFrame "Hr".
    iModIntro. iIntros.
    mod_all "Hwp" "(?&?&?)". iFrame. iModIntro.
    iApply ("IH" with "[$][$]"). }
Qed.

Lemma wpg_mono E T e P Q :
  wpg E T e P -∗
  (∀ v, P v -∗ Q v) -∗
  wpg E T e Q.
Proof.
  iIntros "Hwp HPQ".
  iApply (wpg_strong_mono with "Hwp").
  iIntros. iApply "HPQ". by iFrame.
Qed.

Lemma wpg_proper E T e P1 P2 :
  (forall v, P1 v ≡ P2 v) ->
  wpg E T e P1 ≡ wpg E T e P2.
Proof.
  intros X. iSplit.
  all:iIntros; iApply (wpg_mono with "[$]").
  all:iIntros; rewrite X //.
Qed.

Lemma wpg_frame_step P E T e Q :
  ¬ is_val e ->
  ▷ P -∗
  wpg E T e Q -∗
  wpg E T e (fun v => Q v ∗ P).
Proof.
  iIntros (He) "HP Hwp".
  rewrite !wpg_unfold /wpg_pre.
  replace (to_val e) with (@None val) by (symmetry; by apply is_val_false).
  iIntros. iMod ("Hwp" with "[$]") as "(?&Hwp)". iModIntro. iFrame.
  iIntros. mod_all "Hwp" "(?&?&?)". iFrame.
  iModIntro. iApply (wpg_mono with "[$]"). iIntros. by iFrame.
Qed.

Definition root (t:task_tree) : timestamp :=
  match t with
  | Leaf t => t
  | Node t _ _ => t end.

Lemma step_inv_root σ α G T e σ' α' T' e' :
  step G σ α T e σ' α' T' e' ->
  root T = root T'.
Proof.
  induction 1; try done.
  inversion H; try done.
Qed.

(* ------------------------------------------------------------------------ *)
(* rules of the wpg *)

Lemma wpg_val E t v Q :
  Q v -∗ wpg E (Leaf t) (Val v) Q.
Proof.
  iIntros.
  rewrite wpg_unfold /wpg_pre. simpl. by iFrame.
Qed.

Lemma wpg_if E t (b:bool) e1 e2 Q :
  wpg E (Leaf t) (if b then e1 else e2) Q -∗
  wpg E (Leaf t) (If b e1 e2) Q.
Proof.
  iIntros "Hwp".
  iApply wpg_unfold. wpg_intros; intros_mod.
  iSplitR. { eauto using reducible_if. }
  intros_post. do 2 iModIntro. iMod "Hclose".
  apply invert_step_if in Hstep.
  destruct Hstep as (?&?&?&?); subst.
  iDestruct "Hi" as "(%&?)". iFrame.
  eauto using pureinv_if.
Qed.

Lemma wpg_let_val E t x (v:val) e Q :
  wpg E (Leaf t) (subst' x v e) Q -∗
  wpg E (Leaf t) (Let x v e) Q.
Proof.
  iIntros "Hwp".
  iApply wpg_unfold. wpg_intros. intros_mod.
  iSplitR. { eauto using reducible_let_val. }
  intros_post. do 2 iModIntro. iMod "Hclose". iModIntro.
  apply invert_step_let_val in Hstep.
  destruct Hstep as (?&?&Heq&?); subst.
  iDestruct "Hi" as "(%&?)". iFrame.
  eauto using pureinv_let_val.
Qed.

Lemma val_inv_final_tree σ α G T e σ' α' T' e1 :
  step G σ α T e σ' α' T' e1 ->
  is_val e1 ->
  is_leaf T'.
Proof. induction 1; intros; try done; elim_ctx; inversion H; subst; done. Qed.

(* This is a somewhat odd precondition, but I need it for the induction to succeed.
   See [wpg_bind] for a simpler lemma. *)
Lemma wpg_bind_pre E T K e Q:
  (is_val e -> is_leaf T) ->
  wpg E T e (fun v => wpg E (Leaf (root T)) (fill_item K v) Q) -∗
  wpg E T (fill_item K e) Q.
Proof.
  iIntros (Ht) "Hwp".
  iLöb as "IH" forall (T e Ht).

  destruct (to_val e) eqn:Hvt.
  { apply to_val_Some_inv in Hvt. rewrite Hvt.
    rewrite wpg_val_eq. simpl.
    iApply fupd_wpg. iMod "Hwp" as "(%X&?)".
    destruct T; last done. done. }

  rewrite !wpg_unfold /wpg_pre.
  rewrite to_val_fill_item. wpg_intros. simpl.
  rewrite Hvt.

  iDestruct "Hi" as "(%Hcomp&Hi)".
  assert ((set_Forall (fun t => all_abef G α t (locs K)) (frontier T)) /\ pureinv G α σ T e) as (?&?)  by eauto using pureinv_bind.

  iMod ("Hwp" with "[Hi]") as "(%Hred&Hwp)".
  { by iFrame. }

  iModIntro.
  apply is_val_false in Hvt.
  iSplitR. { eauto using RedCtx. }
  intros_post.

  apply invert_step_fill_item in Hstep; last eauto.
  destruct Hstep as (e1,(?&?)). subst.
  mod_all "Hwp" "(%&?&Hwp)". iFrame.
  iModIntro. iSplitR.
  { iPureIntro. intros. apply pureinv_ctx; eauto.
    intros t HT'.
    destruct Hcomp.
    eapply all_abef_mon_amap; first (eapply step_inv_amap; eauto).
    eauto using step_inv_reach. }
  iApply "IH".
  { iPureIntro. eauto using val_inv_final_tree. }
  erewrite step_inv_root; try done.
Qed.

Lemma wpg_bind E t e K Q :
  wpg E (Leaf t) e (fun v => wpg E (Leaf t) (fill_item K v) Q) -∗
  wpg E (Leaf t) (fill_item K e) Q.
Proof. apply wpg_bind_pre. eauto. Qed.

Lemma wpg_fold E t v Q :
  wpg E (Leaf t) (VFold v) Q -∗
  wpg E (Leaf t) (Fold v) Q.
Proof.
  iIntros "HQ". rewrite !wpg_unfold /wpg_pre. simpl.
  wpg_intros; intros_mod.
  iSplitR. { eauto using reducible_fold. }
  intros_post. do 2 iModIntro. iMod "Hclose" as "_".
  apply invert_step_fold in Hstep.
  destruct Hstep as (->&->&->&->).
  iModIntro.
  iDestruct "Hi" as "(%Hpure&?)". rewrite wpg_unfold. iFrame.
  eauto using pureinv_fold.
Qed.

Lemma wpg_unfold_fold E t (v:val) Q :
  Q v -∗
  wpg E (Leaf t) (Unfold (VFold v)) Q.
Proof.
  iIntros "HQ". rewrite !wpg_unfold /wpg_pre. simpl.
  wpg_intros; intros_mod.
  iSplitR. { eauto using reducible_unfold_fold. }
  intros_post. do 2 iModIntro. iMod "Hclose" as "_".
  apply invert_step_unfold_fold in Hstep.
  destruct Hstep as (->&->&->&->).
  iModIntro.
  iDestruct "Hi" as "(%Hpure&?)". rewrite wpg_unfold. iFrame.
  iSplitR. eauto using pureinv_unfold_fold. done.
Qed.

(* We can open invariants around atomic operations. *)
Lemma wpg_atomic E1 E2 t e Q :
  dislog.lang.atomic.Atomic e ->
  (|={E1,E2}=> wpg E2 (Leaf t) e (fun v => |={E2,E1}=> Q v )) ⊢ wpg E1 (Leaf t) e Q.
Proof.
  iIntros (Hatomic) "Hwp". rewrite !wpg_unfold /wpg_pre.
  rewrite Atomic_no_val //. wpg_intros.
  iMod ("Hwp"  with "[$]") as ">(Hr&Hwp)". iModIntro.
  iFrame "Hr". intros_post.
  mod_all "Hwp" "(%&?&Hwp)".
  apply Atomic_correct in Hstep; eauto. destruct Hstep as (->&(?,->)).
  rewrite wpg_unfold /wpg_pre. simpl.
  iMod "Hwp" as "(%&?)".
  iFrame "%∗".
  iApply wpg_val. by iFrame.
Qed.
End wpg.

Global Instance wpg_ne `{dislogGS Σ} E T e n :
  Proper (pointwise_relation _ (dist n) ==> dist n) (wpg E T e).
Proof.
  revert e T. induction (lt_wf n) as [n _ IH]=> e T Φ Ψ HΦ.
  rewrite !wpg_unfold /wpg_pre /=.
  do 23 (f_contractive || f_equiv).
  rewrite IH //. intros ?. eapply dist_le; [apply HΦ|lia].
Qed.
