From iris.proofmode Require Import proofmode.

From dislog.logic Require Import wpg wpg_alloc wpg_call wpg_call_prim wpg_load wpg_store wpg_par wpg_cas wpg_length wpg_case wpg_proj wps.

From dislog.lang Require Export semantics.
From dislog.lang Require Export notation atomic.
From dislog.logic Require Export interp enc wpg_more all_abef.

(* ------------------------------------------------------------------------ *)
(* This file constructs the [wp] predicate over the [wpg]
   predicate. [wp] features:
   + A single timestamp parameter, rather than a task-tree.
   + A _typed_ post-condition, simplifying the reasoning.
 *)

Section wp.
Context `{interpGS he Σ}.

Definition wp `{Enc A} (E:coPset) (t:timestamp) (e:expr) (Q:A -> iProp Σ) : iProp Σ :=
  wpg E (Leaf t) e (fun v => post Q v).

Lemma wp_eq `{Enc A} E t e (Q:A -> iProp Σ) :
  wp E t e Q = wpg E (Leaf t) e (fun v => post Q v).
Proof. reflexivity. Qed.

Lemma wp_proper E t e P1 P2 :
  (forall v, P1 v ≡ P2 v) ->
  wp E t e P1 ≡ wp E t e P2.
Proof. eauto using wpg_proper, post_proper. Qed.

(* ------------------------------------------------------------------------ *)
(* Structural rules *)

Lemma fupd_wp (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  (|={E}=> wp E t e Q) -∗
  wp E t e Q.
Proof. apply fupd_wpg. Qed.

Lemma bupd_wp (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  (|==> wp E t e Q) -∗
  wp E t e Q.
Proof. iIntros "Hwp". iApply fupd_wp. by iMod "Hwp". Qed.

Lemma wp_fupd (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  wp E t e (fun v => |={E}=> Q v) -∗
  wp E t e Q.
Proof.
  iIntros. iApply wpg_fupd; iApply (wpg_mono with "[$]").
  iIntros. by iApply post_fupd.
Qed.

Lemma wp_strong_mono (A:Type) (EA:Enc A) E t e (P Q:A -> iProp Σ) :
  wp E t e P -∗
  (∀ v, P v ={E}=∗ Q v) -∗
  wp E t e Q.
Proof.
  iIntros "Hwp HPQ". iApply (wpg_strong_mono with "[$] [-]").
  iIntros. iApply (post_strong_mono _ _ E with "[HPQ][$]"). set_solver.
  iIntros. iApply "HPQ". iFrame.
Qed.

Lemma wp_strong_mono_val (A:Type) (EA:Enc A) E t e (P:A -> iProp Σ) (Q:val -> iProp Σ) :
  wp E t e P -∗
  (∀ v, P v ={E}=∗ Q (enc v)) -∗
  wp E t e Q.
Proof.
  iIntros "Hwp HPQ". iApply (wpg_strong_mono with "[$] [-]").
  iIntros (?). rewrite !post_eq.
  iIntros "[% (->&?)]". iExists _. iSplitR; first easy.
  iApply ("HPQ" with "[$]").
Qed.

Lemma wp_mono (A:Type) (EA:Enc A) E t e (P Q:A -> iProp Σ) :
  wp E t e P -∗
  (∀ v, P v -∗ Q v) -∗
  wp E t e Q.
Proof.
  iIntros "H1 H2".
  iApply (wp_strong_mono with "H1").
  iIntros. by iApply "H2".
Qed.

Lemma wp_mono_val (A:Type) (EA:Enc A) E t e (P:A -> iProp Σ) (Q:val -> iProp Σ)  :
  wp E t e P -∗
  (∀ v, P v -∗ Q (enc v)) -∗
  wp E t e Q.
Proof.
  iIntros "H1 H2".
  iApply (wp_strong_mono_val with "H1").
  iIntros. by iApply "H2".
Qed.

Lemma wp_conv `{@Convertible A EA B EB} E t e (Q:B -> iProp Σ) :
  wp E t e (fun (x:A) => Q (conv x)) -∗
  wp E t e Q.
Proof.
  iIntros. iApply (wpg_mono with "[$]").
  iIntros (?). iApply @post_convertible.
Qed.

Lemma wp_frame (A:Type) (EA:Enc A) E t e (P:iProp Σ) (Q:A -> iProp Σ) :
  P ∗ wp E t e Q -∗
  wp E t e (fun v => P ∗ Q v).
Proof.
  iIntros "(?&?)".
  iApply (wp_mono with "[$]").
  iIntros. iFrame.
Qed.

Lemma wp_frame_step (A:Type) (EA:Enc A) P E t e (Q:A -> iProp Σ) :
  ¬ is_val e ->
  ▷ P -∗
  wp E t e (fun v => P -∗ Q v) -∗
  wp E t e Q.
Proof.
  iIntros. iApply (wpg_mono with "[-]").
  iApply (wpg_frame_step P with "[$]"); eauto.
  iIntros (?). rewrite !post_eq. iIntros "([% (?&E)]&?)".
  iExists _. iFrame. iApply "E". done.
Qed.

(* ------------------------------------------------------------------------ *)
(* Timestamp-related rules *)

Lemma wp_mementopre l (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  ¬ is_val e ->
  l ∈ locs e ->
  (l ◷ t -∗ wp E t e Q) -∗
  wp E t e Q.
Proof. apply mementopre. Qed.

Lemma wp_vmementopre v (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  ¬ is_val e ->
  locs v ⊆ locs e ->
  (v ◷? t -∗ wp E t e Q) -∗
  wp E t e Q.
Proof. apply vmementopre. Qed.

Lemma wp_mementopost (A:Type) (EA:Enc A) E t e (Q:A -> iProp Σ) :
  ¬ is_val e ->
  wp E t e (fun v => (enc v) ◷? t -∗ Q v) -∗
  wp E t e Q.
Proof.
  iIntros. iApply mementopost. done. iApply (wpg_mono with "[$]").
  iIntros (?). rewrite !post_eq. iIntros "[%(->&HP)] #?".
  iExists _. iSplitR; first easy. by iApply "HP".
Qed.

Lemma wp_val (A:Type) (EA:Enc A) E t v v' (Q:A -> iProp Σ) :
  v = enc v' ->
  Q v' -∗ wp E t (Val v) Q.
Proof.
  iIntros (->) "HQ". iApply wpg_val.
  iExists _. by iFrame.
Qed.

(* ------------------------------------------------------------------------ *)
(* Syntax-directed rules *)

Lemma wp_if (A:Type) (EA:Enc A) E t (b:bool) (e1 e2:expr) (Q:A -> iProp Σ) :
  wp E t (if b then e1 else e2) Q -∗
  wp E t (If b e1 e2) Q.
Proof. iIntros. iApply  wpg_if. destruct b; iFrame. Qed.

Lemma wp_let_val (A:Type) (EA:Enc A) E t x (v:val) e (Q:A -> iProp Σ) :
  wp E t (subst' x v e) Q -∗
  wp E t (Let x v e) Q.
Proof. apply wpg_let_val. Qed.

Lemma wp_bind K (A:Type) (EA:Enc A) (B:Type) (EB:Enc B) E t (e:expr) (Q:A -> iProp Σ) :
  wp E t e (fun (v:B) => wp E t (fill_item K (enc v)) Q) -∗
  wp E t (fill_item K e) Q.
Proof.
  iIntros "Hwp".
  iApply wpg_bind.
  iApply (wpg_mono with "Hwp"). iIntros (?). rewrite post_eq.
  iIntros "[% (->&?)]". iFrame.
Qed.

Lemma wp_let `{interpGS he Σ} (A:Type) (EA:Enc A) (B:Type) (EB:Enc B) E (t:timestamp) x (e1 e2:expr) (Q: A -> iProp Σ) :
  wp E t e1 (fun (v:B) => wp E t (subst' x (enc v) e2) Q) -∗
  wp E t (Let x e1 e2) Q.
Proof.
  iIntros "Hwp".
  iApply (wp_bind (CtxLet x e2) _ _ B _ _ _ _ Q).
  iApply (wp_mono with "Hwp").
  iIntros. iApply wp_let_val. iFrame.
Qed.

Lemma wp_unfold_fold E t (v:val) Q :
  Q v -∗
  wp E t (Unfold (VFold v)) Q.
Proof.
  iIntros "Hwp".
  iApply wpg_unfold_fold. rewrite post_val //.
Qed.

Lemma wp_fold E t (v:val) Q :
  wp E t (VFold v) Q -∗
  wp E t (Fold v) Q.
Proof. iApply wpg_fold. Qed.

Lemma wp_par (A:Type) (EA:Enc A) (B:Type) (EB:Enc B) E t (v1 v2:val) Q:
  (∀ t1 t2, t ≼ t1 ∗ t ≼ t2 ={E}=∗ ∃ (Q1:A -> iProp Σ) (Q2:B -> iProp Σ),
     wp E t1 (Call v1 [Val VUnit]) Q1 ∗
     wp E t2 (Call v2 [Val VUnit]) Q2 ∗
     (∀ (v1:A) (v2:B) l,
         is_prod l (enc v1) (enc v2) ∗
         t1 ≼ t ∗ t2 ≼ t ∗
         Q1 v1 ∗ Q2 v2 ={E}=∗ (Q l)
     )) -∗
     wp E t (Par v1 v2) Q.
Proof.
  iIntros "Hwp".
  iApply wpg_fork. iModIntro. iIntros.
  iApply fupd_wpg. iMod ("Hwp" with "[$]") as "[%Q1 [%Q2 (H1&H2&Hwp)]]".
  iModIntro. iApply wpg_fupd.
  iApply (wpg_mono with "[H1 H2]").
  { iApply (wpg_par_ctx with "H1 H2"). }
  iIntros (?) "[% [% [% (->&?&?&?&E1&E2)]]]".
  iApply post_loc.
  rewrite !post_eq. iDestruct "E1" as "[% (->&?)]". iDestruct "E2" as "[% (->&?)]".
  iApply ("Hwp" with "[$]").
Qed.

Lemma wp_load (A:Type) (EA:Enc A) E t bs q (i:Z) (v:A) (l:loc) :
  (0 <= i < Z.of_nat (length bs))%Z ->
  bs !! (Z.to_nat i) = Some (enc v) ->
  l ↦{q} bs -∗ (enc v) ◷? t -∗
  wp E t (Load l i) (fun (w:A) => ⌜w=v⌝ ∗ l ↦{q} bs).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  { iApply (wpg_load with "[$]"); eauto. }
  iIntros (?) "(->&?)".
  iExists _. by iFrame.
Qed.

Lemma wp_call (A:Type) (EA:Enc A) E t self args body ts vs v (Q:A -> iProp Σ) :
  length args = length vs →
  ts = Val <$> vs ->
  func v self args body -∗
  ▷ (wp E t (substs' (zip (self::args) (v::vs)) body) Q) -∗
  wp E t (Call v ts) Q.
Proof. apply wpg_call. Qed.

Lemma wp_call_prim (A:Type) (EA:Enc A) E t (p:prim) v1 v2 (v:A) :
  eval_call_prim p v1 v2 = Some (enc v) ->
  ⊢ wp E t (CallPrim p v1 v2) (fun (v':A) => ⌜v'=v⌝)%I.
Proof.
  iIntros. iApply wpg_mono. iApply wpg_call_prim; eauto.
  iIntros (?) "->". iExists _. eauto.
Qed.

Lemma wp_alloc E t (i:Z) (v:val) :
  (0 < i)%Z ->
  ⊢ wp E t (Alloc i v)
      (fun (l:loc) => l ↦ (replicate (Z.to_nat i) v) ∗ meta_token l (⊤ ∖ ↑dislog_nm)).
Proof.
  iIntros.
  iApply wp_mono. iApply wpg_alloc. eauto.
  iIntros (?) "(?&?)"; subst. by iFrame.
Qed.

Lemma wp_prod E t (v1 v2:val) :
  ⊢ wp E t (Prod v1 v2)
      (fun (l:loc) => is_prod l v1 v2).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  { iApply wps_wpg. iApply wps_prod. }
  iIntros (?) "[% (->&?&_)]". rewrite post_loc //.
Qed.

Lemma wp_fst E t (l:loc) (v1 v2:val) :
  is_prod l v1 v2 -∗
  wp E t (Fst l) (fun v => ⌜v=v1⌝).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  { iApply wpg_fst. done. }
  iIntros (?) "->". rewrite post_val //.
Qed.

Lemma wp_snd E t (l:loc) (v1 v2:val) :
  is_prod l v1 v2 -∗
  wp E t (Snd l) (fun v => ⌜v=v2⌝).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  { iApply wpg_snd. done. }
  iIntros (?) "->". rewrite post_val //.
Qed.

Lemma wp_inj (b:bool) E t (v:val) :
  ⊢ wp E t (if b then InL v else InR v)
      (fun l => is_sum l b v).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  { iApply wps_wpg. iApply wps_inj. }
  iIntros (?) "[% (->&?&_)]". rewrite post_loc //.
Qed.

Lemma wp_case E t (l:loc) b (v:val) xl el xr er Q :
  is_sum l b v -∗
  wp E t (if b then subst' xl v el else subst' xr v er) Q -∗
  wp E t (Case l xl el xr er) Q.
Proof.
  iIntros. iApply (wpg_case with "[$]"). done.
Qed.

Lemma wp_closure E t self args code :
  ⊢ wp E t (Clo (Lam self args code))
      (fun (l:loc) =>  func l self args code).
Proof.
  iIntros.
  iApply wpg_mono.
  { iApply wps_wpg. iApply wps_closure. }
  iIntros (?) "[%l (%&?&?)]"; subst.
  iExists l. by iFrame.
Qed.

Lemma wp_store E t bs (l:loc) (i:Z) (v:val) :
  (0 <= i < (Z.of_nat (length bs)))%Z ->
  l ↦ bs -∗
  wp E t (Store l i v) (fun (_:unit) => l ↦ (<[Z.to_nat i:=v]> bs)).
Proof.
  iIntros. iApply (wpg_mono with "[-]"). iApply wpg_store; eauto.
  iIntros (?) "(%&?)". subst. iExists tt. by iFrame.
Qed.

Lemma wp_length E t bs (l:loc) p :
  l ↦{p} bs -∗
  wp E t (Length l) (fun (i:Z) => ⌜i = length bs⌝ ∗ l ↦{p} bs).
Proof.
  iIntros. iApply (wpg_mono with "[-]"). iApply wpg_length; eauto.
  iIntros (?) "(->&?)". iExists _. by iFrame.
Qed.

Lemma wp_cas E t (l:loc) (i:Z) (v1 v1' v2:val) bs q :
  (v1=v1' -> q=DfracOwn 1%Qp) ->
  (0 ≤ i < length bs)%Z ->
  bs !! Z.to_nat i = Some v1' ->
  l ↦{q} bs ∗ v1' ◷? t -∗
  wp E t (CAS l i v1 v2)
    (fun (b:bool) => ⌜b=bool_decide (v1=v1')⌝ ∗
       l ↦{q} (if decide (v1=v1') then <[Z.to_nat i:=v2]> bs else bs)).
Proof.
  iIntros.
  iApply (wpg_mono with "[-]").
  iApply wpg_cas; eauto.
  iIntros (?) "(->&?)". iApply post_bool. eauto.
Qed.

Lemma wp_atomic (A:Type) (EA:Enc A) E1 E2 t e (Q:A -> iProp Σ) :
  Atomic e ->
  (|={E1,E2}=> wp E2 t e (fun v => |={E2,E1}=> Q  v )) ⊢ wp E1 t e Q.
Proof.
  iIntros.
  iApply wpg_atomic; eauto.
  iApply (fupd_mono with "[$]").
  iIntros. iApply (wpg_mono with "[$]"). iIntros.
  by iApply post_fupd.
Qed.
End wp.

(* ------------------------------------------------------------------------ *)
(* Proofmode instances *)

Global Instance is_except_0_wp `{!interpGS he Σ} A (EA:Enc A) E t e (Q:A -> iProp Σ) :
  IsExcept0 (wp E t e Q).
Proof.
  rewrite /IsExcept0. iIntros. iApply fupd_wp.
  rewrite -except_0_fupd -fupd_intro //.
Qed.

Global Instance elim_modal_bupd_wp `{!interpGS he Σ} (A:Type) (EA:Enc A) E t e p P (Q:A -> iProp Σ) :
  ElimModal True p false (|==> P) P (wp E t e Q) (wp E t e Q).
Proof.
  rewrite /ElimModal bi.intuitionistically_if_elim
    (bupd_fupd E) fupd_frame_r bi.wand_elim_r.
  iIntros. by iApply fupd_wp.
Qed.

Global Instance elim_modal_fupd_wp `{!interpGS he Σ} (A:Type) (EA:Enc A) E t e p P (Q:A -> iProp Σ) :
  ElimModal True p false
    (|={E}=> P) P
    (wp E t e Q) (wp E t e Q)%I.
Proof.
  rewrite /ElimModal bi.intuitionistically_if_elim fupd_frame_r bi.wand_elim_r.
  iIntros. by iApply fupd_wp.
Qed.

Global Instance elim_modal_fupd_wp_atomic `{!interpGS he Σ} (A:Type) (EA:Enc A) E1 E2 t e p P (Q:A -> iProp Σ) :
  ElimModal (Atomic e) p false
    (|={E1,E2}=> P) P
    (wp E1 t e Q) (wp E2 t e (fun v => |={E2,E1}=> Q v))%I | 100.
Proof.
  intros ?.
  rewrite bi.intuitionistically_if_elim
    fupd_frame_r bi.wand_elim_r.
  iIntros. iApply (wp_atomic with "[$]"). eauto.
Qed.

Global Instance add_modal_fupd_wp `{!interpGS he Σ} (A:Type) (EA:Enc A) E t e P (Q:A -> iProp Σ) :
  AddModal (|={E}=> P) P (wp E t e Q).
Proof. rewrite /AddModal fupd_frame_r bi.wand_elim_r. iIntros. by iApply fupd_wp. Qed.

Global Instance elim_acc_wp_atomic `{!interpGS he Σ} {X} (A:Type) (EA:Enc A) E1 E2 t α clock γ e (Q:A -> iProp Σ) :
  ElimAcc (X:=X) (Atomic e)
    (fupd E1 E2) (fupd E2 E1)
    α clock γ (wp E1 t e Q)
    (λ x, wp E2 t e (fun v => |={E2}=> clock x ∗ (γ x -∗? Q v)))%I | 100.
Proof.
  iIntros (?) "Hinner >Hacc". iDestruct "Hacc" as (x) "[Hα Hclose]".
  iSpecialize ("Hinner" with "[$]").
  iApply (wp_mono with "[$]").
  iIntros (v) ">[Hclock HΦ]". iApply "HΦ". by iApply "Hclose".
Qed.

Global Instance elim_acc_wp_nonatomic `{!interpGS he Σ} {X} (A:Type) (EA:Enc A) E t α clock γ e (Q:A -> iProp Σ) :
  ElimAcc (X:=X) True (fupd E E) (fupd E E)
    α clock γ (wp E t e Q)
    (λ x, wp E t e (fun v => |={E}=> clock x ∗ (γ x -∗? Q v)))%I .
Proof.
  iIntros (_) "Hinner >Hacc". iDestruct "Hacc" as (x) "[Hα Hclose]".
  iApply wp_fupd.
  iSpecialize ("Hinner" with "[$]").
  iApply (wp_mono with "[$]").
  iIntros (v) ">[Hclock HΦ]". iApply "HΦ". by iApply "Hclose".
Qed.

Global Instance wp_ne `{!interpGS he Σ} E t e n :
  Proper (pointwise_relation _ (dist n) ==> dist n) (wp E t e).
Proof. intros ???. eapply wpg_ne. rewrite !post_eq. repeat f_equiv. Qed.

Global Opaque func is_prod is_sum.
Global Opaque wp.
