From iris.proofmode Require Import base proofmode.
From iris.base_logic.lib Require Import fancy_updates gen_heap.
From iris.program_logic Require Import weakestpre.
From iris.algebra Require Import gset gmap agree frac dfrac.

From dislog.utils Require Import graph more_iris more_stdpp.
From dislog.lang Require Import syntax.
From dislog.newlang Require Import semantics.
From dislog.logic Require Import wpg.

(******************************************************************************)
(* This file defines [interp], and related [clock] and  [prec]. *)

Class interpGS (has_escape : bool) (Σ : gFunctors) :=
  InterpG {
    iinvgs : invGS_gen HasNoLc Σ; (* invariants *)
    istore : gen_heapGS loc storable Σ; (* the store *)
    igraph : inG Σ (authUR (gsetUR (timestamp * timestamp))); (* the graph *)
    γgraph : gname; (* A name for the ghost cell of the graph *)
    }.

#[export] Existing Instance istore.
#[export] Existing Instance igraph.

Ltac destruct_interp Hi :=
  iDestruct Hi as "(#Hκs & Hσ & #Hα & HG)".

Section Interp.

Context `{interpGS has_escape Σ}.
Implicit Type t:timestamp.

(******************************************************************************)
(* [interp] *)

Definition interp_graph (G:graph) : iProp Σ := own γgraph (●□ G).

Definition dislog_nm : namespace := nroot.@"dislog".
Definition allocated_at (l:loc) (t:timestamp) : iProp Σ := meta l dislog_nm t.
Definition interp_amap α : iProp Σ := [∗ map] l ↦ t ∈ α, allocated_at l t.

(******************************************************************************)
(* [prec1] & [prec] *)

(* [prec1] would suffice by itself, but it is not reflexive nor transitive
   without access to the [interp] predicate.
   We close over using the [prec] predicate. *)

Definition prec1 (t1 t2:timestamp) : iProp Σ :=
  own γgraph (◯ {[(t1,t2)]}).

Lemma prec1_exploit G t1 t2 :
  interp_graph G -∗ prec1 t1 t2 -∗ ⌜reachable G t1 t2⌝.
Proof.
  iIntros "Ha Hf".
  iDestruct (auth_gset_elem_of with "Ha Hf") as "%".
  iPureIntro. by apply rtc_once.
Qed.

Fixpoint prec_aux (t1:timestamp) xs (t2:timestamp) : iProp Σ :=
  match xs with
  | nil => ⌜t1=t2⌝
  | x::xs => prec1 t1 x ∗ prec_aux x xs t2 end.

Definition prec (t1 t2:timestamp) : iProp Σ :=
  ∃ (xs:list timestamp), prec_aux t1 xs t2.

(* We reuse the notation ≼ that is already present for cmras,
   but only in bi_scope. *)
Local Notation "t1 ≼ t2" := (prec t1 t2)
  (at level 70, format "t1  ≼  t2") : bi_scope.

Global Instance prec_aux_timeless xs t1 t2 : Timeless (prec_aux t1 xs t2).
Proof. revert t1. induction xs; apply _. Qed.
Global Instance prec_aux_persistent xs t1 t2 : Persistent (prec_aux t1 xs t2).
Proof. revert t1. induction xs; apply _. Qed.
Global Instance prec_timeless t1 t2 : Timeless (t1 ≼ t2).
Proof. apply _. Qed.
Global Instance prec_persistent t1 t2 : Persistent (t1 ≼ t2).
Proof. apply _. Qed.

Lemma prec_refl t : ⊢ t ≼ t.
Proof. iExists nil. easy. Qed.

Lemma prec_once t1 t2 : prec1 t1 t2 -∗ t1 ≼ t2.
Proof. iIntros. iExists [t2]. simpl. eauto. Qed.

Lemma prec_trans t1 t2 t3 : t1 ≼ t2 -∗ t2 ≼ t3 -∗ t1 ≼ t3.
Proof.
  iIntros "[%xs Hxs] [%ys Hys]".
  iExists (xs++ys).
  iInduction xs as [|] "IH" forall (t1 ys).
  { iDestruct "Hxs" as "%". subst. iFrame. }
  { iDestruct "Hxs" as "(?&?)".
    iFrame. iApply ("IH" with "[$][$]"). }
Qed.

Lemma prec_exploit G t1 t2 :
  interp_graph G -∗ t1 ≼ t2 -∗ ⌜reachable G t1 t2⌝.
Proof.
  iIntros "? [%xs Hxs]".
  iInduction xs as [|] "IH" forall (t1).
  { iDestruct "Hxs" as "%". subst. easy. }
  { iDestruct "Hxs" as "(?&?)".
    iDestruct ("IH" with "[$][$]") as "%".
    iDestruct (prec1_exploit with "[$][$]") as "%".
    iPureIntro. etrans; eauto. }
Qed.

(******************************************************************************)
(* ◷ *)

Definition clock (l:loc) (t:timestamp) : iProp Σ :=
  ∃ t0, allocated_at l t0 ∗ t0 ≼ t.

Local Notation "l ◷ t" := (clock l t)
  (at level 20, format "l  ◷  t") : bi_scope.

Global Instance clock_timeless l t : Timeless (l ◷ t).
Proof. apply _. Qed.
Global Instance clock_persistent l t : Persistent (l ◷ t).
Proof. apply _. Qed.

Lemma clock_mon l t t' :
  l ◷ t -∗ t ≼ t' -∗ l ◷ t'.
Proof.
  iIntros "[%u0 (?&H1)] H2". iExists u0. iFrame.
  iApply (prec_trans with "H1 H2").
Qed.

Fixpoint vclock (v:val) (t:timestamp) : iProp Σ :=
  match v with
  | VLoc l => l ◷ t
  | VFold v => vclock v t
  | _ => True end.

Local Notation "v ◷? t" := (vclock v t)
  (at level 20, format "v  ◷?  t") : bi_scope.

Lemma vclock_mon v t t' :
  v ◷? t -∗ t ≼ t' -∗ v ◷? t'.
Proof. induction v; eauto using clock_mon. Qed.

Global Instance vclock_timeless v t : Timeless (v ◷? t).
Proof. induction v; apply _. Qed.
Global Instance vclock_persistent v t : Persistent (v ◷? t).
Proof. induction v; apply _. Qed.

Definition func v self args code : iProp Σ :=
  match v with
  | VCode c => ⌜c=Lam self args code /\ locs code = ∅⌝
  | VLoc l => pointsto l DfracDiscarded (SClo self args code) ∗
                  ∃ t, allocated_at l t ∗ ([∗ set] l' ∈ locs code, l' ◷ t)
  | _ => False end.
Global Instance timeless_func v self args code : Timeless (func v self args code).
Proof. destruct v; apply _. Qed.
Global Instance persistent_func v self args code : Persistent (func v self args code).
Proof. destruct v; apply _. Qed.

Lemma func_eq v self args code :
  func v self args code =
  match v with
  | VCode c => ⌜c=Lam self args code /\ locs code = ∅⌝
  | VLoc l => pointsto l DfracDiscarded (SClo self args code) ∗
                  ∃ t, allocated_at l t ∗ ([∗ set] l' ∈ locs code, l' ◷ t)
  | _ => False end%I.
Proof. done. Qed.

Definition is_prod (l:loc) (v1 v2:val) : iProp Σ :=
  pointsto l DfracDiscarded (SProd v1 v2) ∗
  ∃ t, allocated_at l t ∗ vclock v1 t ∗ vclock v2 t.
Global Instance timeless_prod l v1 v2 : Timeless (is_prod l v1 v2).
Proof. apply _. Qed.
Global Instance persistent_prod l v1 v2 : Persistent (is_prod l v1 v2).
Proof. apply _. Qed.

Definition is_sum (l:loc) (b:bool) (v:val) : iProp Σ :=
  pointsto l DfracDiscarded (if b then SInL v else SInR v) ∗
  ∃ t, allocated_at l t ∗ vclock v t.
Global Instance timeless_is_sum l b v : Timeless (is_sum l b v).
Proof. apply _. Qed.
Global Instance persistent_is_sum l b v : Persistent (is_sum l b v).
Proof. apply _. Qed.

(******************************************************************************)
(* interp itself *)

Definition is_immut s :=
  match s with
  | SBlock _ => false
  | _ => true end.

Definition interp_immut_allocs (σ:store) : iProp Σ :=
  [∗ map] l ↦ v ∈ σ, if decide (is_immut v) then  pointsto l DfracDiscarded v else True.

Definition pointsto_in_dom (g:gset loc) v :=
  match v with
  | SProd v1 v2 => locs v1 ∪ locs v2 ⊆ g
  | SInL v | SInR v => locs v ⊆ g
  | _ => True end.

Definition is_immut_pointsto_in_dom g (σ:store) :=
  map_Forall (fun _ => pointsto_in_dom g) σ.

Global Instance interp_immut_allocs_persist σ : Persistent (interp_immut_allocs σ).
Proof.
  apply big_sepM_persistent'. intros ? []; apply _.
Qed.

Definition interp (σ:store) (α:amap) (G:graph) : iProp Σ :=
  interp_immut_allocs σ ∗
  gen_heap_interp σ ∗
  interp_amap α ∗
  interp_graph G
.

Lemma interp_prec_exploit G t1 t2 σ α :
  interp σ α G -∗ t1 ≼ t2 -∗ ⌜reachable G t1 t2⌝.
Proof.
  iIntros "(?&?&?&?) ?". iApply (prec_exploit with "[$][$]").
Qed.

Lemma meta_in_dom `{Countable A} σ (l : loc) (N : namespace) (x : A) :
  meta l N x -∗ gen_heap_interp  σ -∗ ⌜l ∈ dom σ⌝.
Proof.
  rewrite gen_heap.meta_unseal. iIntros "[% (?&?)]".
  iIntros "[% (%&?&?)]".
  iDestruct (ghost_map.ghost_map_lookup with "[$][$]") as "%X".
  iPureIntro. apply elem_of_dom_2 in X. set_solver.
Qed.

Lemma interp_exploit_clock σ α G l t :
  dom σ = dom α ->
  interp σ α G ∗ l ◷ t -∗ ⌜abef G α t l⌝.
Proof.
  iIntros (Hdom) "(Hi&[% (?&?)])". destruct_interp "Hi".
  iDestruct (meta_in_dom with "[$][$]") as "%Hl".
  rewrite Hdom in Hl. apply elem_of_dom in Hl. destruct Hl as (u0,Hl).
  iDestruct (big_sepM_lookup with "Hα") as "Hx". eauto.
  iDestruct (meta_agree with "[$][$]") as "->".
  iDestruct (prec_exploit with "[$][$]") as  "%".
  iPureIntro. rewrite /abef Hl //.
Qed.

Lemma interp_exploit_vclock σ G α t v :
  dom σ = dom α ->
  interp σ α G ∗ v ◷? t -∗ ⌜vabef G α t v⌝.
Proof.
  iIntros (?) "(Hi&?)".
  destruct_decide (decide (is_loc v)) as Hv.
  { induction v; eauto. iApply interp_exploit_clock; eauto. }
  { eauto using vabef_no_loc. }
Qed.

Lemma interp_exploit_pointsto σ α G l q bs :
  interp σ α G ∗ pointsto l q bs -∗ ⌜σ !! l = Some bs⌝.
Proof.
  iIntros "(Hi&?)". destruct_interp "Hi".
  iApply (gen_heap.gen_heap_valid with "[$][$]").
Qed.

(******************************************************************************)
(* Various update of interp *)

Lemma interp_get_immut_pre l s σ :
  is_immut s ->
  σ !! l = Some s ->
  interp_immut_allocs σ -∗
  pointsto l DfracDiscarded s .
Proof.
  iIntros.
  iDestruct (big_sepM_lookup with "[$]") as "?". 1:done.
  by destruct s.
Qed.

Lemma interp_get_immut l t s σ α G :
  is_immut s ->
  σ !! l = Some s ->
  α !! l = Some t ->
  interp σ α G -∗
  interp σ α G ∗ allocated_at l t ∗ pointsto l DfracDiscarded s.
Proof.
  iIntros (???) "Hi". destruct_interp "Hi".
  iDestruct (interp_get_immut_pre with "[$]") as "#?"; try done.
  iDestruct (big_sepM_lookup with "Hα") as "#?". done.
  by iFrame "#∗".
Qed.

Lemma is_immut_pointsto_in_dom_mon g1 g2 σ :
  g1 ⊆ g2 ->
  is_immut_pointsto_in_dom g1 σ ->
  is_immut_pointsto_in_dom g2 σ.
Proof.
  intros. eapply map_Forall_impl. done.
  intros ? []; simpl; set_solver.
Qed.

Lemma is_immut_pointsto_insert_not_immut g l b σ :
  ¬ is_immut b ->
  is_immut_pointsto_in_dom g σ ->
  is_immut_pointsto_in_dom g (<[l:=b]>σ).
Proof.
  intros. intros l' s.
  rewrite more_stdpp.lookup_insert_case.
  case_decide; last eauto.
  { inversion 1. subst. by destruct s. }
Qed.

Lemma interp_alloc σ α G l b t :
  ¬ is_immut b ->
  l ∉ dom σ ->
  dom σ = dom α ->
  interp σ α G ==∗ allocated_at l t ∗ interp (<[l:=b]> σ) (<[l:=t]> α) G ∗ pointsto l (DfracOwn 1) b ∗ meta_token l (⊤∖↑dislog_nm).
Proof.
  iIntros (Hb ? Hdom) "Hi".
  destruct_interp "Hi".
  iMod (gen_heap_alloc _ l b with "[$]") as "(?&?&Ht)".
  { apply not_elem_of_dom. eauto. }
  iDestruct (meta_token_difference l with "Ht") as "(?&?)"; last iFrame.
  { set_solver. }
  iMod (meta_set with "[$]") as "#?"; last iFrame "#".
  { set_solver. }
  iModIntro.
  iSplitL "Hκs".
  { iApply big_sepM_insert; eauto. apply not_elem_of_dom. eauto.
    rewrite decide_False //. iFrame "#". }
  { iApply big_sepM_insert.
    { apply not_elem_of_dom; set_solver. }
    by iFrame "#". }
Qed.

Local Notation "l ↦{ dq } v" := (pointsto l dq (SBlock v))
  (at level 20, format "l  ↦{ dq }  v") : bi_scope.
Local Notation "l ↦{# dq } v" := (pointsto l (DfracOwn dq) (SBlock v))
  (at level 20, format "l  ↦{# dq }  v") : bi_scope.
Local Notation "l ↦ v" := (pointsto l (DfracOwn 1) (SBlock v))
                            (at level 20, format "l  ↦  v") : bi_scope.


Lemma interp_store σ α G bs (l:loc) (i:Z) (v:val) :
  interp σ α G ∗ l ↦ bs ==∗
  interp (<[l:=SBlock (<[Z.to_nat i:=v]> bs)]> σ) α G ∗ l ↦ (<[Z.to_nat i:=v]> bs).
Proof.
  iIntros "(Hi&?)".
  destruct_interp "Hi". iFrame "#∗".
  iDestruct (gen_heap_valid with "[$][$]") as "%".
  rewrite -assoc.
  iSplitR.
  { rewrite -insert_delete_insert.
    rewrite /interp_immut_allocs. rewrite big_sepM_delete //.
    iDestruct "Hκs" as "(?&?)".
    iApply big_sepM_insert. rewrite lookup_delete //. by iFrame "#". }
  iMod (gen_heap.gen_heap_update with "[$][$]") as "(?&?)". by iFrame.
Qed.

Lemma alloc_prec' t t' G:
  reachable G t t' ->
  interp_graph G ==∗ interp_graph G ∗ t ≼ t'.
Proof.
  iIntros (Hr) "Ha".
  iInduction Hr as [|] "IH".
  { iFrame. by iApply prec_refl. }
  iMod (auth_gset_extract_witness_elem (x,y) with "[$]") as "(?&?)".
  set_solver.
  iMod ("IH" with "[$]") as "(?&?)". iFrame.
  iDestruct (prec_once with "[$]") as "?".
  by iApply (prec_trans x y z with "[$][$]").
Qed.

Lemma interp_insert_abef σ α G l t :
  abef G α t l ->
  interp σ α G ==∗ interp σ α G ∗ l ◷ t.
Proof.
  iIntros (Habef) "Hi". destruct_interp "Hi".
  rewrite /abef in Habef. destruct (α !! l) eqn:Hl; last easy.
  iDestruct (big_sepM_lookup with "[$]") as "#?"; first eauto.
  iMod (alloc_prec' with "[$]") as "(?&#?)". eauto.
  by iFrame "#∗".
Qed.

Lemma alloc_prec t t' σ α G  :
  reachable G t t' ->
  interp σ α G ==∗ interp σ α G ∗ t ≼ t'.
Proof.
  iIntros (?) "Hi". destruct_interp "Hi".
  iMod (alloc_prec' with "[$]") as "(?&?)". eauto. by iFrame "#∗".
Qed.

Lemma alloc_amap (σ:store) (α:amap) :
  dom σ = dom α ->
  ([∗ map] l↦_ ∈ σ, meta_token l ⊤)%I ==∗
  interp_amap α.
Proof.
  iIntros (Hdom) "Hm".
  iInduction σ as [|] "IH" using map_ind forall (α Hdom).
  { rewrite dom_empty_L in Hdom. symmetry in Hdom. apply dom_empty_inv_L in Hdom. subst.
    by iApply big_sepM_empty. }
  { rewrite dom_insert_L in Hdom. assert (i ∈ dom α) as Hi by set_solver.
    rewrite big_sepM_insert_delete. iDestruct "Hm" as "(?&?)".
    apply elem_of_dom in Hi. destruct Hi as (y,?).
    rewrite -(insert_delete α i y) //. iApply big_sepM_insert.
    { rewrite lookup_delete //. }
    iMod (meta_set with "[$]"); last iFrame. set_solver.
    rewrite delete_notin //. apply not_elem_of_dom in H0.
    iApply ("IH" with "[%][$]"). rewrite dom_delete_L. set_solver. }
Qed.

End Interp.

Global Notation "l ↦{ dq } v" := (pointsto l dq (SBlock v))
  (at level 20, format "l  ↦{ dq }  v") : bi_scope.
Global Notation "l ↦{# dq } v" := (pointsto l (DfracOwn dq) (SBlock v))
  (at level 20, format "l  ↦{# dq }  v") : bi_scope.
Global Notation "l ↦ v" := (pointsto l (DfracOwn 1) (SBlock v))
  (at level 20, format "l  ↦  v") : bi_scope.

Global Notation "t1 ≼ t2" := (prec t1 t2)
  (at level 70, format "t1  ≼  t2") : bi_scope.

Global Notation "l ◷ t" := (clock l t)
  (at level 20, format "l  ◷  t") : bi_scope.
Global Notation "v ◷? t" := (vclock v t)
  (at level 20, format "v  ◷?  t") : bi_scope.

Global Instance interpGS_DisLogGS `{interpGS x Σ} : dislogGS x Σ.
Proof.
  exact (DislogGS x iinvgs interp).
Defined.

Lemma elem_of_dom_filter `{Countable K} {V} `{(∀ x:K*V, Decision (P x))} (m:gmap K V) l v :
  m !! l = Some v ->
  P (l,v) ->
  l ∈ dom (filter P m).
Proof.
  intros Hl HP.
  apply elem_of_dom. exists v. rewrite map_lookup_filter Hl.
  simpl. rewrite option_guard_True //.
Qed.

Module Initialization.

  Definition interpΣ : gFunctors :=
    #[ invΣ;
       gen_heapΣ loc storable;
       GFunctor (authUR (gsetUR (timestamp * timestamp)));
       GFunctor (authUR (gsetUR loc))
      ].

  (* The difference between the *PreG and *G is the presence of the names
     of ghost cells. (ie. gname) *)
  Class interpPreG (Σ : gFunctors) :=
  { piinvgs : invGpreS Σ;
    pistore : gen_heapGpreS loc storable Σ;
    pigraph : inG Σ (authUR (gsetUR (timestamp * timestamp)));
    pidom : inG Σ (authUR (gsetUR loc));
  }.

  #[global] Existing Instance piinvgs.
  #[global] Existing Instance pistore.
  #[global] Existing Instance pigraph.
  #[global] Existing Instance pidom.

  Global Instance subG_interpPreG Σ :
    subG interpΣ Σ → interpPreG Σ.
  Proof. solve_inG. Qed.

  Global Instance interpPreG_interpΣ : interpPreG interpΣ.
  Proof. eauto with typeclass_instances. Qed.

  Lemma interp_init `{!interpPreG Σ, hinv:!invGS_gen HasNoLc Σ} x σ α G :
    dom σ = dom α ->
    ⊢ |==> ∃ hi : interpGS x Σ, ⌜@iinvgs x Σ hi = hinv⌝ ∗
    interp σ α G.
  Proof.
    iIntros (Hdom). rewrite /interp.
    iMod (gen_heap_init σ) as "[% (?&?&Hn)]".

    iMod (own_alloc (● G)) as "[%γgraph X']".
    { by apply auth_auth_valid. }
    iMod (own_update with "X'") as "X'".
    { apply auth_update_auth_persist. }

    pose (HΣ := @InterpG x Σ hinv _ _ γgraph).
    iExists HΣ.
    iSplitR; try easy. iFrame.

    iMod (@alloc_amap x Σ HΣ σ α with "Hn") as "#Hα". done.
    iFrame "#".

    iAssert (|==> [∗ map] l↦v ∈ σ, pointsto l DfracDiscarded v)%I  with "[-]" as ">#Hσ".
    { iApply big_sepM_bupd. iApply big_sepM_mono; last done.
      iIntros. iApply (pointsto_persist with "[$]"). }

    iModIntro. iFrame "#".
    clear G. iApply big_sepM_intro.
    iModIntro. iIntros (l s E1).
    destruct_decide (decide (is_immut s)); last done.

    iDestruct (big_sepM_lookup with "Hσ") as "?"; first done.
    iFrame "#".
  Qed.


End Initialization.

Module FullInitialization.
  Export Initialization.
  (* This cannot be global as we want to keep Σ as a parameter:
     we thus do _not_ want coq to use interpΣ *)

  #[export] Instance interpGS_interpΣ x : interpGS x interpΣ.
  Proof.
    constructor; eauto with typeclass_instances.
    3:exact xH.
    { constructor.
      { constructor; eauto with typeclass_instances. all:exact xH. }
      { (* Cannot be solved by solve_inG. *)
        constructor. rewrite /interpΣ /invΣ /lcΣ.
        { now apply InG with (inG_id:=3%fin). }
        all:exact xH. } }
    repeat (constructor; eauto with typeclass_instances).
  Qed.
End FullInitialization.
