From Coq.ssr Require Import ssreflect.
From stdpp Require Import strings binders gmap gmultiset.

From dislog.utils Require Import graph.
From dislog.lang Require Export head_semantics.

(******************************************************************************)
(* Definition of a task_tree and operations on it. *)

Inductive task_tree (A:Type) : Type :=
| Leaf : A -> task_tree A
| Node : A -> task_tree A -> task_tree A -> task_tree A.

Arguments Leaf {A}%_type_scope.
Arguments Node {A}%_type_scope.

Fixpoint leaves `{Countable A} (T:task_tree A) : gset A :=
  match T with
  | Leaf t => {[t]}
  | Node t T1 T2 => {[t]} ∪ leaves T1 ∪ leaves T2 end.

Lemma leaves_node `{Countable A} (t:A) T1 T2 :
  leaves (Node t T1 T2) = {[t]} ∪ leaves T1 ∪ leaves T2.
Proof. done. Qed.

Fixpoint frontier `{Countable A} (T:task_tree A) : gset A :=
  match T with
  | Leaf x => {[x]}
  | Node _ T1 T2 => frontier T1 ∪ frontier T2 end.

Lemma frontier_subset_leaves `{Countable A} (T:task_tree A) :
  frontier T ⊆ leaves T.
Proof.
  induction T; set_solver.
Qed.

Definition is_leaf `{Countable A} (T:task_tree A) := match T with | Leaf _ => true | _ => false end.

Global Instance task_tree_inhabited `{Inhabited A} : Inhabited (task_tree A) := populate (Leaf inhabitant).

(******************************************************************************)
(* [sched_step] either a head step, fork or join *)

Inductive sched_step `{Countable A} :
  store -> gmap loc A -> graph A -> task_tree A -> expr ->
  store -> gmap loc A -> graph A -> task_tree A -> expr -> Prop :=
| SchedHead : forall σ α G t e σ' α' e',
    head_step G t σ α e σ' α' e' ->
    sched_step σ α G (Leaf t) e σ' α' G (Leaf t) e'
| SchedFork : forall σ α G G' t0 t1 t2 (v1 v2:val),
    t1 ∉ vertices G ->
    t2 ∉ vertices G ->
    t1 ≠ t2 ->
    G' = graph_fork G t0 t1 t2 ->
    sched_step
      σ α G (Leaf t0) (Par v1 v2)
      σ α G' (Node t0 (Leaf t1) (Leaf t2)) (RunPar (Call v1 [Val VUnit]) (Call v2 [Val VUnit]))
| SchedJoin : forall σ α G G' t1 t2 t3 (v1 v2:val) σ' α' (l:loc),
    G' = graph_join G t1 t2 t3 ->
    l ∉ dom σ ->
    l ∉ dom α ->
    σ' = <[l:=SProd v1 v2]> σ ->
    α' = <[l:=t3]> α ->
    sched_step
      σ  α  G (Node t3 (Leaf t1) (Leaf t2)) (RunPar v1 v2)
      σ' α' G' (Leaf t3) l.
#[export] Hint Constructors sched_step : sched_step.

Lemma sched_step_no_val `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  sched_step σ α G T e σ' α' G' T' e' -> ¬ is_val e.
Proof. destruct 1; eauto using head_step_no_val. Qed.

Lemma sched_step_no_ctx `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' K :
  ¬ is_val e → ¬ sched_step σ α G T (fill_item K e) σ' α' G' T' e'.
Proof.
  intros ? Hr. inversion Hr; subst.
  { eapply head_step_no_ctx; eauto. }
  all:destruct K; naive_solver.
Qed.

Lemma sched_step_inv_graph `{Countable A} σ α (G:graph A) T2 e2 σ' α' G' T2' e2' :
  sched_step σ α G T2 e2 σ' α' G' T2' e2' ->
  G ⊆ G'.
Proof.
  destruct 1; subst; eauto; [apply graph_fork_incl | apply graph_join_incl].
Qed.

Lemma sched_step_inv_amap `{Countable A} σ α (G:graph A) T e σ' α' G' T' e':
  dom σ = dom α ->
  sched_step σ α G T e σ' α' G' T' e' ->
  α ⊆ α'.
Proof.
  intros ?.
  destruct 1; subst; eauto using head_step_inv_amap.
  apply insert_subseteq. apply not_elem_of_dom. eauto.
Qed.

(******************************************************************************)
(* Distant semantics. *)

Inductive step `{Countable A} :
  store -> gmap loc A -> graph A -> task_tree A -> expr ->
  store -> gmap loc A -> graph A -> task_tree A -> expr -> Prop :=
| StepHead : forall σ α G T2 e2 σ' α' G' T2' e2',
    sched_step σ α G T2 e2 σ' α' G' T2' e2' ->
    step σ α G T2 e2 σ' α' G' T2' e2'
| StepBind : forall σ α G T e σ' α' G' T' e' K,
    step σ α G T e σ' α' G' T' e' ->
    step σ α G T (fill_item K e) σ' α' G' T' (fill_item K e')
| StepParL : forall σ α G G' e1 e2 T1 T2 σ' α' T1' e1' t,
    step σ α G T1 e1 σ' α' G' T1' e1' ->
    step σ α G (Node t T1 T2) (RunPar e1 e2) σ' α' G' (Node t T1' T2) (RunPar e1' e2)
| StepParR : forall σ α G G' e1 e2 T1 T2 σ' α' T2' e2' t,
    step σ α G T2 e2 σ' α' G' T2' e2' ->
    step σ α G (Node t T1 T2) (RunPar e1 e2) σ' α' G' (Node t T1 T2') (RunPar e1 e2')
.
#[export] Hint Constructors step : step.

Lemma step_no_val `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  step σ α G T e σ' α' G' T' e' ->
  ¬ is_val e.
Proof.
  inversion 1; subst; eauto using sched_step_no_val.
  destruct K; eauto.
Qed.

Lemma step_inv_graph `{Countable A} σ α (G:graph A) T e σ' α' G' T' e'  :
  step σ α G T e σ' α' G' T' e' ->
  G ⊆ G'.
Proof.
  induction 1; subst; eauto using sched_step_inv_graph.
Qed.

Lemma step_inv_amap `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  dom σ = dom α ->
  step σ α G T e σ' α' G' T' e' ->
  α ⊆ α'.
Proof.
  induction 2; subst; eauto using sched_step_inv_amap.
Qed.

Lemma head_step_preserves_dom `{Countable A} σ α (G:graph A) e σ' α' e' t:
  dom σ = dom α ->
  head_step G t σ α e σ' α' e' ->
  dom σ' = dom α'.
Proof.
  inversion 2; subst; eauto.
  1,2,4,6:rewrite ?dom_insert_L; set_solver.
  { rewrite dom_insert_lookup_L //. }
  { case_bool_decide; last done. rewrite dom_insert_lookup_L //. }
Qed.

Lemma sched_step_preserves_dom `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  dom σ = dom α ->
  sched_step σ α G T e σ' α' G' T' e' ->
  dom σ' = dom α'.
Proof.
  inversion 2; subst; eauto using head_step_preserves_dom.
  rewrite !dom_insert_L. set_solver.
Qed.

Lemma step_preserves_dom `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  dom σ = dom α ->
  step σ α G T e σ' α' G' T' e' ->
  dom σ' = dom α'.
Proof.
  induction 2; eauto using sched_step_preserves_dom.
Qed.


(******************************************************************************)
(* A variant of [step] which is parameterized by a path, indicating which task
   took a step. [pstep] is used only for semantics_equiv *)

Inductive pstep `{Countable A} : graph.path ->
  store -> gmap loc A -> graph A -> task_tree A -> expr ->
  store -> gmap loc A -> graph A -> task_tree A -> expr -> Prop :=
| PStepHead : forall σ α G T2 e2 σ' α' G' T2' e2',
    sched_step σ α G T2 e2 σ' α' G' T2' e2' ->
    pstep [] σ α G T2 e2 σ' α' G' T2' e2'
| PStepBind : forall p σ α G T e σ' α' G' T' e' K,
    pstep p σ α G T e σ' α' G' T' e' ->
    pstep p σ α G T (fill_item K e) σ' α' G' T' (fill_item K e')
| PStepParL : forall p σ α G G' e1 e2 T1 T2 σ' α' T1' e1' t,
    pstep p σ α G T1 e1 σ' α' G' T1' e1' ->
    pstep (Left::p) σ α G (Node t T1 T2) (RunPar e1 e2) σ' α' G' (Node t T1' T2) (RunPar e1' e2)
| PStepParR : forall p σ α G G' e1 e2 T1 T2 σ' α' T2' e2' t,
    pstep p σ α G T2 e2 σ' α' G' T2' e2' ->
    pstep (Right::p) σ α G (Node t T1 T2) (RunPar e1 e2) σ' α' G' (Node t T1 T2') (RunPar e1 e2')
.

Lemma step_equiv_pstep `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  step σ α G T e σ' α' G' T' e' <-> exists p, pstep p σ α G T e σ' α' G' T' e'.
Proof.
  split.
  { induction 1. eauto using PStepHead.
    all:destruct IHstep; eauto using PStepBind, PStepParL, PStepParR. }
  { intros (?&Hstep). induction Hstep; eauto with step. }
Qed.
