From Coq.ssr Require Import ssreflect.
From stdpp Require Import strings binders gmap gmultiset.

From dislog.utils Require Import graph.
From dislog.lang Require Export head_semantics.

(******************************************************************************)
(* Definition of a task_tree and operations on it. *)

Inductive task_tree_old (A:Type) : Type :=
| Leaf_old : A -> task_tree_old A
| Node_old : task_tree_old A -> task_tree_old A -> task_tree_old A.

Arguments Leaf_old {A}%_type_scope.
Arguments Node_old {A}%_type_scope.

Fixpoint leaves_old `{Countable A} (T:task_tree_old A) : gset A :=
  match T with
  | Leaf_old t => {[t]}
  | Node_old T1 T2 => leaves_old T1 ∪ leaves_old T2 end.

Lemma leaves_old_node `{Countable A} (T1 T2:task_tree_old A) :
  leaves_old (Node_old T1 T2) = leaves_old T1 ∪ leaves_old T2.
Proof. done. Qed.

Definition is_leaf_old {A:Type} (T:task_tree_old A) := match T with | Leaf_old _ => true | _ => false end.

(******************************************************************************)
(* [sched_step_old] either a head step, fork or join *)

Inductive sched_step_old `{Countable A} :
  store -> gmap loc A -> gset (A*A) -> task_tree_old A -> expr ->
  store -> gmap loc A -> gset (A*A) -> task_tree_old A -> expr -> Prop :=
| SchedHead_old : forall σ α G t e σ' α' e',
    head_step G t σ α e σ' α' e' ->
    sched_step_old σ α G (Leaf_old t) e σ' α' G (Leaf_old t) e'
| SchedFork_old : forall σ α G G' t0 t1 t2 (v1 v2:val),
    t1 ∉ vertices G ->
    t2 ∉ vertices G ->
    t1 ≠ t2 ->
    G' = graph_fork G t0 t1 t2 ->
    sched_step_old σ α G (Leaf_old t0) (Par v1 v2)
         σ α G' (Node_old (Leaf_old t1) (Leaf_old t2)) (RunPar (Call v1 [Val VUnit]) (Call v2 [Val VUnit]))
| SchedJoin_old : forall σ α G G' t1 t2 t3 (v1 v2:val) σ' α' (l:loc),
    t3 ∉ vertices G ->
    G' = graph_join G t1 t2 t3 ->
    l ∉ dom σ ->
    l ∉ dom α ->
    σ' = <[l:=SProd v1 v2]> σ ->
    α' = <[l:=t3]> α ->
    sched_step_old σ  α  G (Node_old (Leaf_old t1) (Leaf_old t2)) (RunPar v1 v2)
      σ' α' G' (Leaf_old t3) l.

Lemma sched_step_old_no_val `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' :
  sched_step_old σ α G T e σ' α' G' T' e' -> ¬ is_val e.
Proof. destruct 1; eauto using head_step_no_val. Qed.

Lemma sched_step_old_no_ctx `{Countable A} σ α (G:graph A) T e σ' α' G' T' e' K :
  ¬ is_val e → ¬ sched_step_old σ α G T (fill_item K e) σ' α' G' T' e'.
Proof.
  intros ? Hr. inversion Hr; subst.
  { eapply head_step_no_ctx; eauto. }
  all:destruct K; naive_solver.
Qed.

Lemma sched_step_old_inv_graph `{Countable A} σ α (G:graph A) T2 e2 σ' α' G' T2' e2' :
  sched_step_old σ α G T2 e2 σ' α' G' T2' e2' ->
  G ⊆ G'.
Proof.
  destruct 1; subst; eauto; [apply graph_fork_incl | apply graph_join_incl].
Qed.

Lemma sched_step_old_inv_amap `{Countable A} σ α (G:graph A) T e σ' α' G' T' e':
  dom σ = dom α ->
  sched_step_old σ α G T e σ' α' G' T' e' ->
  α ⊆ α'.
Proof.
  intros ?.
  destruct 1; subst; eauto using head_step_inv_amap.
  apply insert_subseteq. apply not_elem_of_dom. eauto.
Qed.

(******************************************************************************)
(* Distant semantics. *)

(* We parameterize the step relation with a path,
   which describe the task being reduced.
   It allows to state theorems about the scheduling. *)

Inductive step_old `{Countable A} : path ->
  store -> gmap loc A -> graph A -> task_tree_old A -> expr ->
  store -> gmap loc A -> graph A -> task_tree_old A -> expr -> Prop :=
| StepHead_old : forall σ α G T2 e2 σ' α' G' T2' e2',
    sched_step_old σ α G T2 e2 σ' α' G' T2' e2' ->
    step_old [] σ α G T2 e2 σ' α' G' T2' e2'
| StepBind_old : forall p σ α G T e σ' α' G' T' e' K,
    step_old p σ α G T e σ' α' G' T' e' ->
    step_old p σ α G T (fill_item K e) σ' α' G' T' (fill_item K e')
| StepParL_old : forall p σ α G G' e1 e2 T1 T2 σ' α' T1' e1',
    step_old p σ α G T1 e1 σ' α' G' T1' e1' ->
    step_old (Left::p) σ α G (Node_old T1 T2) (RunPar e1 e2) σ' α' G' (Node_old T1' T2) (RunPar e1' e2)
| StepParR_old : forall p σ α G G' e1 e2 T1 T2 σ' α' T2' e2',
    step_old p σ α G T2 e2 σ' α' G' T2' e2' ->
    step_old (Right::p) σ α G (Node_old T1 T2) (RunPar e1 e2) σ' α' G' (Node_old T1 T2') (RunPar e1 e2')
.

Lemma step_old_no_val `{Countable A} p σ α (G:graph A) T e σ' α' G' T' e' :
  step_old p σ α G T e σ' α' G' T' e' ->
  ¬ is_val e.
Proof.
  inversion 1; subst; eauto using sched_step_old_no_val.
  destruct K; eauto.
Qed.

Lemma step_old_inv_graph `{Countable A} p σ α (G:graph A) T e σ' α' G' T' e'  :
  step_old p σ α G T e σ' α' G' T' e' ->
  G ⊆ G'.
Proof.
  induction 1; subst; eauto using sched_step_old_inv_graph.
Qed.

Lemma step_old_inv_amap `{Countable A} p σ α (G:graph A) T e σ' α' G' T' e' :
  dom σ = dom α ->
  step_old p σ α G T e σ' α' G' T' e' ->
  α ⊆ α'.
Proof.
  induction 2; subst; eauto using sched_step_old_inv_amap.
Qed.
