From Coq.ssr Require Import ssreflect.
From stdpp Require Import base gmap fin_sets.

From dislog.utils Require Import graph.
From dislog.lang Require Import syntax semantics_cycle.

Section go.

Context `{Countable A}.

(******************************************************************************)
(* [abef] & [vabef]: allocated before *)

(* [abef G α t l] asserts that l was allocated (as in α) before t in the
   graph G *)
Definition abef (G:graph.graph A) (α:gmap loc A) t l :=
  match α !! l with
  | Some v' => reachable G v' t
  | None => False end.

(* vabef extends abef to arbitrary values *)
Fixpoint vabef G α t v :=
  match v with
  | VLoc l => abef G α t l
  | VFold v => vabef G α t v
  | _ => True end.

Lemma vabef_no_loc G α t v :
  ¬ is_loc v ->
  vabef G α t v.
Proof. induction v; naive_solver. Qed.

Lemma abef_mon_graph G G' α t l :
  G ⊆ G' ->
  abef G α t l ->
  abef G' α t l.
Proof.
  rewrite /abef.
  intros HG Hl.
  destruct (α !! l); eauto using reachable_mon.
Qed.

Lemma abef_mon_amap G α α' t l :
  α ⊆ α' ->
  abef G α t l ->
  abef G α' t l.
Proof.
  rewrite map_subseteq_spec /abef.
  intros Hi.
  destruct (α !! l) eqn:E; last easy.
  apply Hi in E. rewrite E //.
Qed.

Lemma abef_pre_reachable G α t t' x :
  reachable G t t' ->
  abef G α t x → abef G α t' x.
Proof.
  rewrite /abef.
  destruct (α !! x); last easy.
  intros. now transitivity t.
Qed.

Lemma abef_insert G l t α :
  abef G (<[l:=t]> α) t l.
Proof.
  rewrite /abef lookup_insert //.
Qed.

(* ------------------------------------------------------------------------ *)
(* [all_abef] the iteration of [abef] *)

Definition all_abef G α (t:A) (L:gset loc) :=
  set_Forall (abef G α t) L.

Lemma all_abef_union G α t L1 L2 :
  all_abef G α t (L1 ∪ L2) <-> all_abef G α t L1 /\ all_abef G α t L2.
Proof.
  split.
  { intros. split.
    { eapply set_Forall_union_inv_1. eauto. }
    { eapply set_Forall_union_inv_2. eauto. } }
  { intros (?&?). apply set_Forall_union; eauto. }
Qed.

Lemma all_abef_mon_set G α t L L' :
  L ⊆ L' ->
  all_abef G α t L' ->
  all_abef G α t L.
Proof.
  rewrite /all_abef /set_Forall.
  set_solver.
Qed.

Lemma all_abef_mon_graph G G' α t L :
  G ⊆ G' ->
  all_abef G α t L ->
  all_abef G' α t L.
Proof.
  intros ??. eapply set_Forall_impl; eauto using abef_mon_graph.
Qed.

Lemma all_abef_mon_amap G α α' t L :
  α ⊆ α' ->
  all_abef G α t L ->
  all_abef G α' t L.
Proof.
  intros ??. eapply set_Forall_impl; eauto using abef_mon_amap.
Qed.

Lemma all_abef_pre_reachable G α t t' L :
  reachable G t t' ->
  all_abef G α t L ->
  all_abef G α t' L.
Proof.
  intros ??. eapply set_Forall_impl; eauto using abef_pre_reachable.
Qed.

Lemma all_abef_elem l G α t L :
  l ∈ L ->
  all_abef G α t L ->
  abef G α t l.
Proof.
  intros.
  rewrite -(@set_Forall_singleton _ (gset loc)).
  eapply all_abef_mon_set. 2:eauto. set_solver.
Qed.

(******************************************************************************)
(* main predicate *)

Inductive disentangled : graph.graph A -> gmap loc A -> semantics_cycle.task_tree A -> expr -> Prop :=
| DELeaf : forall G α (t:A) (e:expr),
  all_abef G α t (locs e) ->
  disentangled G α (Leaf t) e
| DEBind : forall G α t T1 T2 (K:ctx) (e:expr),
  disentangled G α (Node t T1 T2) e ->
  (set_Forall (fun t => all_abef G α t (locs K)) (frontier T1 ∪ frontier T2)) ->
  disentangled G α (Node t T1 T2) (fill_item K e)
| DEPar : forall G α T1 T2 (e1 e2:expr) t,
  disentangled G α T1 e1 ->
  disentangled G α T2 e2 ->
  disentangled G α (Node t T1 T2) (RunPar e1 e2).

End go.
